/**
 * 
 */
package org.gcube.informationsystem.model.reference.properties;

import org.gcube.com.fasterxml.jackson.databind.annotation.JsonDeserialize;
import org.gcube.informationsystem.model.impl.properties.PropagationConstraintImpl;
import org.gcube.informationsystem.types.annotations.Final;
import org.gcube.informationsystem.types.annotations.ISProperty;
import org.gcube.informationsystem.types.reference.Change;
import org.gcube.informationsystem.types.reference.Changelog;
import org.gcube.informationsystem.types.reference.TypeMetadata;
import org.gcube.informationsystem.utils.Version;

/**
 * Defines the propagation behavior for a relation when context-related events
 * (add, remove, delete) occur on the source entity or the relation itself.
 * <p>
 * The default propagation constraints are:
 * <ul>
 * <li><b>IsRelatedTo:</b> delete=keep, remove=keep, add=unpropagate</li>
 * <li><b>ConsistsOf:</b> delete=cascade, remove=cascade, add=propagate</li>
 * </ul>
 * For more details, refer to the
 * <a href="https://wiki.gcube-system.org/gcube/Facet_Based_Resource_Model#Propagation_Constraint">
 * gCube Facet Based Resource Model Wiki</a>.
 *
 * @author Luca Frosini (ISTI - CNR)
 * @see Property
 */
@JsonDeserialize(as=PropagationConstraintImpl.class)
@TypeMetadata(name = PropagationConstraint.NAME, description = "This type provides propagation constraint for Relation", version = "1.1.0")
@Changelog({
	@Change(version = "1.1.0", description = "Added 'delete' propagation constraint"),
	@Change(version = Version.MINIMAL_VERSION_STRING, description = Version.MINIMAL_VERSION_DESCRIPTION)
})
@Final
public interface PropagationConstraint extends Property {

	/** The name of the PropagationConstraint property type. */
	public static final String NAME = "PropagationConstraint"; // PropagationConstraint.class.getSimpleName();
	
	/** The property name for the 'add' constraint. */
	public static final String ADD_PROPERTY = "add";
	/** The property name for the 'delete' constraint. */
	public static final String DELETE_PROPERTY = "delete";
	/** The property name for the 'remove' constraint. */
	public static final String REMOVE_PROPERTY = "remove";

	/**
	 * Defines the behavior for an 'add' operation.
	 */
	public enum AddConstraint {
		
		/**
		 * When the source entity is added to a context, the relation and its
		 * target entity are also added to that context.
		 */
		propagate,
		
		/**
		 * When the source entity is added to a context, the action is not
		 * propagated to the target entity.
		 */
		unpropagate
			
	}
	
	/**
	 * Defines the behavior for a 'delete' operation.
	 */
	public enum DeleteConstraint {
		
		/**
		 * When the source entity is deleted, the target entity is also deleted,
		 * but only if it has no other incoming relations.
		 */
		cascadeWhenOrphan,
		
		/**
		 * When the source entity is deleted, the relation and its target entity
		 * are also deleted.
		 */
		cascade,
		
		/**
		 * When the source entity is deleted, the relation is deleted, but the
		 * target entity is preserved.
		 */
		keep
		
	}

	
	/**
	 * Defines the behavior for a 'remove' from context operation.
	 */
	public enum RemoveConstraint {
		
		/**
		 * When the source entity is removed from a context, the target entity is
		 * also removed, but only if it has no other incoming relations within
		 * that context.
		 */
		cascadeWhenOrphan,
		
		/**
		 * When the source entity is removed from a context, the relation and its
		 * target entity are also removed from that context.
		 */
		cascade,
		
		/**
		 * When the source entity is removed from a context, the relation is
		 * removed, but the target entity remains in the context.
		 */
		keep
		
	}

	/**
	 * Returns the propagation constraint for the 'add' operation.
	 *
	 * @return The {@link AddConstraint}.
	 */
	@ISProperty(name=ADD_PROPERTY, description = "It indicates the behaviour to implement for the target Entity when an 'add' action is performed on the source Resource. Add action is the operation of adding an instance to a context.", mandatory = true, nullable = false)
	public AddConstraint getAddConstraint();
	
	/**
	 * Sets the propagation constraint for the 'add' operation.
	 *
	 * @param addConstraint The {@link AddConstraint} to set.
	 */
	public void setAddConstraint(AddConstraint addConstraint);
	
	
	/**
	 * Returns the propagation constraint for the 'delete' operation.
	 *
	 * @return The {@link DeleteConstraint}.
	 */
	@ISProperty(name=DELETE_PROPERTY, description = "It indicates the behaviour to implement for the target Entity when a 'delete' action is performed on the source Resource. Delet action  is the operation of deleting an instance (it has an impact on all contexts).", mandatory = true, nullable = false)
	public DeleteConstraint getDeleteConstraint();
	
	/**
	 * Sets the propagation constraint for the 'delete' operation.
	 *
	 * @param deleteConstraint The {@link DeleteConstraint} to set.
	 */
	public void setDeleteConstraint(DeleteConstraint deleteConstraint);
	
	
	/**
	 * Returns the propagation constraint for the 'remove' operation.
	 *
	 * @return The {@link RemoveConstraint}.
	 */
	@ISProperty(name=REMOVE_PROPERTY, description = "It indicates the behaviour to implement for the target Entity when a 'remove' action is performed on the source Resource. Remove actions is the operation of removing an instance from a context.", mandatory = true, nullable = false)
	public RemoveConstraint getRemoveConstraint();
	
	/**
	 * Sets the propagation constraint for the 'remove' operation.
	 *
	 * @param removeConstraint The {@link RemoveConstraint} to set.
	 */
	public void setRemoveConstraint(RemoveConstraint removeConstraint);
	
}
