/**
 * 
 */
package org.gcube.informationsystem.model.reference.properties;

import java.util.Date;

import org.gcube.com.fasterxml.jackson.annotation.JsonFormat;
import org.gcube.com.fasterxml.jackson.databind.annotation.JsonDeserialize;
import org.gcube.informationsystem.base.reference.Element;
import org.gcube.informationsystem.base.reference.IdentifiableElement;
import org.gcube.informationsystem.model.impl.properties.MetadataImpl;
import org.gcube.informationsystem.types.annotations.Final;
import org.gcube.informationsystem.types.annotations.ISProperty;
import org.gcube.informationsystem.types.reference.Change;
import org.gcube.informationsystem.types.reference.TypeMetadata;
import org.gcube.informationsystem.utils.Version;

/**
 * A property type that provides metadata for an {@link IdentifiableElement}.
 * <p>
 * This includes information about the element's creation and last update,
 * such as the user responsible and the timestamp of the event.
 *
 * @author Luca Frosini (ISTI - CNR)
 * @see Property
 * @see org.gcube.informationsystem.base.reference.IdentifiableElement
 */
@JsonDeserialize(as=MetadataImpl.class)
@TypeMetadata(name = Metadata.NAME, description = "This type provides metadata per every IdentifiableElement", version = Version.MINIMAL_VERSION_STRING)
@Change(version = Version.MINIMAL_VERSION_STRING, description = Version.MINIMAL_VERSION_DESCRIPTION)
@Final
public interface Metadata extends Property {

	/** The name of the Metadata property type. */
	public static final String NAME = "Metadata"; // Metadata.class.getSimpleName();

	/** A constant for an unknown user. */
	public static final String UNKNOWN_USER = "UNKNOWN_USER";
	/** A constant for a user hidden for privacy reasons. */
	public static final String HIDDEN_FOR_PRIVACY_USER = "HIDDEN_FOR_PRIVACY";

	/** The property name for the creator. */
	public static final String CREATED_BY_PROPERTY = "createdBy";
	/** The property name for the creation time. */
	public static final String CREATION_TIME_PROPERTY = "creationTime";
	/** The property name for the last updater. */
	public static final String LAST_UPDATE_BY_PROPERTY = "lastUpdateBy";
	/** The property name for the last update time. */
	public static final String LAST_UPDATE_TIME_PROPERTY = "lastUpdateTime";

	/**
	 * Returns the user who created the element.
	 *
	 * @return The creator's identifier.
	 */
	@ISProperty(name = CREATED_BY_PROPERTY, description = "The user that created the Entity or the Relation. It is initialized at Creation Time.", readonly = true, mandatory = true, nullable = false)
	public String getCreatedBy();

	/**
	 * Returns the time when the element was created.
	 *
	 * @return The creation date and time.
	 */
	@JsonFormat(shape = JsonFormat.Shape.STRING, pattern = Element.DATETIME_PATTERN)
	@ISProperty(name = CREATION_TIME_PROPERTY, description = "Creation time. It is represented in the format " + Element.DATETIME_PATTERN + ".", readonly = true, mandatory = true, nullable = false)
	public Date getCreationTime();

	/**
	 * Returns the user who last updated the element.
	 *
	 * @return The last updater's identifier.
	 */
	@ISProperty(name = LAST_UPDATE_BY_PROPERTY, description = "The user that made the last update to the Entity or the Relation. At Creation Time, it assumes the same value of " + CREATED_BY_PROPERTY + ".", mandatory = true, nullable = false)
	public String getLastUpdateBy();

	/**
	 * Returns the time when the element was last updated.
	 *
	 * @return The last update date and time.
	 */
	@JsonFormat(shape = JsonFormat.Shape.STRING, pattern = Element.DATETIME_PATTERN)
	@ISProperty(name = LAST_UPDATE_TIME_PROPERTY, description = "Last Update time. At creation time it assumes the same value of " + CREATION_TIME_PROPERTY + ". It is represented in the format " + Element.DATETIME_PATTERN, mandatory = true, nullable = false)
	public Date getLastUpdateTime();
	
}