/**
 * 
 */
package org.gcube.informationsystem.model.reference.properties;

import java.util.Date;

import org.gcube.com.fasterxml.jackson.annotation.JsonFormat;
import org.gcube.com.fasterxml.jackson.annotation.JsonInclude;
import org.gcube.com.fasterxml.jackson.databind.annotation.JsonDeserialize;
import org.gcube.informationsystem.base.reference.Element;
import org.gcube.informationsystem.model.impl.properties.EventImpl;
import org.gcube.informationsystem.types.annotations.ISProperty;
import org.gcube.informationsystem.types.reference.Change;
import org.gcube.informationsystem.types.reference.TypeMetadata;
import org.gcube.informationsystem.utils.Version;

/**
 * A property type that describes an event using the "Five Ws" checklist
 * (What, When, Who, Where, Why) and "How".
 * <p>
 * This provides a structured way to record information about significant
 * occurrences. For more details, see the
 * <a href="https://en.wikipedia.org/wiki/Five_Ws">Five Ws on Wikipedia</a>.
 *
 * @author Luca Frosini (ISTI - CNR)
 * @see Property
 */
@JsonDeserialize(as=EventImpl.class)
@TypeMetadata(name = Event.NAME, description = "This type provides information regarding an event using the Five Ws (checklist) https://en.wikipedia.org/wiki/Five_Ws", version = Version.MINIMAL_VERSION_STRING)
@Change(version = Version.MINIMAL_VERSION_STRING, description = Version.MINIMAL_VERSION_DESCRIPTION)
@JsonInclude(JsonInclude.Include.NON_NULL)
public interface Event extends Property, Comparable<Event> {

	/** The name of the Event property type. */
	public static final String NAME = "Event"; // Event.class.getSimpleName();

	/** The property name for what happened. */
	public static final String WHAT_PROPERTY = "what";
	/** The property name for when the event occurred. */
	public static final String WHEN_PROPERTY = "when";
	/** The property name for who triggered the event. */
	public static final String WHO_PROPERTY = "who";
	/** The property name for where the event occurred. */
	public static final String WHERE_PROPERTY = "where";
	/** The property name for why the event occurred. */
	public static final String WHY_PROPERTY = "why";
	/** The property name for how the event occurred. */
	public static final String HOW_PROPERTY = "how";

	/**
	 * Returns a description of what happened.
	 *
	 * @return The event description.
	 */
	@ISProperty(name = WHAT_PROPERTY, description = "WHAT happened.", readonly = true, mandatory = true, nullable = false)
	public String getWhat();
	
	/**
	 * Sets the description of what happened.
	 *
	 * @param what The event description.
	 */
	public void setWhat(String what);

	/**
	 * Returns when the event occurred.
	 *
	 * @return The date and time of the event.
	 */
	@JsonFormat(shape = JsonFormat.Shape.STRING, pattern = Element.DATETIME_PATTERN)
	@ISProperty(name = WHEN_PROPERTY, description = "WHEN the event occured. It is represented in the format " + Element.DATETIME_PATTERN + ".", readonly = true, mandatory = true, nullable = false)
	public Date getWhen();
	
	/**
	 * Sets when the event occurred.
	 *
	 * @param when The date and time of the event.
	 */
	public void setWhen(Date when);
	
	/**
	 * Returns who triggered the event.
	 *
	 * @return The identifier of the actor.
	 */
	@ISProperty(name = WHO_PROPERTY, description = "WHO triggered the event.", readonly = true, mandatory = false, nullable = false)
	public String getWho();
	
	/**
	 * Sets who triggered the event.
	 *
	 * @param who The identifier of the actor.
	 */
	public void setWho(String who);

	/**
	 * Returns where the event occurred.
	 *
	 * @return The location (physical or virtual).
	 */
	@ISProperty(name = WHERE_PROPERTY, description = "The location (can be virtual) WHERE the event occurred.", readonly = true, mandatory = false, nullable = false)
	public String getWhere();
	
	/**
	 * Sets where the event occurred.
	 *
	 * @param where The location.
	 */
	public void setWhere(String where);

	/**
	 * Returns the reason why the event occurred.
	 *
	 * @return The reason.
	 */
	@ISProperty(name = WHY_PROPERTY, description = "The reason WHY the event occurred.", readonly = true, mandatory = false, nullable = false)
	public String getWhy();
	
	/**
	 * Sets the reason why the event occurred.
	 *
	 * @param why The reason.
	 */
	public void setWhy(String why);
	
	/**
	 * Returns how the event occurred.
	 *
	 * @return The manner in which the event occurred.
	 */
	@ISProperty(name = HOW_PROPERTY, description = "How the event occurred.", readonly = true, mandatory = false, nullable = false)
	public String getHow();
	
	/**
	 * Sets how the event occurred.
	 *
	 * @param how The manner in which the event occurred.
	 */
	public void setHow(String how);
	
}