/**
 * 
 */
package org.gcube.informationsystem.model.reference.entities;

import java.util.List;
import java.util.UUID;

import org.gcube.com.fasterxml.jackson.annotation.JsonIgnore;
import org.gcube.com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import org.gcube.com.fasterxml.jackson.annotation.JsonPropertyOrder;
import org.gcube.informationsystem.base.reference.Element;
import org.gcube.informationsystem.base.reference.IdentifiableElement;
import org.gcube.informationsystem.model.reference.ERElement;
import org.gcube.informationsystem.model.reference.ModelElement;
import org.gcube.informationsystem.model.reference.relations.ConsistsOf;
import org.gcube.informationsystem.model.reference.relations.IsRelatedTo;
import org.gcube.informationsystem.types.annotations.Abstract;
import org.gcube.informationsystem.types.annotations.RelatedResourcesEntry;
import org.gcube.informationsystem.types.annotations.ResourceSchema;
import org.gcube.informationsystem.types.annotations.ResourceSchemaEntry;
import org.gcube.informationsystem.types.reference.Change;
import org.gcube.informationsystem.types.reference.TypeMetadata;
import org.gcube.informationsystem.utils.Version;

/**
 * Represents a resource, the central entity in the facet-based model.
 * <p>
 * A resource is composed of one or more {@link Facet}s, which describe its
 * different aspects. It can also be related to other resources. This interface
 * provides methods for accessing and managing these relationships. For more
 * details, refer to the
 * <a href="https://wiki.gcube-system.org/gcube/Facet_Based_Resource_Model#Resource">
 * gCube Facet Based Resource Model Wiki</a>.
 *
 * @author Luca Frosini (ISTI - CNR)
 * @see Entity
 * @see Facet
 * @see ConsistsOf
 * @see IsRelatedTo
 */
@Abstract
@JsonIgnoreProperties(ignoreUnknown=true)
@JsonPropertyOrder({ Element.TYPE_PROPERTY, ModelElement.SUPERTYPES_PROPERTY, ModelElement.EXPECTED_TYPE_PROPERTY, IdentifiableElement.ID_PROPERTY, IdentifiableElement.METADATA_PROPERTY, ERElement.CONTEXTS_PROPERTY, Resource.CONSISTS_OF_PROPERTY, Resource.IS_RELATED_TO_PROPERTY })
// @JsonDeserialize(as=ResourceImpl.class) Do not uncomment to manage subclasses
@ResourceSchema(
	facets={
		@ResourceSchemaEntry(relation=ConsistsOf.class, facet=Facet.class, min=1, description="Any Resource consists of one or more Facets which describes the different aspects of the resource."),
	},
	resources= {
		@RelatedResourcesEntry(source=Resource.class, relation=IsRelatedTo.class, target=Resource.class, description="Any Resource can be related to any other resource.")
	}
)
@TypeMetadata(name = Resource.NAME, description = "This is the base type for any Resource", version = Version.MINIMAL_VERSION_STRING)
@Change(version = Version.MINIMAL_VERSION_STRING, description = Version.MINIMAL_VERSION_DESCRIPTION)
public interface Resource extends Entity {
	
	/** The name of the resource type. */
	public static final String NAME = "Resource"; //Resource.class.getSimpleName();
	
	/** The property name for the 'consistsOf' relations. */
	public static final String CONSISTS_OF_PROPERTY = "consistsOf";
	/** The property name for the 'isRelatedTo' relations. */
	public static final String IS_RELATED_TO_PROPERTY = "isRelatedTo";
	
	/**
	 * Returns all {@link ConsistsOf} relations for this resource, linking it to its facets.
	 *
	 * @return A list of {@code ConsistsOf} relations.
	 */
	// @JsonManagedReference
	public List<ConsistsOf<? extends Resource, ? extends Facet>> getConsistsOf();
	
	/**
	 * Returns {@link ConsistsOf} relations of a specific type.
	 *
	 * @param <C> The specific type of {@code ConsistsOf} relation.
	 * @param clz The class of the relation type.
	 * @return A list of matching {@code ConsistsOf} relations.
	 */
	@JsonIgnore
	public <C extends ConsistsOf<? extends Resource, ? extends Facet>> List<C> getConsistsOf(Class<C> clz);
	
	/**
	 * Returns {@link ConsistsOf} relations of a specific type, targeting a specific facet type.
	 *
	 * @param <F>    The type of the target facet.
	 * @param <C>    The type of the {@code ConsistsOf} relation.
	 * @param clz    The class of the relation type.
	 * @param target The class of the target facet type.
	 * @return A list of matching {@code ConsistsOf} relations.
	 */
	@JsonIgnore
	public <F extends Facet, C extends ConsistsOf<? extends Resource, F>> List<C> getConsistsOf(Class<C> clz, Class<F> target);
	
	
	/**
	 * Returns all {@link IsRelatedTo} relations for this resource, linking it to other resources.
	 *
	 * @return A list of {@code IsRelatedTo} relations.
	 */
	// @JsonManagedReference
	public List<IsRelatedTo<? extends Resource, ? extends Resource>> getIsRelatedTo();
	
	/**
	 * Returns {@link IsRelatedTo} relations of a specific type.
	 *
	 * @param <I> The specific type of {@code IsRelatedTo} relation.
	 * @param clz The class of the relation type.
	 * @return A list of matching {@code IsRelatedTo} relations.
	 */
	@JsonIgnore
	public <I extends IsRelatedTo<? extends Resource, ? extends Resource>> List<I> getIsRelatedTo(Class<I> clz);
	
	
	/**
	 * Returns all facets associated with this resource.
	 *
	 * @return A list of all facets.
	 */
	@JsonIgnore
	public List<? extends Facet> getFacets();
	
	/**
	 * Returns all facets of a specific type associated with this resource.
	 *
	 * @param <F> The type of the facet.
	 * @param clz The class of the facet type.
	 * @return A list of matching facets.
	 */
	@JsonIgnore
	public <F extends Facet> List<F> getFacets(Class<F> clz);
	
	/**
	 * Returns facets of a specific type, linked by a specific {@code ConsistsOf} relation type.
	 *
	 * @param <F>    The type of the facet.
	 * @param <C>    The type of the {@code ConsistsOf} relation.
	 * @param clz    The class of the relation type.
	 * @param target The class of the facet type.
	 * @return A list of matching facets.
	 */
	@JsonIgnore
	public <F extends Facet, C extends ConsistsOf<? extends Resource, F>> List<F> getFacets(Class<C> clz, Class<F> target);
	
	
	/**
	 * Adds a facet to the resource by its UUID.
	 *
	 * @param uuid The UUID of the facet to add.
	 */
	public void addFacet(UUID uuid);
	
	/**
	 * Adds a facet to the resource.
	 *
	 * @param <F>   The type of the facet.
	 * @param facet The facet instance to add.
	 */
	public <F extends Facet> void addFacet(F facet);
	
	/**
	 * Adds a facet to the resource via a {@code ConsistsOf} relation.
	 *
	 * @param <C>      The type of the {@code ConsistsOf} relation.
	 * @param relation The relation instance to add.
	 */
	public <C extends ConsistsOf<? extends Resource, ? extends Facet>> void addFacet(C relation);
	
	
	/*
	public void attachResource(UUID uuid);
	
	public <R extends Resource> void attachResource(R resource);
	*/
	
	/**
	 * Attaches another resource to this one via an {@code IsRelatedTo} relation.
	 *
	 * @param <I>      The type of the {@code IsRelatedTo} relation.
	 * @param relation The relation instance to add.
	 */
	public <I extends IsRelatedTo<? extends Resource, ? extends Resource>> void attachResource(I relation);
	
}
