/**
 * 
 */
package org.gcube.informationsystem.model.impl.relations;

import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;

import org.gcube.com.fasterxml.jackson.annotation.JsonIgnore;
import org.gcube.com.fasterxml.jackson.annotation.JsonSetter;
import org.gcube.com.fasterxml.jackson.annotation.JsonTypeName;
import org.gcube.informationsystem.base.impl.relations.RelationElementImpl;
import org.gcube.informationsystem.base.reference.Element;
import org.gcube.informationsystem.model.reference.ERElement;
import org.gcube.informationsystem.model.reference.entities.Entity;
import org.gcube.informationsystem.model.reference.entities.Resource;
import org.gcube.informationsystem.model.reference.properties.PropagationConstraint;
import org.gcube.informationsystem.model.reference.properties.Property;
import org.gcube.informationsystem.model.reference.relations.Relation;
import org.gcube.informationsystem.serialization.ElementMapper;

/**
 * The default abstract implementation of the {@link Relation} interface.
 *
 * @author Luca Frosini (ISTI - CNR)
 */
@JsonTypeName(value = Relation.NAME)
public abstract class RelationImpl<S extends Resource, T extends Entity>
		extends RelationElementImpl<S,T> implements Relation<S, T> {

	/**
	 * The serial version UID.
	 */
	private static final long serialVersionUID = -6249979476879235053L;
	
	/** The supertypes of the relation. */
	protected List<String> supertypes;
	/** The expected type of the relation. */
	protected String expectedtype;
	/** The contexts of the relation. */
	protected Map<UUID,String> contexts;
	
	/** The propagation constraint of the relation. */
	protected PropagationConstraint propagationConstraint;

	/** The additional properties of the relation. */
	@JsonIgnore
	protected Map<String, Object> additionalProperties;

	/**
	 * A set of keys that are allowed as additional properties, even if they start
	 * with '_' or '@'.
	 */
	protected final Set<String> allowedAdditionalKeys;

	/**
	 * Default constructor.
	 */
	protected RelationImpl() {
		super();
		this.additionalProperties = new HashMap<>();
		this.allowedAdditionalKeys = new HashSet<>();
	}

	/**
	 * Constructs a new {@code RelationImpl} with the given source, target, and
	 * propagation constraint.
	 *
	 * @param source                The source of the relation.
	 * @param target                The target of the relation.
	 * @param propagationConstraint The propagation constraint of the relation.
	 */
	protected RelationImpl(S source, T target,
			PropagationConstraint propagationConstraint) {
		this();
		this.source = source;
		this.target = target;
		this.propagationConstraint = propagationConstraint;
	}


	/**
	 * {@inheritDoc}
	 */
	@Override
	public List<String> getSupertypes() {
		return this.supertypes;
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getExpectedtype() {
		return this.expectedtype;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public Map<UUID,String> getContexts(){
		return this.contexts;
	}
	
	/**
	 * Sets the contexts of the relation.
	 *
	 * @param contexts The contexts to set.
	 */
	@JsonSetter(value = ERElement.CONTEXTS_PROPERTY)
	protected void setContexts(Map<UUID,String> contexts) {
		this.contexts = contexts;
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public PropagationConstraint getPropagationConstraint() {
		return this.propagationConstraint;
	}

	/**
	 * Sets the propagation constraint for this relation.
	 *
	 * @param propagationConstraint The {@link PropagationConstraint} to set.
	 */
	@JsonIgnore
	@Override
	public void setPropagationConstraint(PropagationConstraint propagationConstraint) {
		this.propagationConstraint = propagationConstraint;
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public Map<String, Object> getAdditionalProperties() {
		return additionalProperties;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setAdditionalProperties(Map<String, Object> additionalProperties) {
		this.additionalProperties = additionalProperties;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Object getAdditionalProperty(String key) {
		return additionalProperties.get(key);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setAdditionalProperty(String key, Object value) {
		if (!allowedAdditionalKeys.contains(key)) {
			if (key.startsWith("_")) {
				return;
			}
			if (key.startsWith("@")) {
				return;
			}
			if (key.compareTo(PROPAGATION_CONSTRAINT_PROPERTY) == 0) {
				return;
			}
			if (key.compareTo(TARGET_PROPERTY) == 0) {
				return;
			}
			if (key.compareTo(SOURCE_PROPERTY) == 0) {
				return;
			}
		}
		
		/*
		Additional properties are not deserialized to the proper Property type.
		The first attempt was to try to write a specific deserializer but it fails.	
		This fix the issue.
		*/
		try {
			if(value instanceof Map<?,?>) {
				@SuppressWarnings("unchecked")
				Map<String, Object> map = (Map<String,Object>) value;
				if(map.containsKey(Element.TYPE_PROPERTY)) {
					String reserialized = ElementMapper.getObjectMapper().writeValueAsString(map);
					Property property = ElementMapper.unmarshal(Property.class, reserialized);
					value = property;
				}
			}
		}catch (Throwable e) {
			e.getMessage();
			// Any type of error/exception must be catched
		}
		/* END of fix to properly deserialize Property types*/
		
		this.additionalProperties.put(key, value);
	}

}
