package org.gcube.informationsystem.discovery.knowledge;

import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;

import org.gcube.informationsystem.types.reference.properties.LinkedEntity;

/**
 * A class that stores and manages knowledge about the usage of linked entities
 * within the model.
 * <p>
 * It maintains a map where keys are type names (source, relation, or target)
 * and values are sets of {@link LinkedEntity} objects that use that type.
 *
 * @author Luca Frosini (ISTI - CNR)
 */
public class UsageKnowledge {
	
	/** A map from type names to the set of linked entities that use them. */
	protected Map<String, Set<LinkedEntity>> map;
	
	/**
	 * Default constructor. Initializes the usage map.
	 */
	public UsageKnowledge(){
		this.map = new LinkedHashMap<>();
	}
	
	/**
	 * Adds a linked entity to the usage map for a specific type name.
	 *
	 * @param typeName The type name (source, relation, or target).
	 * @param linkedEntity The linked entity to add.
	 */
	protected void add(String typeName, LinkedEntity linkedEntity) {
		Set<LinkedEntity> list = map.get(typeName);
		if(list==null) {
			list = new TreeSet<>();
			map.put(typeName, list);
		}
		list.add(linkedEntity);
	}
	
	/**
	 * Adds a linked entity to the knowledge base, indexing it by its source,
	 * relation, and target types.
	 *
	 * @param linkedEntity The linked entity to add.
	 */
	public void add(LinkedEntity linkedEntity) {
		if(linkedEntity!=null) {
			String source = linkedEntity.getSource();
			add(source, linkedEntity);
			String relation = linkedEntity.getRelation();
			add(relation, linkedEntity);
			String target = linkedEntity.getTarget();
			add(target, linkedEntity);
		}
	}
	
	/**
	 * Adds a collection of linked entities to the knowledge base.
	 *
	 * @param linkedEntities The collection of linked entities to add.
	 */
	public void addAll(Collection<LinkedEntity> linkedEntities) {
		if(linkedEntities!=null) {
			for(LinkedEntity le : linkedEntities) {
				add(le);
			}
		}
	}
	
	/**
	 * Returns the set of linked entities that use a specific type.
	 *
	 * @param typeName The name of the type.
	 * @return A set of {@link LinkedEntity} objects, or {@code null} if the type
	 *         is not found.
	 */
	public Set<LinkedEntity> getUsage(String typeName){
		Set<LinkedEntity> list = map.get(typeName);
		return list;
	}
	
}
