package org.gcube.informationsystem.discovery.knowledge;

import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import org.gcube.informationsystem.base.reference.AccessType;
import org.gcube.informationsystem.base.reference.Element;
import org.gcube.informationsystem.discovery.Discovery;
import org.gcube.informationsystem.serialization.ElementMappingAction;
import org.gcube.informationsystem.tree.Tree;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Manages the discovered knowledge for a specific model, including type
 * hierarchies and usage information.
 *
 * @author Luca Frosini (ISTI - CNR)
 */
public class ModelKnowledge {
	
	private static final Logger logger = LoggerFactory.getLogger(ModelKnowledge.class);
	
	/** Whether the knowledge base has been created. */
	protected boolean created;
	/** A map of usage knowledge, keyed by access type. */
	protected Map<AccessType, UsageKnowledge> usageKnowledges;
	/** A map of class hierarchy trees, keyed by access type. */
	protected Map<AccessType, Tree<Class<Element>>> trees;
	/** A map of discovery instances, keyed by access type. */
	protected Map<AccessType, Discovery<Element>> discoveries;
	/** The set of packages to be scanned. */
	protected Set<Package> packages;
	
	/**
	 * Constructs a new {@code ModelKnowledge} instance.
	 */
	public ModelKnowledge() {
		this.packages = new HashSet<>();
		reset();
	}
	
	/**
	 * Resets the knowledge base.
	 */
	private void reset() {
		this.created = false;
		this.usageKnowledges = new HashMap<>();
		this.trees = new HashMap<>();
		this.discoveries = new HashMap<>();
	}
	
	/**
	 * Returns the usage knowledge for a specific access type.
	 *
	 * @param accessType The access type.
	 * @return The {@link UsageKnowledge} for the type.
	 * @throws RuntimeException if the knowledge has not yet been created.
	 */
	public UsageKnowledge getUsageKnowledge(AccessType accessType) {
		if(!created) {
			throw new RuntimeException("You must invoke createKnowledge() first");
		}
		return usageKnowledges.get(accessType);
	}
	
	/**
	 * Creates the knowledge base, with an option to force recreation.
	 *
	 * @param force {@code true} to force recreation of the knowledge base.
	 * @throws Exception if an error occurs during creation.
	 */
	public synchronized void createKnowledge(boolean force) throws Exception {
		if(force) {
			logger.info("Going to force Knowledge recreation");
			reset();
		}
		createKnowledge();
	}
	
	/**
	 * Adds a collection of packages to be scanned for model types.
	 *
	 * @param packages The packages to add.
	 */
	public void addPackages(Collection<Package> packages) {
		this.packages.addAll(packages);
	}
	
	/**
	 * Creates the knowledge base by discovering types, building hierarchies, and
	 * gathering usage information.
	 *
	 * @throws Exception if an error occurs during creation.
	 */
	public synchronized void createKnowledge() throws Exception {
		
		if(created) {
			logger.trace("Knowledge has been already created.");
			return;
		}
		
		AccessType[] modelTypes = AccessType.getModelTypes();
		
		ClassInformation classInformation = new ClassInformation();
		
		for(AccessType accessType : modelTypes) {
			
			Class<Element> clz = accessType.getTypeClass();
			Tree<Class<Element>> tree = new Tree<>(clz, classInformation);
			trees.put(accessType, tree);
			AddElementToTreeAction aetta = new AddElementToTreeAction(tree);
			if(accessType == AccessType.RESOURCE) {
				UsageKnowledge facetKnowledge = new UsageKnowledge();
				this.usageKnowledges.put(AccessType.FACET, facetKnowledge);
				aetta.setFacetKnowledge(facetKnowledge);
				
				UsageKnowledge resourceKnowledge = new UsageKnowledge();
				this.usageKnowledges.put(AccessType.RESOURCE, resourceKnowledge);
				aetta.setResourceKnowledge(resourceKnowledge);
			}
			
			Discovery<Element> discovery = new Discovery<>(clz);
			discoveries.put(accessType, discovery);
			discovery.addPackages(packages);
			discovery.addDiscoveredElementActions(new ElementMappingAction());
			discovery.addDiscoveredElementActions(aetta);
			
			discovery.discover();
		}
		
		this.created = true;
		
	}
	
	/**
	 * Returns the class hierarchy tree for a specific access type.
	 *
	 * @param accessType The access type.
	 * @return The {@link Tree} of classes.
	 */
	public synchronized Tree<Class<Element>> getClassesTree(AccessType accessType) {
		return trees.get(accessType);
	}
	
	
	/**
	 * Returns the discovery instance for a specific access type.
	 *
	 * @param accessType The access type.
	 * @return The {@link Discovery} instance.
	 */
	public synchronized Discovery<Element> getDiscovery(AccessType accessType) {
		return discoveries.get(accessType);
	}
}
