package org.gcube.informationsystem.discovery.knowledge;

import java.util.LinkedHashMap;
import java.util.Map;
import java.util.ServiceLoader;

import org.gcube.informationsystem.base.reference.AccessType;
import org.gcube.informationsystem.base.reference.Element;
import org.gcube.informationsystem.discovery.Discovery;
import org.gcube.informationsystem.discovery.RegistrationProvider;
import org.gcube.informationsystem.tree.Tree;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * A singleton class that manages the discovery and validation of model knowledge
 * from various {@link RegistrationProvider}s.
 *
 * @author Luca Frosini (ISTI - CNR)
 */
public class Knowledge {

	/** The logger for this class. */
	public static Logger logger = LoggerFactory.getLogger(Knowledge.class);
	
	private static Knowledge instance;
	
	/**
	 * Returns the singleton instance of the {@code Knowledge} class.
	 *
	 * @return The singleton instance.
	 * @throws Exception if an error occurs during discovery.
	 */
	public static Knowledge getInstance() throws Exception {
		return getInstance(false);
	}
	
	/**
	 * Returns the singleton instance of the {@code Knowledge} class, with an
	 * option to force re-discovery.
	 *
	 * @param forceRediscover {@code true} to force re-discovery of all models.
	 * @return The singleton instance.
	 * @throws Exception if an error occurs during discovery.
	 */
	public static Knowledge getInstance(boolean forceRediscover) throws Exception {
		if(forceRediscover) {
			instance = null;
		}
		
		if(instance == null) {
			instance = new Knowledge();
			instance.discover();
		}
		return instance;
	}
	
	/** The aggregated knowledge from all registered models. */
	protected ModelKnowledge allKnowledge;
	/** A map of model knowledge, keyed by model name. */
	protected Map<String, ModelKnowledge> modelKnowledges;
	/** A map of registration providers, keyed by model name. */
	protected Map<String, RegistrationProvider> registrationProviders;
	
	/**
	 * Private constructor to enforce singleton pattern.
	 */
	private Knowledge() {
	}
	
	/**
	 * Returns the aggregated knowledge from all registered models.
	 *
	 * @return The {@link ModelKnowledge} for all models.
	 */
	public ModelKnowledge getAllKnowledge() {
		return allKnowledge;
	}
	
	/**
	 * Returns the knowledge for a specific model.
	 *
	 * @param rp The {@link RegistrationProvider} for the model.
	 * @return The {@link ModelKnowledge} for the specified model.
	 */
	public ModelKnowledge getModelKnowledge(RegistrationProvider rp) {
		return modelKnowledges.get(rp.getModelName());
	}
	
	/**
	 * Returns the knowledge for a specific model by name.
	 *
	 * @param modelName The name of the model.
	 * @return The {@link ModelKnowledge} for the specified model.
	 */
	public ModelKnowledge getModelKnowledge(String modelName) {
		return modelKnowledges.get(modelName);
	}
	
	/**
	 * Validates the discovered knowledge for a specific model.
	 *
	 * @param rp The {@link RegistrationProvider} for the model to validate.
	 * @throws Exception if the validation fails.
	 */
	public void validateModelKnowledge(RegistrationProvider rp) throws Exception {
		ModelKnowledge modelKnowledge = getModelKnowledge(rp);
		ModelKnowledgeValidator ra = new ModelKnowledgeValidator(rp);
		
		AccessType[] accessTypes = AccessType.getModelTypes();
		for(AccessType accessType : accessTypes) {
			logger.trace("Going to analise discovered types of '{}' for model '{}'", accessType.getName(), rp.getModelName());
			Discovery<Element> discovery = modelKnowledge.getDiscovery(accessType);
			discovery.executeDiscoveredElementActions(ra);
			
			logger.trace("Going to analise tree of '{}' type for model '{}'", accessType.getName(), rp.getModelName());
			Tree<Class<Element>> tree = modelKnowledge.getClassesTree(accessType);
			tree.elaborate(ra);
		}
	}
	
	/**
	 * Validates the discovered knowledge for a specific model by name.
	 *
	 * @param modelName The name of the model to validate.
	 * @throws Exception if the validation fails.
	 */
	public void validateModelKnowledge(String modelName) throws Exception {
		RegistrationProvider rp = registrationProviders.get(modelName);
		validateModelKnowledge(rp);
	}
	
	/**
	 * Discovers and builds the knowledge base from all available
	 * {@link RegistrationProvider}s found via {@link ServiceLoader}.
	 *
	 * @throws Exception if an error occurs during discovery.
	 */
	public void discover() throws Exception {
		
		allKnowledge = new ModelKnowledge();
		
		modelKnowledges = new LinkedHashMap<>();
		registrationProviders = new LinkedHashMap<>();
		
		ServiceLoader<? extends RegistrationProvider> providers = ServiceLoader
				.load(RegistrationProvider.class);
		
		for(RegistrationProvider rp : providers) {
			
			registrationProviders.put(rp.getModelName(), rp);
			
			ModelKnowledge modelKnowledge = new ModelKnowledge();
			modelKnowledges.put(rp.getModelName(), modelKnowledge);
			modelKnowledge.addPackages(rp.getPackagesToRegister());
			modelKnowledge.createKnowledge();
			
			allKnowledge.addPackages(rp.getPackagesToRegister());
		}
		
		// After gathering all packages, create the aggregated knowledge
		// TODO find a better way to do this without rescanning everything
		allKnowledge.createKnowledge();
		
	}
	
	
	
}
