/**
 * 
 */
package org.gcube.informationsystem.discovery.knowledge;

import org.gcube.informationsystem.base.reference.Element;
import org.gcube.informationsystem.discovery.DiscoveredElementAction;
import org.gcube.informationsystem.model.reference.entities.Resource;
import org.gcube.informationsystem.tree.Tree;
import org.gcube.informationsystem.types.TypeMapper;
import org.gcube.informationsystem.types.reference.Type;
import org.gcube.informationsystem.types.reference.entities.ResourceType;

/**
 * A {@link DiscoveredElementAction} that adds discovered elements to a
 * {@link Tree} and updates usage knowledge.
 *
 * @author Luca Frosini (ISTI - CNR)
 */
public class AddElementToTreeAction implements DiscoveredElementAction<Element> {

	/** The tree to which discovered elements will be added. */
	protected Tree<Class<Element>> tree;
	
	/** The knowledge base for facet usage. */
	protected UsageKnowledge facetKnowledge;
	/** The knowledge base for resource usage. */
	protected UsageKnowledge resourceKnowledge;
	
	/**
	 * Returns the facet usage knowledge.
	 *
	 * @return The facet usage knowledge.
	 */
	public UsageKnowledge getFacetKnowledge() {
		return facetKnowledge;
	}

	/**
	 * Sets the facet usage knowledge.
	 *
	 * @param facetKnowledge The facet usage knowledge.
	 */
	public void setFacetKnowledge(UsageKnowledge facetKnowledge) {
		this.facetKnowledge = facetKnowledge;
	}

	/**
	 * Returns the resource usage knowledge.
	 *
	 * @return The resource usage knowledge.
	 */
	public UsageKnowledge getResourceKnowledge() {
		return resourceKnowledge;
	}

	/**
	 * Sets the resource usage knowledge.
	 *
	 * @param resourceKnowledge The resource usage knowledge.
	 */
	public void setResourceKnowledge(UsageKnowledge resourceKnowledge) {
		this.resourceKnowledge = resourceKnowledge;
	}

	/**
	 * Constructs a new action with the specified tree.
	 *
	 * @param tree The tree to which discovered elements will be added.
	 */
	public AddElementToTreeAction(Tree<Class<Element>> tree) {
		this.tree = tree;
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public void analizeElement(Class<Element> e) throws Exception {
		tree.addNode(e);
		createUsageKnowledge(e);
	}
	
	/**
	 * Creates usage knowledge for the given element by analyzing its type definition.
	 *
	 * @param e The element class.
	 */
	protected void createUsageKnowledge(Class<Element> e) {
		if (e.isAssignableFrom(Resource.class)) {
			return;
		}
		
		if(facetKnowledge!=null && resourceKnowledge!=null) {
			Type type = TypeMapper.createTypeDefinition(e);
			ResourceType rt = (ResourceType) type;
			facetKnowledge.addAll(rt.getFacets());
			resourceKnowledge.addAll(rt.getResources());
		}
	}

}
