/**
 * 
 */
package org.gcube.informationsystem.discovery;

import java.lang.reflect.Method;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.gcube.informationsystem.base.reference.Element;
import org.gcube.informationsystem.base.reference.properties.PropertyElement;
import org.gcube.informationsystem.types.annotations.ISProperty;
import org.gcube.informationsystem.utils.ReflectionUtility;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * A generic class for discovering all subtypes of a given root element within a
 * set of packages.
 *
 * @param <E> The root element type for the discovery process.
 * @author Luca Frosini (ISTI - CNR)
 */
public class Discovery<E extends Element> {
	
	/** The logger for this class. */
	public static Logger logger = LoggerFactory.getLogger(Discovery.class);
	
	/** The root class of the element hierarchy to discover. */
	protected final Class<E> root;
	/** The set of packages to be scanned. */
	protected final Set<Package> packages;
	/** The list of discovered element classes. */
	protected final List<Class<E>> discoveredElements;
	/** The list of actions to be executed on each discovered element. */
	protected final List<DiscoveredElementAction<Element>> discoveredElementActions;
	
	/** 
	 * A set of classes where the discovery process should stop. 
	 * It is mainly used to avoid analyzing Model Classes (Resource, Facet, IsRelatedTo, ConsistsOf, Property)
	 * when discoverying lower level classes.
	 * As an example when when root is EntityElement the discovery must stop to Resource and Facet classes
	 * which hierarchy is not interesting for that level and will be discovered
	 * by using Resource and Facet as root classes respectively.
	 */
	protected Set<Class<? extends E>> stopClasses;

	/** Whether the discovery process has been completed. */
	protected boolean discovered;
	
	/**
	 * Constructs a new {@code Discovery} instance for the given root type.
	 *
	 * @param root The root class of the element hierarchy to discover.
	 * @throws Exception if an error occurs during initialization.
	 */
	public Discovery(Class<E> root) throws Exception {
		this.root = root;
		this.packages = new HashSet<>();
		this.discovered = false;
		this.discoveredElements = new ArrayList<>();
		this.discoveredElementActions = new ArrayList<>();
		this.stopClasses = new HashSet<>();
		add(root);
	}
	
	/**
	 * Adds a class to the set of stop classes for the discovery process.
	 * The stop classes are not analyzed or added to the discovered elements.
	 * @param stopClass The class to add as a stop class.
	 */
	public void addStopClass(Class<? extends E> stopClass) {
		this.stopClasses.add(stopClass);
	}

	/**
	 * Returns the list of discovered element classes.
	 *
	 * @return A list of discovered classes.
	 */
	public List<Class<E>> getDiscoveredElements() {
		return discoveredElements;
	}
	
	/**
	 * Adds a {@link DiscoveredElementAction} to be executed on each discovered element.
	 *
	 * @param discoveredElementAction The action to add.
	 * @throws Exception if an error occurs while analyzing existing elements.
	 */
	@SuppressWarnings("unchecked")
	public synchronized void addDiscoveredElementActions(DiscoveredElementAction<Element> discoveredElementAction) throws Exception {
		discoveredElementActions.add(discoveredElementAction);
		if(discovered) {
			for(Class<E> clz : discoveredElements) {
				discoveredElementAction.analizeElement((Class<Element>) clz);
			}
		}
	}
	
	/**
	 * Executes a given {@link DiscoveredElementAction} on all already discovered elements.
	 *
	 * @param discoveredElementAction The action to execute.
	 * @throws Exception if an error occurs during execution.
	 */
	@SuppressWarnings("unchecked")
	public synchronized void executeDiscoveredElementActions(DiscoveredElementAction<Element> discoveredElementAction) throws Exception {
		if(discovered) {
			for(Class<E> clz : discoveredElements) {
				discoveredElementAction.analizeElement((Class<Element>) clz);
			}
		}
	}
	
	/**
	 * Adds a package to be scanned during discovery.
	 *
	 * @param p The package to add.
	 */
	public void addPackage(Package p) {
		packages.add(p);
	}
	
	/**
	 * Adds a collection of packages to be scanned.
	 *
	 * @param packages The collection of packages.
	 */
	public void addPackages(Collection<Package> packages) {
		for(Package p : packages) {
			addPackage(p);
		}
	}

	/**
	 * Adds a class to the list of discovered elements and notifies any registered actions.
	 *
	 * @param clz The class to add.
	 * @throws Exception if an error occurs in a {@link DiscoveredElementAction}.
	 */
	@SuppressWarnings("unchecked")
	protected void add(Class<E> clz) throws Exception {
		if(!discoveredElements.contains(clz)) {
			discoveredElements.add(clz);
			for(DiscoveredElementAction<Element> discoveredElementAction : discoveredElementActions) {
				discoveredElementAction.analizeElement((Class<Element>) clz);
			}
			logger.info("+ Added {}.", clz);
		}
	}
	
	/**
	 * Recursively analyzes a class to find subtypes of the root element.
	 *
	 * @param clz The class to analyze.
	 * @throws Exception if an error occurs during analysis.
	 */
	protected void analizeElement(Class<E> clz) throws Exception {
		logger.trace("Analizyng {}", clz);
		
		if(!clz.isInterface()) {
			logger.trace("- Discarding {} because is not an interface", clz);
			return;
		}
		
		if(!root.isAssignableFrom(clz)) {
			logger.trace("- Discarding {} because is not a {}", clz, root.getSimpleName());
			return;
		}
		
		if(discoveredElements.contains(clz)) {
			logger.trace("- Discarding {} because was already managed", clz);
			return;
		}

		if(stopClasses.contains(clz)) {
			logger.trace("- Discarding {} because is a stop class", clz);
			return;
		}
		
		Class<?>[] interfaces = clz.getInterfaces();
		
		for(Class<?> interfaceClass : interfaces) {
			@SuppressWarnings("unchecked")
			Class<E> parent = (Class<E>) interfaceClass;
			analizeElement(parent);
		}
		
		if(root == PropertyElement.class) {
			
			for(Method m : clz.getDeclaredMethods()) {
				m.setAccessible(true);
				if(m.isAnnotationPresent(ISProperty.class)) {
					if(Map.class.isAssignableFrom(m.getReturnType()) || Set.class.isAssignableFrom(m.getReturnType())
							|| List.class.isAssignableFrom(m.getReturnType())) {
						
						Type[] typeArguments = ((ParameterizedType) m.getGenericReturnType()).getActualTypeArguments();
						for(Type t : typeArguments) {
							@SuppressWarnings("unchecked")
							Class<? extends PropertyElement> tClass = (Class<? extends PropertyElement>) t;
							if(root.isAssignableFrom(tClass)) {
								@SuppressWarnings("unchecked")
								Class<E> type = (Class<E>) tClass;
								analizeElement(type);
							}
						}
						
					} else if(root.isAssignableFrom(m.getReturnType())) {
						@SuppressWarnings("unchecked")
						Class<E> eClz = (Class<E>) m.getReturnType();
						analizeElement(eClz);
					}
				}
				
			}
		}
		add(clz);
	}
	
	/**
	 * Starts the discovery process, scanning all registered packages.
	 *
	 * @throws Exception if an error occurs during discovery.
	 */
	public synchronized void discover() throws Exception {
		this.discovered = false;
		for(Package p : packages) {
			List<Class<?>> classes = ReflectionUtility.getClassesForPackage(p);
			for(Class<?> clz : classes) {
				@SuppressWarnings("unchecked")
				Class<E> eClz = (Class<E>) clz;
				analizeElement(eClz);
			}
		}
		this.discovered = true;
	}
	
}
