/**
 * 
 */
package org.gcube.informationsystem.contexts.reference.entities;

import java.util.List;
import java.util.Map;
import java.util.UUID;

import org.gcube.com.fasterxml.jackson.annotation.JsonAnyGetter;
import org.gcube.com.fasterxml.jackson.annotation.JsonAnySetter;
import org.gcube.com.fasterxml.jackson.annotation.JsonGetter;
import org.gcube.com.fasterxml.jackson.annotation.JsonIgnore;
import org.gcube.com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import org.gcube.com.fasterxml.jackson.annotation.JsonPropertyOrder;
import org.gcube.com.fasterxml.jackson.databind.annotation.JsonSerialize;
import org.gcube.informationsystem.base.reference.Element;
import org.gcube.informationsystem.base.reference.IdentifiableElement;
import org.gcube.informationsystem.base.reference.SchemaMixedElement;
import org.gcube.informationsystem.base.reference.entities.EntityElement;
import org.gcube.informationsystem.contexts.impl.entities.ContextImpl;
import org.gcube.informationsystem.contexts.reference.relations.IsParentOf;
import org.gcube.informationsystem.model.reference.relations.Relation;
import org.gcube.informationsystem.serialization.AdditionalPropertiesSerializer;
import org.gcube.informationsystem.types.annotations.Deserialize;
import org.gcube.informationsystem.types.annotations.Final;
import org.gcube.informationsystem.types.annotations.ISProperty;
import org.gcube.informationsystem.types.reference.Change;
import org.gcube.informationsystem.types.reference.TypeMetadata;
import org.gcube.informationsystem.utils.Version;

/**
 * Represents a context, which is a hierarchical grouping mechanism for resources.
 * <p>
 * Contexts form a tree structure and have a lifecycle managed by their state.
 * For more details, refer to the
 * <a href="https://wiki.gcube-system.org/gcube/Facet_Based_Resource_Model#Context">
 * gCube Facet Based Resource Model Wiki</a>.
 *
 * @author Luca Frosini (ISTI - CNR)
 */
//@JsonDeserialize(as = ContextImpl.class)
@Deserialize(as = ContextImpl.class)
@JsonPropertyOrder({ Element.TYPE_PROPERTY, IdentifiableElement.ID_PROPERTY, IdentifiableElement.METADATA_PROPERTY})
@TypeMetadata(name = Context.NAME, description = "This type is the used to define a Context", version = Version.MINIMAL_VERSION_STRING)
@Change(version = Version.MINIMAL_VERSION_STRING, description = Version.MINIMAL_VERSION_DESCRIPTION)
@Final
public interface Context extends EntityElement, SchemaMixedElement {

	/** The name of the Context entity type. */
	public static final String NAME = "Context"; // Context.class.getSimpleName();

	/** The property name for the context's name. */
	public static final String NAME_PROPERTY = "name";
	/** The property name for the parent relationship. */
	public static final String PARENT_PROPERTY = "parent";
	/** The property name for the children relationships. */
	public static final String CHILDREN_PROPERTY = "children";
	
	/**
	 * The property name for the state of the context.
	 * <p>
	 * The resource-registry manages the following states:
	 * <ul>
	 * <li><b>created:</b> A newly created context.</li>
	 * <li><b>active:</b> An operational context.</li>
	 * <li><b>deleted:</b> A context marked for deletion.</li>
	 * </ul>
	 * Other states can be defined and managed by a Manager.
	 */
	public static final String STATE = "state";
	
	/**
	 * Returns the name of the context.
	 *
	 * @return The context name.
	 */
	@ISProperty(name = NAME_PROPERTY, mandatory = true, nullable = false)
	public String getName();

	/**
	 * Sets the name of the context.
	 *
	 * @param name The name to set.
	 */
	public void setName(String name);
	
	/**
	 * Returns the current state of the context as a string.
	 *
	 * @return The context state.
	 * @see org.gcube.informationsystem.contexts.reference.ContextState
	 */
	@ISProperty(name = STATE, mandatory=true, nullable=false, regexpr = "[a-z]+")
	public String getState();
	
	/**
	 * Sets the state of the context.
	 *
	 * @param state The state to set.
	 */
	public void setState(String state);

	/**
	 * Returns the parent of this context.
	 *
	 * @return The {@link IsParentOf} relation to the parent, or {@code null} if this is a root context.
	 */
	@JsonGetter
	@JsonIgnoreProperties({ Relation.TARGET_PROPERTY })
	public IsParentOf getParent();

	/**
	 * Sets the parent of this context by its UUID.
	 *
	 * @param uuid The UUID of the parent context.
	 */
	@JsonIgnore
	public void setParent(UUID uuid);

	/**
	 * Sets the parent of this context.
	 *
	 * @param context The parent context.
	 */
	@JsonIgnore
	public void setParent(Context context);

	/**
	 * Sets the parent of this context via an {@link IsParentOf} relation.
	 *
	 * @param isParentOf The relation to the parent.
	 */
	@JsonIgnore
	public void setParent(IsParentOf isParentOf);

	/**
	 * Returns the children of this context.
	 *
	 * @return A list of {@link IsParentOf} relations to the children.
	 */
	@JsonGetter
	@JsonIgnoreProperties({ Relation.SOURCE_PROPERTY })
	public List<IsParentOf> getChildren();

	/**
	 * Adds a child to this context by its UUID.
	 *
	 * @param uuid The UUID of the child context.
	 */
	public void addChild(UUID uuid);

	/**
	 * Adds a child to this context.
	 *
	 * @param child The child context.
	 */
	public void addChild(Context child);

	/**
	 * Adds a child to this context via an {@link IsParentOf} relation.
	 *
	 * @param isParentOf The relation to the child.
	 */
	public void addChild(IsParentOf isParentOf);
	
	
	/**
	 * {@inheritDoc}
	 */
	@JsonAnyGetter
	@JsonSerialize(using = AdditionalPropertiesSerializer.class)
	@Override
	public Map<String,Object> getAdditionalProperties();
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setAdditionalProperties(Map<String,Object> additionalProperties);
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public Object getAdditionalProperty(String key);
	
	/**
	 * {@inheritDoc}
	 */
	@JsonAnySetter
	@Override
	public void setAdditionalProperty(String key, Object value);

}
