package org.gcube.informationsystem.contexts.reference;

import java.util.Collections;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

/**
 * Represents the lifecycle state of a {@link org.gcube.informationsystem.contexts.reference.entities.Context}.
 *
 * @author Luca Frosini (ISTI - CNR)
 */
public enum ContextState {

	/** The context has been created but is not yet active. */
	CREATED("created"),
	/** The context is active and fully operational. */
	ACTIVE("active"),
	/** The context has been temporarily suspended and is not operational. */
	SUSPENDED("suspended"),
	/** The context has been marked for deletion. */
	DELETED("deleted");
	
	private static final Map<String,ContextState> ENUM_MAP;
	
	static {
		ContextState[] array = ContextState.values();
        Map<String,ContextState> map = new ConcurrentHashMap<>(array.length);
        for (ContextState cs : array) {
            map.put(cs.getState().toLowerCase(),cs);
        }
        ENUM_MAP = Collections.unmodifiableMap(map);
    }
	
	/** The string representation of the state. */
	protected String state;
	
	ContextState(String state) {
		this.state = state;
	}
	
	/**
	 * Returns the string representation of the state.
	 *
	 * @return The state as a lowercase string.
	 */
	public String getState() {
		return state;
	}

	/**
	 * Returns the {@code ContextState} enum constant from its string representation.
	 *
	 * @param state The string representation of the state (case-insensitive).
	 * @return The corresponding {@code ContextState}, or {@code null} if not found.
	 */
	public static ContextState fromString(String state) {
        return ENUM_MAP.get(state.toLowerCase());
    }
	
}
