package org.gcube.informationsystem.contexts.impl.entities;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;

import org.gcube.com.fasterxml.jackson.annotation.JsonIgnore;
import org.gcube.com.fasterxml.jackson.annotation.JsonSetter;
import org.gcube.com.fasterxml.jackson.annotation.JsonTypeName;
import org.gcube.com.fasterxml.jackson.core.JsonProcessingException;
import org.gcube.informationsystem.base.impl.entities.EntityElementImpl;
import org.gcube.informationsystem.base.reference.Element;
import org.gcube.informationsystem.contexts.impl.relations.IsParentOfImpl;
import org.gcube.informationsystem.contexts.reference.entities.Context;
import org.gcube.informationsystem.contexts.reference.relations.IsParentOf;
import org.gcube.informationsystem.model.impl.properties.MetadataImpl;
import org.gcube.informationsystem.model.reference.properties.Property;
import org.gcube.informationsystem.serialization.ElementMapper;
import org.gcube.informationsystem.utils.UUIDManager;

/**
 * The default implementation of the {@link Context} interface.
 *
 * @author Luca Frosini (ISTI - CNR)
 */
@JsonTypeName(value = Context.NAME)
public class ContextImpl extends EntityElementImpl implements Context {

	/**
	 * The generated serial version UID for serialization.
	 */
	private static final long serialVersionUID = -5070590328223454087L;

	/** The name of the context. */
	protected String name;

	/** The state of the context. */
	protected String state;

	/** The parent of the context. */
	protected IsParentOf parent;
	/** The children of the context. */
	protected List<IsParentOf> children;

	/** Additional properties of the context. */
	@JsonIgnore
	protected Map<String, Object> additionalProperties;

	/**
	 * A set of keys that are allowed as additional properties, even if they start
	 * with '_' or '@'.
	 */
	@JsonIgnore
	protected final Set<String> allowedAdditionalKeys;

	/**
	 * Default constructor.
	 */
	protected ContextImpl() {
		super();
		this.parent = null;
		this.children = new ArrayList<>();
		this.additionalProperties = new HashMap<>();
		this.allowedAdditionalKeys = new HashSet<>();
	}

	/**
	 * Constructs a new context with a given UUID.
	 *
	 * @param uuid The UUID.
	 */
	public ContextImpl(UUID uuid) {
		this(null, uuid);
	}

	/**
	 * Constructs a new context with a given name.
	 *
	 * @param name The name.
	 */
	public ContextImpl(String name) {
		this(name, null);
	}

	/**
	 * Constructs a new context with a given name and UUID.
	 *
	 * @param name The name.
	 * @param uuid The UUID.
	 */
	public ContextImpl(String name, UUID uuid) {
		this();
		this.name = name;
		if (uuid == null) {
			uuid = UUIDManager.getInstance().generateValidUUID();
		}
		this.uuid = uuid;
		this.metadata = new MetadataImpl();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getName() {
		return this.name;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setName(String name) {
		this.name = name;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getState() {
		return this.state;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setState(String state) {
		this.state = state;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public IsParentOf getParent() {
		return parent;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setParent(UUID uuid) {
		Context parent = null;
		if (uuid != null) {
			parent = new ContextImpl(uuid);
			parent.setMetadata(new MetadataImpl());
		}
		setParent(parent);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setParent(Context context) {
		IsParentOf isParentOf = null;
		if (context != null) {
			isParentOf = new IsParentOfImpl(context, this);
		}
		setParent(isParentOf);
	}

	/**
	 * Sets the parent from a JSON-deserialized {@link IsParentOf} object.
	 *
	 * @param isParentOf The deserialized parent relationship.
	 * @throws JsonProcessingException if an error occurs during processing.
	 */
	@JsonSetter(value = PARENT_PROPERTY)
	protected void setParentFromJson(IsParentOf isParentOf) throws JsonProcessingException {
		if (isParentOf != null) {
			Context parent = isParentOf.getSource();
			isParentOf.setTarget(this);
			((ContextImpl) parent).addChild(isParentOf);
		}
		setParent(isParentOf);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setParent(IsParentOf isParentOf) {
		this.parent = isParentOf;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public List<IsParentOf> getChildren() {
		return children;
	}

	/**
	 * Sets the children from a JSON-deserialized list of {@link IsParentOf}
	 * objects.
	 *
	 * @param children The list of deserialized child relationships.
	 * @throws JsonProcessingException if an error occurs during processing.
	 */
	@JsonSetter(value = CHILDREN_PROPERTY)
	protected void setChildrenFromJson(List<IsParentOf> children) throws JsonProcessingException {
		for (IsParentOf isParentOf : children) {
			addChildFromJson(isParentOf);
		}
	}

	/**
	 * Adds a child from a JSON-deserialized {@link IsParentOf} object.
	 *
	 * @param isParentOf The deserialized child relationship.
	 * @throws JsonProcessingException if an error occurs during processing.
	 */
	protected void addChildFromJson(IsParentOf isParentOf) throws JsonProcessingException {
		isParentOf.setSource(this);
		addChild(isParentOf);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void addChild(UUID uuid) {
		Context child = new ContextImpl(uuid);
		child.setMetadata(new MetadataImpl());
		addChild(child);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void addChild(Context child) {
		IsParentOf isParentOf = new IsParentOfImpl(this, child);
		this.addChild(isParentOf);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void addChild(IsParentOf isParentOf) {
		// ((ContextImpl) isParentOf.getTarget()).setParent(this);
		isParentOf.setSource(this);
		children.add(isParentOf);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Map<String, Object> getAdditionalProperties() {
		return additionalProperties;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setAdditionalProperties(Map<String, Object> additionalProperties) {
		this.additionalProperties = additionalProperties;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Object getAdditionalProperty(String key) {
		return additionalProperties.get(key);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setAdditionalProperty(String key, Object value) {

		if (!allowedAdditionalKeys.contains(key)) {
			if (key.startsWith("_")) {
				return;
			}
			if (key.startsWith("@")) {
				return;
			}
		}

		/*
		 * Additional properties are not deserialized to the proper property type. The
		 * first attempt was to try to write a specific deserializer but it fails. This
		 * fix the issue.
		 */
		try {
			if (value instanceof Map<?, ?>) {
				@SuppressWarnings("unchecked")
				Map<String, Object> map = (Map<String, Object>) value;
				if (map.containsKey(Element.TYPE_PROPERTY)) {
					String reserialized = ElementMapper.getObjectMapper().writeValueAsString(map);
					Property property = ElementMapper.unmarshal(Property.class, reserialized);
					value = property;
				}
			}
		} catch (Throwable e) {
			e.getMessage();
			// Any type of error/exception must be catched
		}
		/* END of fix to properly deserialize Property types */

		this.additionalProperties.put(key, value);
	}

}
