package org.gcube.informationsystem.base.reference.relations;

import java.util.UUID;

import org.gcube.com.fasterxml.jackson.annotation.JsonGetter;
import org.gcube.com.fasterxml.jackson.annotation.JsonIgnore;
import org.gcube.informationsystem.base.reference.Element;
import org.gcube.informationsystem.base.reference.IdentifiableElement;
import org.gcube.informationsystem.base.reference.entities.EntityElement;
import org.gcube.informationsystem.model.reference.properties.Metadata;
import org.gcube.informationsystem.types.annotations.Abstract;
import org.gcube.informationsystem.types.annotations.ISProperty;
import org.gcube.informationsystem.types.reference.Change;
import org.gcube.informationsystem.types.reference.TypeMetadata;
import org.gcube.informationsystem.utils.Version;

/**
 * The base interface for all relation elements in the Information System model.
 * <p>
 * This interface defines a directed relationship between a source and a target
 * entity, both of which must be subtypes of {@link EntityElement}. It extends
 * {@link IdentifiableElement} to ensure that every relation is uniquely
 * identifiable.
 *
 * @param <S> The type of the source entity.
 * @param <T> The type of the target entity.
 * @author Luca Frosini (ISTI - CNR)
 */
@SuppressWarnings("unused")
@Abstract
//@JsonDeserialize(as=RelationElementImpl.class) Do not uncomment to manage subclasses
@TypeMetadata(name = RelationElement.NAME, description = "This is the base type for any RelationElement", version = Version.MINIMAL_VERSION_STRING)
@Change(version = Version.MINIMAL_VERSION_STRING, description = Version.MINIMAL_VERSION_DESCRIPTION)
public interface RelationElement<S extends EntityElement, T extends EntityElement> extends Element, IdentifiableElement {
	
	/** The name of the relation element type. */
	public static final String NAME = "RelationElement"; // RelationElement.class.getSimpleName();
	
	/** The property name for the source of the relation. */
	public static final String SOURCE_PROPERTY = "source";
	/** The property name for the target of the relation. */
	public static final String TARGET_PROPERTY = "target";
	
	/**
	 * {@inheritDoc}
	 * <p>
	 * This UUID is used to uniquely identify the relation instance. It is read-only,
	 * mandatory, and cannot be null.
	 */
	@ISProperty(name = ID_PROPERTY, description = "This UUID is be used to identify the instance univocally.", readonly = true, mandatory = true, nullable = false)
	@Override
	public UUID getID();

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setID(UUID uuid);
	
	/**
	 * {@inheritDoc}
	 * <p>
	 * Metadata is automatically managed by the system and is mandatory.
	 */
	@ISProperty(name=METADATA_PROPERTY, mandatory=true, nullable=false, description="Metadata associated with the instance that is automatically created/updated by the system.")
	@Override
	public Metadata getMetadata();
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setMetadata(Metadata metadata);
	
	/**
	 * Returns the source entity of the relation.
	 *
	 * @return The source entity.
	 */
	@JsonGetter(value=SOURCE_PROPERTY)
	public S getSource();
	
	/**
	 * Sets the source entity of the relation.
	 *
	 * @param source The source entity to set.
	 */
	@JsonIgnore
	public void setSource(S source);
	
	/**
	 * Returns the target entity of the relation.
	 *
	 * @return The target entity.
	 */
	@JsonGetter(value=TARGET_PROPERTY)
	public T getTarget();
	
	/**
	 * Sets the target entity of the relation.
	 *
	 * @param target The target entity to set.
	 */
	@JsonIgnore
	public void setTarget(T target);
	
}