package org.gcube.informationsystem.base.reference;

import java.util.UUID;

import org.gcube.com.fasterxml.jackson.annotation.JsonGetter;
import org.gcube.com.fasterxml.jackson.annotation.JsonInclude;
import org.gcube.com.fasterxml.jackson.annotation.JsonInclude.Include;
import org.gcube.com.fasterxml.jackson.annotation.JsonPropertyOrder;
import org.gcube.com.fasterxml.jackson.annotation.JsonSetter;
import org.gcube.informationsystem.model.reference.properties.Metadata;

/**
 * An element that can be uniquely identified by a {@link UUID} and may contain {@link Metadata}.
 * <p>
 * This interface extends the base {@link Element} to add properties for a unique
 * identifier (ID) and associated metadata. It is designed for elements that
 * require a persistent and globally unique identity.
 *
 * @author Luca Frosini (ISTI - CNR)
 * @see Element
 * @see Metadata
 */
@JsonPropertyOrder({ Element.TYPE_PROPERTY, IdentifiableElement.ID_PROPERTY, IdentifiableElement.METADATA_PROPERTY })
public interface IdentifiableElement extends Element {
	
	/**
	 * The name of the property used to store the unique identifier (UUID) of the element.
	 */
	public static final String ID_PROPERTY = "id";

	/**
	 * The name of the property used to store the metadata associated with the element.
	 */
	public static final String METADATA_PROPERTY = "metadata";
	
	/**
	 * Returns the unique identifier (UUID) of the element.
	 *
	 * @return The {@link UUID} of the element.
	 */
	@JsonGetter(value = ID_PROPERTY)
	public UUID getID();

	/**
	 * Sets the unique identifier (UUID) for the element.
	 *
	 * @param uuid The {@link UUID} to set.
	 */
	@JsonSetter(value = ID_PROPERTY)
	public void setID(UUID uuid);
	
	/**
	 * Returns the metadata associated with the element.
	 * <p>
	 * The metadata is included in JSON serialization only if it is not null.
	 *
	 * @return The {@link Metadata} of the element, or {@code null} if none is set.
	 */
	@JsonInclude(Include.NON_NULL)
	@JsonGetter(value = METADATA_PROPERTY)
	public Metadata getMetadata();
	
	/**
	 * Sets the metadata for the element.
	 *
	 * @param metadata The {@link Metadata} to associate with the element.
	 */
	@JsonSetter(value = METADATA_PROPERTY)
	public void setMetadata(Metadata metadata);
	
}
