package org.gcube.informationsystem.base.reference;

import org.gcube.com.fasterxml.jackson.annotation.JsonIgnore;
import org.gcube.informationsystem.types.PropertyTypeName;

/**
 * Defines the contract for an attribute, which represents a characteristic of an
 * entity or property.
 * <p>
 * An attribute has a name, description, type, and optional constraints such as
 * minimum/maximum values and a regular expression for validation. It can also
_ * have a default value.
 *
 * @author Luca Frosini (ISTI - CNR)
 */
public interface Attribute {
	
	/** The name of the attribute type. */
	public static final String NAME = "Attribute"; // Attribute.class.getSimpleName();
	
	/** The property name for the attribute's name. */
	public static final String NAME_PROPERTY = "name";
	/** The property name for the attribute's description. */
	public static final String DESCRIPTION_PROPERTY = "description";
	/** The property name for the attribute's maximum value constraint. */
	public static final String MAX_PROPERTY = "max";
	/** The property name for the attribute's minimum value constraint. */
	public static final String MIN_PROPERTY = "min";
	/** The property name for the attribute's regular expression constraint. */
	public static final String REGEX_PROPERTY = "regexp";
	/** The property name for the attribute's property type. */
	public static final String PROPERTY_TYPE_PROPERTY = "propertyType";
	/** The property name for the attribute's default value. */
	public static final String DEFAULT_VALUE_PROPERTY = "defaultValue";
	
	/**
	 * Returns the name of the attribute.
	 *
	 * @return The attribute's name.
	 */
	public String getName();
	
	/**
	 * Sets the name of the attribute.
	 *
	 * @param name The name to set.
	 */
	public void setName(String name);
	
	/**
	 * Returns the description of the attribute.
	 *
	 * @return The attribute's description.
	 */
	public String getDescription();
	
	/**
	 * Sets the description of the attribute.
	 *
	 * @param description The description to set.
	 */
	public void setDescription(String description);
	
	/**
	 * Returns the property type of the attribute as a string.
	 *
	 * @return The property type.
	 */
	public String getPropertyType();
	
	/**
	 * Sets the property type of the attribute.
	 *
	 * @param type The property type to set.
	 */
	public void setPropertyType(String type);
	
	/**
	 * Returns the property type as a {@link PropertyTypeName} enum.
	 * <p>
	 * This method is ignored during JSON serialization.
	 *
	 * @return The {@link PropertyTypeName} enum constant.
	 */
	@JsonIgnore
	public PropertyTypeName getPropertyTypeName();
	
	/**
	 * Returns the default value of the attribute.
	 *
	 * @return The default value.
	 */
	public Object getDefaultValue();

	/**
	 * Sets the default value of the attribute.
	 *
	 * @param defaultValue The default value to set.
	 */
	public void setDefaultValue(Object defaultValue);
	
	/**
	 * Returns the maximum allowed value for the attribute.
	 *
	 * @return The maximum value, or {@code null} if not set.
	 */
	public Integer getMax();
	
	/**
	 * Sets the maximum allowed value for the attribute.
	 *
	 * @param max The maximum value to set.
	 */
	public void setMax(Integer max);
	
	/**
	 * Returns the minimum allowed value for the attribute.
	 *
	 * @return The minimum value, or {@code null} if not set.
	 */
	public Integer getMin();
	
	/**
	 * Sets the minimum allowed value for the attribute.
	 *
	 * @param min The minimum value to set.
	 */
	public void setMin(Integer min);
	
	/**
	 * Returns the regular expression used to validate the attribute's value.
	 *
	 * @return The regular expression, or {@code null} if not set.
	 */
	public String getRegexp();
	
	/**
	 * Sets the regular expression for validating the attribute's value.
	 *
	 * @param regexp The regular expression to set.
	 */
	public void setRegexp(String regexp);
	
}