package org.gcube.informationsystem.base.reference;

import java.util.Arrays;
import java.util.HashSet;
import java.util.Set;

import org.gcube.informationsystem.base.impl.entities.EntityElementImpl;
import org.gcube.informationsystem.base.impl.properties.PropertyElementImpl;
import org.gcube.informationsystem.base.impl.relations.RelationElementImpl;
import org.gcube.informationsystem.base.reference.entities.EntityElement;
import org.gcube.informationsystem.base.reference.properties.PropertyElement;
import org.gcube.informationsystem.base.reference.relations.RelationElement;
import org.gcube.informationsystem.contexts.impl.entities.ContextImpl;
import org.gcube.informationsystem.contexts.impl.relations.IsParentOfImpl;
import org.gcube.informationsystem.contexts.reference.entities.Context;
import org.gcube.informationsystem.contexts.reference.relations.IsParentOf;
import org.gcube.informationsystem.model.impl.entities.DummyFacet;
import org.gcube.informationsystem.model.impl.entities.DummyResource;
import org.gcube.informationsystem.model.impl.entities.EntityImpl;
import org.gcube.informationsystem.model.impl.entities.FacetImpl;
import org.gcube.informationsystem.model.impl.entities.ResourceImpl;
import org.gcube.informationsystem.model.impl.properties.PropertyImpl;
import org.gcube.informationsystem.model.impl.relations.ConsistsOfImpl;
import org.gcube.informationsystem.model.impl.relations.DummyIsRelatedTo;
import org.gcube.informationsystem.model.impl.relations.IsRelatedToImpl;
import org.gcube.informationsystem.model.impl.relations.RelationImpl;
import org.gcube.informationsystem.model.reference.entities.Entity;
import org.gcube.informationsystem.model.reference.entities.Facet;
import org.gcube.informationsystem.model.reference.entities.Resource;
import org.gcube.informationsystem.model.reference.properties.Property;
import org.gcube.informationsystem.model.reference.relations.ConsistsOf;
import org.gcube.informationsystem.model.reference.relations.IsRelatedTo;
import org.gcube.informationsystem.model.reference.relations.Relation;
import org.gcube.informationsystem.queries.templates.impl.entities.QueryTemplateImpl;
import org.gcube.informationsystem.queries.templates.impl.properties.TemplateVariableImpl;
import org.gcube.informationsystem.queries.templates.reference.entities.QueryTemplate;
import org.gcube.informationsystem.queries.templates.reference.properties.TemplateVariable;
import org.gcube.informationsystem.types.impl.entities.EntityTypeImpl;
import org.gcube.informationsystem.types.impl.properties.PropertyDefinitionImpl;
import org.gcube.informationsystem.types.impl.properties.PropertyTypeImpl;
import org.gcube.informationsystem.types.impl.relations.RelationTypeImpl;
import org.gcube.informationsystem.types.reference.entities.EntityType;
import org.gcube.informationsystem.types.reference.properties.PropertyDefinition;
import org.gcube.informationsystem.types.reference.properties.PropertyType;
import org.gcube.informationsystem.types.reference.relations.RelationType;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Enumerates the fundamental types of elements within the Information System model.
 * <p>
 * This enum serves as a central registry for all element types, mapping them to
 * their corresponding interface, implementation, and (where applicable) dummy
 * implementation classes. It provides utility methods for type lookup and
 * classification.
 *
 * @author Luca Frosini (ISTI - CNR)
 */
public enum AccessType {
	
	/** Represents a property element, the base for all property types. */
	PROPERTY_ELEMENT(PropertyElement.class, PropertyElement.NAME, PropertyElementImpl.class, null),
	/** Represents the definition of a property, including constraints and metadata. */
	PROPERTY_DEFINITION(PropertyDefinition.class, PropertyDefinition.NAME, PropertyDefinitionImpl.class, null),
	/** Represents the type of a property, defining its data type and structure. */
	PROPERTY_TYPE(PropertyType.class, PropertyType.NAME, PropertyTypeImpl.class, null),
	/** Represents a variable used within a query template. */
	TEMPLATE_VARIABLE(TemplateVariable.class, TemplateVariable.NAME, TemplateVariableImpl.class, null),
	/** Represents a generic property instance. */
	PROPERTY(Property.class, Property.NAME, PropertyImpl.class, null),
	
	/** Represents an entity element, the base for all entity types. */
	ENTITY_ELEMENT(EntityElement.class, EntityElement.NAME, EntityElementImpl.class, null),
	/** Represents the type of an entity, defining its structure and attributes. */
	ENTITY_TYPE(EntityType.class, EntityType.NAME, EntityTypeImpl.class, null),
	/** Represents a template for constructing queries. */
	QUERY_TEMPLATE(QueryTemplate.class, QueryTemplate.NAME, QueryTemplateImpl.class, null),
	/** Represents a hierarchical context for organizing resources. */
	CONTEXT(Context.class, Context.NAME, ContextImpl.class, null),
	/** Represents a generic entity instance. */
	ENTITY(Entity.class, Entity.NAME, EntityImpl.class, null),
	/** Represents a primary resource in the system. */
	RESOURCE(Resource.class, Resource.NAME, ResourceImpl.class, DummyResource.class),
	/** Represents a facet, which adds supplementary information to a resource. */
	FACET(Facet.class, Facet.NAME, FacetImpl.class, DummyFacet.class),
		
	/** Represents a relation element, the base for all relation types. */
	RELATION_ELEMENT(RelationElement.class, RelationElement.NAME, RelationElementImpl.class, null),
	/** Represents the type of a relation, defining its semantics and constraints. */
	RELATION_TYPE(RelationType.class, RelationType.NAME, RelationTypeImpl.class, null),
	/** Represents a parent-child relationship between contexts. */
	IS_PARENT_OF(IsParentOf.class, IsParentOf.NAME, IsParentOfImpl.class, null),
	/** Represents a generic relation instance. */
	RELATION(Relation.class, Relation.NAME, RelationImpl.class, null),
	/** Represents a relationship between two resources. */
	IS_RELATED_TO(IsRelatedTo.class, IsRelatedTo.NAME, IsRelatedToImpl.class, DummyIsRelatedTo.class),
	/** Represents a composition relationship, where one resource is part of another. */
	CONSISTS_OF(ConsistsOf.class, ConsistsOf.NAME, ConsistsOfImpl.class, null);
	
	
	private static Logger logger = LoggerFactory.getLogger(AccessType.class);
	
	private static AccessType[] modelTypes;
	private static AccessType[] erTypes;
	private static Set<String> names;
	
	static {
		names = new HashSet<>();
		AccessType[] accessTypes = AccessType.values();
		for (AccessType accessType : accessTypes) {
			String name = accessType.getName();
			names.add(name);
		}
		
		modelTypes = new AccessType[] {
				AccessType.PROPERTY,
				AccessType.RESOURCE, AccessType.FACET,
				AccessType.IS_RELATED_TO, AccessType.CONSISTS_OF
		};
		
		erTypes = new AccessType[] {
				AccessType.RESOURCE, AccessType.FACET,
				AccessType.IS_RELATED_TO, AccessType.CONSISTS_OF
		};
	}
	
	private final Class<? extends Element> clz;
	private final Class<? extends Element> implementationClass;
	private final Class<? extends Element> dummyImplementationClass;
	
	private final String name;
	private final String lowerCaseFirstCharacter;
	
	<ISM extends Element, ISMC extends ISM, ISMD extends ISMC>
	AccessType(Class<ISM> clz, String name, Class<ISMC> implementationClass, Class<ISMD> dummyImplementationClass){
		this.clz = clz;
		this.implementationClass = implementationClass;
		this.dummyImplementationClass = dummyImplementationClass;
		this.name = name;
		this.lowerCaseFirstCharacter = name.substring(0, 1).toLowerCase() + name.substring(1);
	}
	
	/**
	 * Returns the interface class associated with this access type.
	 *
	 * @param <ISM> The element type.
	 * @return The {@link Class} of the element's interface.
	 */
	@SuppressWarnings("unchecked")
	public <ISM extends Element> Class<ISM> getTypeClass(){
		return (Class<ISM>) clz;
	}
	
	/**
	 * Returns the primary implementation class for this access type.
	 *
	 * @param <ISM>  The element type.
	 * @param <ISMC> The element's implementation type.
	 * @return The {@link Class} of the primary implementation.
	 */
	@SuppressWarnings("unchecked")
	public <ISM extends Element, ISMC extends ISM> Class<ISMC> getImplementationClass() {
		return (Class<ISMC>) implementationClass;
	}
	
	/**
	 * Returns the dummy implementation class for this access type, if one exists.
	 * <p>
	 * Dummy implementations are used for lightweight representations or placeholders.
	 *
	 * @param <ISM>  The element type.
	 * @param <ISMC> The element's implementation type.
	 * @param <ISMD> The element's dummy implementation type.
	 * @return The {@link Class} of the dummy implementation, or {@code null}.
	 */
	@SuppressWarnings("unchecked")
	public <ISM extends Element, ISMC extends ISM, ISMD extends ISMC> Class<ISMD> getDummyImplementationClass() {
		return (Class<ISMD>) dummyImplementationClass;
	}
	
	/**
	 * Returns the official name of the access type.
	 *
	 * @return The type name.
	 */
	public String getName(){
		return name;
	}
	
	/**
	 * Returns the name of the access type with its first character in lowercase.
	 *
	 * @return The name with a lowercase first character.
	 */
	public String lowerCaseFirstCharacter() {
		return lowerCaseFirstCharacter;
	}
	
	@Override
	public String toString(){
		return name;
	}
	
	/**
	 * Returns a set of all defined access type names.
	 *
	 * @return A {@link Set} of type names.
	 */
	public static Set<String> names(){
		return names;
	}
	
	/**
	 * Retrieves an {@code AccessType} enum constant by its type name.
	 *
	 * @param typeName The name of the type to find.
	 * @return The corresponding {@code AccessType}, or {@code null} if no match is found.
	 */
	public static AccessType getAccessType(String typeName) {
		AccessType[] accessTypes = AccessType.values();
		for (AccessType accessType : accessTypes) {
			String name = accessType.getName();
			if(name.compareTo(typeName)==0) {
				return accessType;
			}
		}
		return null;
	}
	
	/**
	 * Determines the most specific {@code AccessType} for a given class.
	 * <p>
	 * This method traverses the class hierarchy to find the most derived
	 * matching access type.
	 *
	 * @param clz The class to analyze.
	 * @return The most specific {@code AccessType} for the class.
	 * @throws RuntimeException if the class does not correspond to any defined access type.
	 */
	public static AccessType getAccessType(Class<?> clz) {
		AccessType ret  =null;
		
		AccessType[] accessTypes = AccessType.values();
		for (AccessType accessType : accessTypes) {
			Class<? extends Element> typeClass = accessType.getTypeClass();
			if (typeClass.isAssignableFrom(clz)) {
				if(ret==null || ret.getTypeClass().isAssignableFrom(typeClass)){
					ret = accessType;
				}
			}
		}
		
		if(ret !=null){
			return ret;
		}else{
			String error = String
					.format("The provided class %s does not belong to any of defined AccessTypes %s",
							clz.getSimpleName(), Arrays.toString(accessTypes));
			logger.trace(error);
			throw new RuntimeException(error);
		}
	}
	
	/**
	 * Returns an array of access types that are part of the core data model.
	 * <p>
	 * These types include {@code Property}, {@code Resource}, {@code Facet},
	 * {@code IsRelatedTo}, and {@code ConsistsOf}.
	 *
	 * @return An array of core model {@code AccessType}s.
	 */
	public static AccessType[] getModelTypes() {
		return modelTypes;
	}
	
	/**
	 * Returns an array of access types representing entities and relations in the model.
	 * <p>
	 * These types include {@code Resource}, {@code Facet}, {@code IsRelatedTo},
	 * and {@code ConsistsOf}.
	 *
	 * @return An array of entity/relation {@code AccessType}s.
	 */
	public static AccessType[] getERTypes() {
		return erTypes;
	}
	
}
