package org.gcube.informationsystem.base.impl.properties;

import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.gcube.com.fasterxml.jackson.annotation.JsonIgnore;
import org.gcube.com.fasterxml.jackson.annotation.JsonTypeName;
import org.gcube.informationsystem.base.impl.ElementImpl;
import org.gcube.informationsystem.base.reference.Element;
import org.gcube.informationsystem.base.reference.properties.PropertyElement;
import org.gcube.informationsystem.model.reference.properties.Property;
import org.gcube.informationsystem.serialization.ElementMapper;

/**
 * The default implementation of the {@link PropertyElement} interface.
 *
 * @author Luca Frosini (ISTI - CNR)
 */
@JsonTypeName(value = PropertyElement.NAME)
public class PropertyElementImpl extends ElementImpl implements Property {

	/**
	 * The generated serial version UID for serialization.
	 */
	private static final long serialVersionUID = 1396998430221747445L;

	/** The list of supertypes for this property. */
	protected List<String> supertypes;
	/** The expected type of the property's value. */
	protected String expectedtype;

	/** 
	 * A map to store additional, non-standard properties. 
	 * 
	 */
	@JsonIgnore
	protected Map<String, Object> additionalProperties;

	/**
	 * A set of keys that are allowed as additional properties, even if they start
	 * with '_' or '@'.
	 */
	@JsonIgnore
	protected final Set<String> allowedAdditionalKeys;

	/**
	 * Default constructor. Initializes the additional properties map and the set of
	 * allowed keys.
	 */
	public PropertyElementImpl() {
		super();
		this.additionalProperties = new HashMap<>();
		this.allowedAdditionalKeys = new HashSet<>();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public List<String> getSupertypes() {
		return this.supertypes;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getExpectedtype() {
		return this.expectedtype;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Map<String, Object> getAdditionalProperties() {
		return additionalProperties;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setAdditionalProperties(Map<String, Object> additionalProperties) {
		this.additionalProperties = additionalProperties;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Object getAdditionalProperty(String key) {
		return additionalProperties.get(key);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setAdditionalProperty(String key, Object value) {
		if (!allowedAdditionalKeys.contains(key)) {
			if (key.startsWith("_")) {
				return;
			}
			if (key.startsWith("@")) {
				return;
			}
		}

		/*
		 * Additional properties are not deserialized to the proper property type. The
		 * first attempt was to try to write a specific deserializer but it fails. This
		 * fix the issue.
		 */
		try {
			if (value instanceof Map<?, ?>) {
				@SuppressWarnings("unchecked")
				Map<String, Object> map = (Map<String, Object>) value;
				if (map.containsKey(Element.TYPE_PROPERTY)) {
					String reserialized = ElementMapper.getObjectMapper().writeValueAsString(map);
					Property property = ElementMapper.unmarshal(Property.class, reserialized);
					value = property;
				}
			}
		} catch (Throwable e) {
			e.getMessage();
			// Any type of error/exception must be catched
		}
		/* END of fix to properly deserialize Property types */

		this.additionalProperties.put(key, value);
	}

	/**
	 * Adds a key to the set of allowed additional keys.
	 *
	 * @param allowedAdditionalKey The key to allow.
	 */
	public void addAllowedAdditionalKey(String allowedAdditionalKey) {
		this.allowedAdditionalKeys.add(allowedAdditionalKey);
	}

}