/**
 * 
 */
package org.gcube.informationsystem.utils.discovery;

import java.lang.reflect.Method;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import org.gcube.informationsystem.base.reference.Element;
import org.gcube.informationsystem.base.reference.entities.EntityElement;
import org.gcube.informationsystem.base.reference.properties.PropertyElement;
import org.gcube.informationsystem.base.reference.relations.RelationElement;
import org.gcube.informationsystem.types.annotations.ISProperty;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * @author Luca Frosini (ISTI - CNR)
 */
public class ElementSpecilizationDiscovery<E extends Element> {
	
	private static Logger logger = LoggerFactory.getLogger(ElementSpecilizationDiscovery.class);
	
	protected final Class<E> root;
	protected final List<Package> packages;
	protected final List<Class<? extends E>> discovered;
	
	public List<Class<? extends E>> getDiscovered() {
		return discovered;
	}
	
	public ElementSpecilizationDiscovery(Class<E> root) {
		this.root = root;
		this.packages = new ArrayList<>();
		addPackage(root.getPackage());
		this.discovered = new ArrayList<>();
		add(root);
	}
	
	public void addPackage(Package p) {
		packages.add(p);
	}
	
	protected void add(Class<? extends E> clz) {
		if(!discovered.contains(clz)) {
			discovered.add(clz);
			logger.info("+ Added {}.", clz);
		}
	}
	
	protected void analizeISM(Class<? extends E> clz) {
		logger.trace("Analizyng {}", clz);
		
		if(!clz.isInterface()) {
			logger.trace("- Discarding {} that is not an interface", clz);
			return;
		}
		
		if(!root.isAssignableFrom(clz)) {
			logger.trace("- Discarding {} because is not a {}", clz, root.getSimpleName());
			return;
		}
		
		if(discovered.contains(clz)) {
			logger.trace("- Discarding {} because was already managed", clz);
			return;
		}
		
		Class<?>[] interfaces = clz.getInterfaces();
		
		for(Class<?> interfaceClass : interfaces) {
			@SuppressWarnings("unchecked")
			Class<E> parent = (Class<E>) interfaceClass;
			analizeISM(parent);
		}
		
		if(root == PropertyElement.class) {
			
			for(Method m : clz.getDeclaredMethods()) {
				m.setAccessible(true);
				if(m.isAnnotationPresent(ISProperty.class)) {
					if(Map.class.isAssignableFrom(m.getReturnType()) || Set.class.isAssignableFrom(m.getReturnType())
							|| List.class.isAssignableFrom(m.getReturnType())) {
						
						Type[] typeArguments = ((ParameterizedType) m.getGenericReturnType()).getActualTypeArguments();
						for(Type t : typeArguments) {
							@SuppressWarnings("unchecked")
							Class<? extends PropertyElement> tClass = (Class<? extends PropertyElement>) t;
							if(root.isAssignableFrom(tClass)) {
								@SuppressWarnings("unchecked")
								Class<E> type = (Class<E>) tClass;
								analizeISM(type);
							}
						}
						
					} else if(root.isAssignableFrom(m.getReturnType())) {
						@SuppressWarnings("unchecked")
						Class<E> type = (Class<E>) m.getReturnType();
						analizeISM(type);
					}
				}
				
			}
		}
		add(clz);
	}
	
	public void discover() throws Exception {
		for(Package p : packages) {
			List<Class<?>> classes = ReflectionUtility.getClassesForPackage(p);
			for(Class<?> clz : classes) {
				@SuppressWarnings("unchecked")
				Class<E> ism = (Class<E>) clz;
				analizeISM(ism);
			}
		}
		
	}
	
	public static void manageISM(SchemaAction schemaAction, List<Package> packages) throws Exception {
		if(Objects.nonNull(packages) && packages.size() > 0) {
			manageISM(schemaAction, packages.stream().toArray(Package[]::new));
		} else {
			manageISM(schemaAction);
		}
	}
	
	@SuppressWarnings("unchecked")
	public static void manageISM(SchemaAction schemaAction, Package... packages) throws Exception {
		ElementSpecilizationDiscovery<PropertyElement> propertyDiscovery = new ElementSpecilizationDiscovery<>(PropertyElement.class);
		if(Objects.nonNull(packages)) {
			Arrays.stream(packages).forEach(p -> propertyDiscovery.addPackage(p));
		}
		propertyDiscovery.discover();
		for(Class<? extends PropertyElement> property : propertyDiscovery.getDiscovered()) {
			logger.trace("Going to manage : {}", property);
			schemaAction.managePropertyClass(property);
		}
		
		ElementSpecilizationDiscovery<EntityElement> entityDiscovery = new ElementSpecilizationDiscovery<>(EntityElement.class);
		if(Objects.nonNull(packages)) {
			Arrays.stream(packages).forEach(p -> entityDiscovery.addPackage(p));
		}
		entityDiscovery.discover();
		
		for(Class<? extends EntityElement> entity : entityDiscovery.getDiscovered()) {
			logger.trace("Going to manage : {}", entity);
			schemaAction.manageEntityClass(entity);
		}
		
		@SuppressWarnings("rawtypes")
		ElementSpecilizationDiscovery<RelationElement> relationDiscovery = new ElementSpecilizationDiscovery<RelationElement>(RelationElement.class);
		if(Objects.nonNull(packages))
			Arrays.stream(packages).forEach(p -> relationDiscovery.addPackage(p));
		relationDiscovery.discover();
		
		for(@SuppressWarnings("rawtypes")
		Class<? extends RelationElement> relation : relationDiscovery.getDiscovered()) {
			logger.trace("Going to manage : {}", relation);
			schemaAction.manageRelationClass(relation);
		}
	}
	
}
