package org.gcube.informationsystem.types.impl.properties;

import java.lang.reflect.Method;
import java.net.URI;
import java.net.URL;
import java.util.Objects;
import java.util.UUID;

import org.gcube.com.fasterxml.jackson.annotation.JsonIgnore;
import org.gcube.com.fasterxml.jackson.annotation.JsonSetter;
import org.gcube.com.fasterxml.jackson.annotation.JsonTypeName;
import org.gcube.informationsystem.types.PropertyTypeName;
import org.gcube.informationsystem.types.annotations.ISProperty;
import org.gcube.informationsystem.types.reference.properties.PropertyDefinition;
import org.gcube.informationsystem.utils.Version;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * @author Luca Frosini (ISTI - CNR)
 */
// @JsonAutoDetect(fieldVisibility=JsonAutoDetect.Visibility.ANY)
@JsonTypeName(value=PropertyDefinition.NAME)
public final class PropertyDefinitionImpl implements PropertyDefinition {

	/**
	 * Generated Serial Version UID
	 */
	private static final long serialVersionUID = -5925314595659292025L;

	private static Logger logger = LoggerFactory.getLogger(PropertyDefinitionImpl.class);
	
	public final static String UUID_REGEX = "^([a-fA-F0-9]{8}-[a-fA-F0-9]{4}-[a-fA-F0-9]{4}-[a-fA-F0-9]{4}-[a-fA-F0-9]{12}){1}$";
	public final static String URI_REGEX = null;
	public final static String URL_REGEX = null;
	
	private String name= "";
	private String description= "";
	private boolean mandatory = false;
	private boolean readonly = false;
	private boolean notnull = false;
	private Integer max= null;
	private Integer min= null;
	private String regexp= null;

	private PropertyTypeName propertyTypeName = null; 

	private static String getPropertyNameFromMethodName(Method method){
		String name = method.getName();
		if(name.startsWith("get")){
			name = name.replace("get", "");

		}
		if(name.startsWith("is")){
			name = name.replace("is", "");
		}
		
		if(name.length() > 0){
			name = Character.toLowerCase(name.charAt(0)) + (name.length() > 1 ? name.substring(1) : "");
		}
		
		return name;
	}
	
	protected PropertyDefinitionImpl() {
		
	}
	
	public PropertyDefinitionImpl(ISProperty propertyAnnotation, Method method) {
		String name = propertyAnnotation.name().isEmpty()?getPropertyNameFromMethodName(method):propertyAnnotation.name();
		this.name = name;
		this.description = propertyAnnotation.description();
		this.mandatory= propertyAnnotation.mandatory();
		this.notnull = !propertyAnnotation.nullable();
		this.readonly = propertyAnnotation.readonly();
		if(propertyAnnotation.max()>0) { 
			this.max = propertyAnnotation.max();
		}
		if(propertyAnnotation.max()>=propertyAnnotation.min() && propertyAnnotation.min()>0) {
			this.min = propertyAnnotation.min();
		}
		
		this.propertyTypeName = new PropertyTypeName(method);
		
		Class<?> clz = method.getReturnType();
		logger.trace("Return Type for method {} is {}", method, clz);
		
		if(!propertyAnnotation.regexpr().isEmpty()) this.regexp = propertyAnnotation.regexpr();
		if(this.regexp==null || this.regexp.compareTo("")==0 ){
			if(Enum.class.isAssignableFrom(clz)){
				Object[] constants = clz.getEnumConstants();
				StringBuilder stringBuilder = new StringBuilder("^(");
				for(int i=0; i<constants.length; i++){
					stringBuilder.append(constants[i].toString());
					if(i<constants.length-1){
						stringBuilder.append("|");
					}
				}
				stringBuilder.append(")$");
				this.regexp = stringBuilder.toString();
			}
			if(UUID.class.isAssignableFrom(clz)){
				this.regexp = UUID_REGEX;
			}
			if(URI.class.isAssignableFrom(clz)){
				this.regexp = URI_REGEX;
			}
			if(URL.class.isAssignableFrom(clz)){
				this.regexp = URL_REGEX;
			}
			if(Version.class.isAssignableFrom(clz)){
				this.regexp = Version.VERSION_REGEX;
			}
			
		}

	}
	
	@Override
	public String getName() {
		return name;
	}
	
	@Override
	public String getDescription() {
		return description;
	}
	
	@Override
	public boolean isMandatory() {
		return mandatory;
	}

	@Override
	public boolean isReadonly() {
		return readonly;
	}
	
	@Override
	public boolean isNotnull() {
		return notnull;
	}
	
	@Override
	public Integer getMax() {
		return max;
	}
	
	@Override
	public Integer getMin() {
		return min;
	}
	
	@Override
	public String getRegexp() {
		return regexp;
	}
	
	@Override
	public String getType() {
		return propertyTypeName.toString();
	}

	@JsonSetter(value = PropertyDefinition.TYPE_PROPERTY)
	public void setType(String type) {
		this.propertyTypeName = new PropertyTypeName(type);
	}
	
	@JsonIgnore
	public PropertyTypeName getPropertyTypeName() {
		return propertyTypeName;
	}
	
	@Override
	public String toString() {
		return "Property [name=" + name + ", description=" + description
				+ ", mandatory=" + mandatory + ", readonly=" + readonly
				+ ", notnull=" + notnull + ", max=" + max + ", min="
				+ min + ", regexpr=" + regexp + ", type=" + propertyTypeName.toString() + "]";
	}

	@Override
	public int hashCode() {
		return Objects.hash(description, mandatory, max, min, name, notnull, propertyTypeName.toString(), readonly, regexp);
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		PropertyDefinitionImpl other = (PropertyDefinitionImpl) obj;
		return Objects.equals(description, other.description) && mandatory == other.mandatory
				&& Objects.equals(max, other.max) && Objects.equals(min, other.min) && Objects.equals(name, other.name)
				&& notnull == other.notnull && Objects.equals(propertyTypeName.toString(), other.propertyTypeName.toString())
				&& readonly == other.readonly && Objects.equals(regexp, other.regexp);
	}

	

}
