package org.gcube.informationsystem.types.impl;

import java.lang.reflect.Method;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.TypeVariable;
import java.util.HashSet;
import java.util.Set;

import org.gcube.informationsystem.base.reference.Element;
import org.gcube.informationsystem.base.reference.entities.EntityElement;
import org.gcube.informationsystem.base.reference.properties.PropertyElement;
import org.gcube.informationsystem.base.reference.relations.RelationElement;
import org.gcube.informationsystem.model.reference.properties.Header;
import org.gcube.informationsystem.types.TypeMapper;
import org.gcube.informationsystem.types.annotations.Abstract;
import org.gcube.informationsystem.types.annotations.ISProperty;
import org.gcube.informationsystem.types.impl.entities.EntityTypeImpl;
import org.gcube.informationsystem.types.impl.properties.PropertyDefinitionImpl;
import org.gcube.informationsystem.types.impl.properties.PropertyTypeImpl;
import org.gcube.informationsystem.types.impl.relations.RelationTypeImpl;
import org.gcube.informationsystem.types.reference.Type;
import org.gcube.informationsystem.types.reference.properties.PropertyDefinition;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonInclude.Include;
import com.fasterxml.jackson.annotation.JsonProperty;

public class TypeImpl implements Type {
	
	private static Logger logger = LoggerFactory.getLogger(TypeImpl.class);
	
	/**
	 * Generated Serial version UID
	 */
	private static final long serialVersionUID = -4333954207969059451L;
	
	public final static String DESCRIPTION = "DESCRIPTION";
	
	protected Header header;
	
	protected String name;
	protected String description;
	@JsonProperty(value="abstract")
	protected boolean abstractType;
	protected Set<String> superClasses;
	
	protected Set<PropertyDefinition> properties;
	
	protected <E extends Element> Set<String> retrieveSuperClasses(Class<? extends E> type, Class<E> baseClass, String topSuperClass){
		Set<String> interfaceList = new HashSet<>();
		
		if(type==baseClass){
			if(topSuperClass!=null) {
				interfaceList.add(topSuperClass);
			}
			return interfaceList;
		}
		
		Class<?>[] interfaces = type.getInterfaces();
		
		for (Class<?> interfaceClass : interfaces) {
			
			if(!baseClass.isAssignableFrom(interfaceClass)){
				continue;
			}
			
			@SuppressWarnings("unchecked")
			Class<? extends Element> clz  = (Class<? extends Element>) interfaceClass;
			interfaceList.add(TypeMapper.getType(clz));
		}

		return interfaceList;
	}
	
	protected Set<PropertyDefinition> retrieveListOfProperties(Class<?> type){
		Set<PropertyDefinition> properties = new HashSet<>();
		for (Method m : type.getDeclaredMethods()){
			m.setAccessible(true);
			if(m.isAnnotationPresent(ISProperty.class)){
				if(m.isBridge()) {
					continue;
				}
				ISProperty propAnnotation = m.getAnnotation(ISProperty.class);
				PropertyDefinition prop = new PropertyDefinitionImpl(propAnnotation, m);
				properties.add(prop);
				logger.trace("Property {} retrieved in type {} ", prop, type.getSimpleName());
			}  

		}
		return properties;
	}
	
	protected Class<?> getGenericClass(java.lang.reflect.Type type){
		TypeVariable<?> typeVariable = (TypeVariable<?>) type;
		java.lang.reflect.Type[] bounds = typeVariable.getBounds();
		java.lang.reflect.Type t = bounds[0];
		if(t instanceof ParameterizedType) {
			ParameterizedType parameterizedType = (ParameterizedType) t;
			return (Class<?>) parameterizedType.getRawType();
		}
		return (Class<?>) t; 
	}
	
	@SuppressWarnings({"rawtypes", "unchecked"})
	public static Type getInstance(Class<? extends Element> clz) {
		Type typeDefinition = null;
		try {
			if(EntityElement.class.isAssignableFrom(clz)) {
				typeDefinition = EntityTypeImpl.getEntityTypeDefinitionInstance((Class<? extends EntityElement>) clz);
				return typeDefinition;
			} else if(RelationElement.class.isAssignableFrom(clz)){
				typeDefinition = RelationTypeImpl.getRelationTypeDefinitionInstance((Class<? extends RelationElement<?,?>>) clz);
				return typeDefinition;
			} else if(PropertyElement.class.isAssignableFrom(clz)){
				typeDefinition = new PropertyTypeImpl(clz);
				return typeDefinition;
			} else if(Type.class.isAssignableFrom(clz)) {
				typeDefinition = new TypeImpl(clz);
				return typeDefinition;
			} else {
				throw new RuntimeException("Serialization required");
			}
		} finally {
			if(typeDefinition!=null) {
				logger.debug("{} : {} ", clz, typeDefinition);
			}
		}
	}
	
	protected TypeImpl() {}
	
	protected TypeImpl(Class<? extends Element> clz) {
		this.name = TypeMapper.getType(clz);
		this.description = TypeMapper.getStaticStringFieldByName(clz, DESCRIPTION, "");
		this.abstractType = false;
		
		if(clz.isAnnotationPresent(Abstract.class)){
			this.abstractType = true;
		}
		
	}
	
	@Override
	public Header getHeader() {
		return header;
	}
	
	@Override
	public void setHeader(Header header){
		this.header = header;
	}
	
	
	@Override
	public String getName() {
		return name;
	}

	@Override
	public String getDescription() {
		return description;
	}

	@Override
	public boolean isAbstract() {
		return abstractType;
	}

	@Override
	public Set<String> getSuperClasses() {
		return superClasses;
	}
	
	@JsonInclude(Include.NON_EMPTY)
	public Set<PropertyDefinition> getProperties() {
		return properties;
	}
}
