package org.gcube.indexmanagement.storagehandling;

import java.rmi.RemoteException;
import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;

import javax.xml.namespace.QName;

import org.apache.axis.message.addressing.EndpointReferenceType;
import org.gcube.common.core.contexts.GCUBEServiceContext;
import org.gcube.common.core.contexts.GHNContext;
import org.gcube.common.core.informationsystem.client.ISClient;
import org.gcube.common.core.informationsystem.client.RPDocument;
import org.gcube.common.core.informationsystem.client.queries.WSResourceQuery;
import org.gcube.common.core.scope.GCUBEScope;
import org.gcube.common.core.security.GCUBESecurityManager;
import org.gcube.common.core.utils.logging.GCUBELog;
import org.gcube.indexmanagement.common.IndexException;
import org.gcube.indexmanagement.common.IndexNotificationConsumer;
import org.gcube.indexmanagement.common.IndexWSResource;
import org.gcube.indexmanagement.common.notifications.NotifierRequestQueue;
import org.gcube.indexmanagement.common.notifications.SubscribeToNotificationRequest;
import org.gcube.indexmanagement.common.notifications.UnsubscribeFromNotificationRequest;
import org.gcube.indexmanagement.storagehandling.exceptions.NoIndexManagerFoundException;
import org.gcube.indexmanagement.storagehandling.stubs.ConnectLookup;
import org.gcube.indexmanagement.storagehandling.stubs.ConnectUpdater;
import org.gcube.indexmanagement.storagehandling.stubs.ConnectUpdaterResponse;
import org.gcube.indexmanagement.storagehandling.stubs.DeltaFileInfoType;
import org.gcube.indexmanagement.storagehandling.stubs.DeltaListManagementProviderPortType;
import org.gcube.indexmanagement.storagehandling.stubs.GetDeltaCollectionID;
import org.gcube.indexmanagement.storagehandling.stubs.GetDeltaFileList;
import org.gcube.indexmanagement.storagehandling.stubs.service.DeltaListManagementProviderServiceAddressingLocator;
import org.globus.wsrf.core.notification.SubscriptionManager;
import org.oasis.wsn.PauseFailedFaultType;
import org.oasis.wsn.PauseSubscription;
import org.oasis.wsn.PauseSubscriptionResponse;
import org.oasis.wsn.ResumeFailedFaultType;
import org.oasis.wsn.ResumeSubscription;
import org.oasis.wsn.ResumeSubscriptionResponse;
import org.oasis.wsrf.lifetime.Destroy;
import org.oasis.wsrf.lifetime.DestroyResponse;
import org.oasis.wsrf.lifetime.ResourceNotDestroyedFaultType;
import org.oasis.wsrf.lifetime.ResourceUnknownFaultType;
import org.oasis.wsrf.lifetime.SetTerminationTime;
import org.oasis.wsrf.lifetime.SetTerminationTimeResponse;
import org.oasis.wsrf.lifetime.TerminationTimeChangeRejectedFaultType;
import org.oasis.wsrf.lifetime.UnableToSetTerminationTimeFaultType;
import org.oasis.wsrf.properties.GetResourcePropertyResponse;
import org.oasis.wsrf.properties.InvalidResourcePropertyQNameFaultType;


public class RemoteDeltaListManager implements DeltaListManagementWrapper {
	
	private static final int MAX_ATTEMPTS = 5;

	private static final long WAIT_ATTEMPT = 10000;

	/** Logger */
    static GCUBELog logger = new GCUBELog(RemoteDeltaListManager.class);
    
    /** The index manager interface */
    private DeltaListManagementProviderPortType indexManager;
    
    /** The EPR of the index manager */
    private EndpointReferenceType managerEPR;
    
    /** The index ID */
    private String indexManagementID;
    
    /** The service context to be used for security and scoping */
    private GCUBEServiceContext sctx;
    
    /** The index management service Namespace */
    private String managementServiceNamespace = null;
    
    /**
     * Constructs a new RemoteDeltaListManager
     * @param indexManagementID the ID of the index whose delta list is to be managed
     * @param sctx the service context to be used for security and scoping
     * @throws IndexException an error occurred
     */
    public RemoteDeltaListManager(String indexManagementID, GCUBEServiceContext sctx, String managementServiceNamespace) throws Exception {
    	logger.info("Creating a RemoteDeltaListManager with no explicit manager EPR declaration.");
    	try {
	        this.sctx = sctx;
	        this.managementServiceNamespace = managementServiceNamespace;
	        this.indexManagementID = indexManagementID;	        
        	EndpointReferenceType managerEPR = findManagementEPR(indexManagementID, managementServiceNamespace);
        	logger.debug("Found index management EPR:\n" + managerEPR);
        	setIndexManagerEPR(managerEPR);
    	} catch (NoIndexManagerFoundException e) {
    		this.indexManager = null;
    		this.managerEPR = null;
    		logger.info("No manager found for the indexID: " + indexManagementID);
    	} catch(Exception e1) {
        	logger.error("Failed to construct RemoteDeltaListManager.", e1);
            throw new Exception(e1.getMessage());
        }
    }
    
    /**
     * Constructs a new RemoteDeltaListManager
     * @param indexManagementID the ID of the index whose delta list is to be managed
     * @param indexManagerEPR the EPR of the index manager
     * @param sctx the service context to be used for security and scoping
     * @throws Exception an error occurred
     */
    public RemoteDeltaListManager(String indexManagementID, EndpointReferenceType indexManagerEPR, GCUBEServiceContext sctx) throws Exception {
    	logger.info("Creating a RemoteDeltaListManager using the given manager EPR.");
    	try {
	    	this.sctx = sctx;
	    	this.indexManagementID = indexManagementID;
	    	setIndexManagerEPR(indexManagerEPR);
    	} catch(Exception e1) {
        	logger.error("Failed to construct RemoteDeltaListManager.", e1);
            throw new Exception(e1.getMessage());
        }
    }
    
    /**
     * Returns a ID of the collection with the delta related documents
     * @return delta collection ID
     * @throws Exception when contacting the {@link DeltaListManagementProviderPortType}
     */
    public String getDeltaCollectionID() throws Exception{
    	
	    try{
	    	//if the indexManager is not initialized try again to find a Manager EPR
	    	if(indexManager == null)
	    	{
	    		//if we don't have the namespace just return
	    		if(this.managementServiceNamespace == null)
	    		{
	    			logger.error("there is not managementServiceNamespace ");
	    			return null;
	    		}
	    		EndpointReferenceType managerEPR = findManagementEPR(indexManagementID, this.managementServiceNamespace);
	        	logger.debug("Found index management EPR:\n" + managerEPR);
	        	setIndexManagerEPR(managerEPR);    	
	    	}
	    } catch (Exception e) {
	    	logger.error("Could not retrieve a portType to the Manager, for Index ID : " + indexManagementID, e);
	    	return null;
	    }
    	
    	String colID = indexManager.getDeltaCollectionID(new GetDeltaCollectionID());
    	logger.trace("retrieved delta collection id: " + colID);
    	return colID;    	
    }
    
    /**
     * Finds the EPR of the index management resource.
     * @param indexID the indexID to find the management resource for
     * @param managementServiceNamespace the namespace of the management service
     * @return EndpointReferenceType the found management resource EPR
     * @throws IndexException an error occured
     */
    private EndpointReferenceType findManagementEPR(String indexID, String managementServiceNamespace) throws Exception {
    	List<EndpointReferenceType> result = null;
    	EndpointReferenceType epr = null;
        try{
        	int attempts = 0;
        	while((result == null || result.size() == 0) && attempts<MAX_ATTEMPTS)
        	{
	        	List<String[]> props = new LinkedList<String[]>();
	        	props.add(new String[] { IndexWSResource.RP_INDEX_ID, indexID });
	        	result = getWSResourceEPRsFromPropValuesAndNamespace(props, managementServiceNamespace, sctx.getScope());
	        	
	        	//if no management EPR is found try again
	        	if(result == null || result.size() == 0){
	        		Thread.sleep(WAIT_ATTEMPT);
	        	}
	        	attempts++;
	        }
        	logger.trace("number of attempts : " + attempts + " for indexID: " + indexID);
        }
        catch(Exception e){
        	logger.error("Failed to retrieve the management resource EPR: " + e.getMessage());
            throw new Exception("Failed to retrieve the management resource EPR: " + e.getMessage());
        }
        
        if(result == null || result.size() == 0){
        	logger.error("Unable to find management epr in IS");
            throw new NoIndexManagerFoundException();
        }

        epr = result.get(0);
        logger.info("Got the EPR for the index [" + indexID + "]");
        return epr;
    }
    
    /**
     * Searches for a resource given a namespace and a set of resource property values.
     * @param properties the set of RP values
     * @param namespace the namespace that the RPs belong to
     * @param scope the scope to use for searching
     * @return the EPR of the found resource
     * @throws Exception an error occured
     */
    private List<EndpointReferenceType> getWSResourceEPRsFromPropValuesAndNamespace(List<String[]> properties, String namespace, GCUBEScope scope) throws Exception {
		String filter = "$result/child::*[local-name()='"+properties.get(0)[0].trim()+"' and namespace-uri(.)='"+namespace.trim()+"']/string() eq '"+properties.get(0)[1].trim()+"'";
		for (int i=1; i<properties.size(); i++)
			filter += " and $result/child::*[local-name()='"+properties.get(i)[0].trim()+"' and namespace-uri(.)='"+namespace.trim()+"']/string() eq '"+properties.get(i)[1].trim()+"'";
		
		ISClient client = GHNContext.getImplementation(ISClient.class);
		WSResourceQuery gquery = client.getQuery(WSResourceQuery.class);
		gquery.addGenericCondition(filter);
		List<EndpointReferenceType> ret = new LinkedList<EndpointReferenceType>();
		for(RPDocument d : client.execute(gquery, scope)){
			ret.add(d.getEndpoint());
		}
		return ret;
	}
    
    /**
     * Sets the EPR of the remote delta list manager
     * @param managerEPR the EPR of the remote delta list manager
     */
    public void setIndexManagerEPR(EndpointReferenceType managerEPR) throws Exception {
    	try {
    		this.managerEPR = managerEPR;
	    	DeltaListManagementProviderServiceAddressingLocator managerLocator = new DeltaListManagementProviderServiceAddressingLocator();
	        this.indexManager = managerLocator.getDeltaListManagementProviderPortTypePort(managerEPR);
	        
	        ServiceContext.getContext().prepareCall(indexManager, ServiceContext.getContext().getServiceClass(), ServiceContext.getContext().getName(), sctx.getScope());
	        ServiceContext.getContext().setSecurity(indexManager, GCUBESecurityManager.AuthMode.BOTH, GCUBESecurityManager.DelegationMode.FULL);
    	} catch (Exception e) {
    		this.indexManager = null;
    		this.managerEPR = null;
    		logger.error("Failed to get the remote delta list manager's porttype. Setting RemoteDeltaListManager's state to 'empty'.", e);
    	}
    }
    
    /* (non-Javadoc)
	 * @see org.gcube.indexmanagement.storagehandling.DeltaListManagementWrapper#getIndexManagerEPR()
	 */
	public EndpointReferenceType getIndexManagerEPR() {
		return this.managerEPR;
	}
	
	/* (non-Javadoc)
	 * @see org.gcube.indexmanagement.storagehandling.DeltaListManagementWrapper#isActive()
	 */
	public boolean isEmpty() {
		return (indexManager == null);
	}

    /**
     * Connects an updater to the index management resource.
     * @return ConnectUpdaterResponse the response
     * @throws Exception an error occured
     */
    public ConnectUpdaterResponse connectUpdater() throws Exception {
    	if (isEmpty())
    		throw new Exception("The remote delta list manager is not currently connected to a delta list manager.");
        return indexManager.connectUpdater(new ConnectUpdater());
    }

    /**
     * Connects a lookup to the index management resource.
     * @return int the new connection ID
     * @throws Exception an error occurred
     */
    public int connectLookup() throws Exception {
    	if (isEmpty())
    		throw new Exception("The remote delta list manager is not currently connected to a delta list manager.");
        return indexManager.connectLookup(new ConnectLookup());
    }

    /**
     * Disconnects an updater from the index management resource.
     * @param updaterID the ID of the updater to disconnect
     * @throws Exception an error occurred
     */
    public void disconnectUpdater(int updaterID) throws Exception {
    	if (isEmpty())
    		throw new Exception("The remote delta list manager is not currently connected to a delta list manager.");
        indexManager.disconnectUpdater(updaterID);
    }

    /**
     * Returns info about a given delta file.
     * @param idx the index of delta file in question
     * @return DeltaFileInfoType the delta file info
     * @throws Exception an error occurred
     */
    public DeltaFileInfoType getDeltaFileInfo(int idx) throws Exception {
    	if (isEmpty())
    		throw new Exception("The remote delta list manager is not currently connected to a delta list manager.");
    	
    	String str = new String();
    	for (DeltaFileInfoType d : getDeltaFileList()) {
    		str += d.getDeltaFileID();
    	}
    	
    	logger.info("DeltasInfoTypes : " + str);
    	
        return indexManager.getDeltaFileInfo(idx);
    }

    /**
     * Returns the delta file list.
     * @return DeltaFileInfoType[] info about the delta files that make up the index
     * @throws Exception an error occurred
     */
    public DeltaFileInfoType[] getDeltaFileList() throws Exception {
    	if (isEmpty())
    		throw new Exception("The remote delta list manager is not currently connected to a delta list manager.");

    	DeltaFileInfoType[] list = indexManager.getDeltaFileList(new GetDeltaFileList()).getDeltaFiles();
        if(list == null){
            return new DeltaFileInfoType[0];
        }
        return list;
    }

    /**
     * Merges a delta file with the main index.
     * @param deltaInfo info about the delta file to merge
     * @throws Exception an error occurred
     */
    public void mergeDeltaFile(DeltaFileInfoType deltaInfo) throws Exception {
    	if (isEmpty())
    		throw new Exception("The remote delta list manager is not currently connected to a delta list manager.");

    	indexManager.mergeDeltaFile(deltaInfo);
    }

    /**
     * Subsribes this RemoteDeltaListManager for notifications concerning additions of data to the index.
     * @return SubscriptionManager the created subscription manager
     * @throws Exception an error occured
     */
    public SubscriptionManager subscribeForAdditions(IndexNotificationConsumer consumer) throws Exception {
        QName topic = new QName("http://gcube-system.org/namespaces/indexmanagement/DeltaListManagementProvider/" + indexManagementID, "AddDelta");
        logger.debug("Subscribing for topic: " + topic);
        return new RemoteSubscriptionManager(topic, consumer);
    }
    
    /**
     * Subsribes this RemoteDeltaListManager for notifications concerning deletions of data to the index.
     * @return SubscriptionManager the created subscription manager
     * @throws Exception an error occured
     */
    public SubscriptionManager subscribeForDeletions(IndexNotificationConsumer consumer) throws Exception {
        QName topic = new QName("http://gcube-system.org/namespaces/indexmanagement/DeltaListManagementProvider/" + indexManagementID, "DeleteDelta");
        logger.debug("Subscribing for topic: " + topic);
        return new RemoteSubscriptionManager(topic, consumer);
    }
    
    public SubscriptionManager subscribeForIndexRemoval(IndexNotificationConsumer consumer) throws Exception {
        QName topic = new QName("http://gcube-system.org/namespaces/indexmanagement/DeltaListManagementProvider/" + indexManagementID, "IndexRemoved");
        logger.debug("Subscribing for topic: " + topic);
        return new RemoteSubscriptionManager(topic, consumer);
    }

    public SubscriptionManager subscribeForManagerCreation(IndexNotificationConsumer consumer) throws Exception {
    	QName topic = new QName("http://gcube-system.org/namespaces/indexmanagement/DeltaListManagementProvider/" + indexManagementID, "IndexManagerCreated");
        logger.debug("Subscribing for topic: " + topic);
        return new RemoteSubscriptionManager(topic, consumer);
    }
    
    /**
     * Utility class that handles subscriptions for brokered notifications concerning various 
     * remote index-related events through the IS.
     */
    private class RemoteSubscriptionManager implements SubscriptionManager {
        
        /**
         * The notification consumer
         */
        private IndexNotificationConsumer consumer;
        
        /**
         * The topics that the consumer is subscribed to
         */
        private ArrayList<QName> topicList;
        
        /**
         * Constructs a new RemoteSubscriptionManager
         * @param topic the topic to subscribe to
         * @throws IndexException an error occured
         */
        public RemoteSubscriptionManager(QName topic, IndexNotificationConsumer consumer) throws IndexException{
            try{
                this.topicList = new ArrayList<QName>();
                if (topic != null)
                	topicList.add(topic);
                
                //ISNotifier notifier = GHNContext.getImplementation(ISNotifier.class);                
                //notifier.registerToISNotification(topicList, consumer, sctx, sctx.getScope());
                NotifierRequestQueue.getInstance().add(
        				new SubscribeToNotificationRequest(
        						topicList, consumer, sctx, sctx.getScope()
        				)
        		);
            }
            catch(Exception e){
                throw new IndexException(e.getMessage());
            }
        }
        
        public DestroyResponse destroy(Destroy arg0) throws RemoteException, ResourceNotDestroyedFaultType, ResourceUnknownFaultType {
            try{
            	//ISNotifier notifier = GHNContext.getImplementation(ISNotifier.class);
                //notifier.unregisterFromISNotification(topicList, sctx, sctx.getScope());
            	NotifierRequestQueue.getInstance().add(
    					new UnsubscribeFromNotificationRequest(
    							topicList, sctx,
    							sctx.getScope()
    					)
    			);
                return new DestroyResponse();
            }
            catch(Exception e){
                throw new RemoteException(e.getMessage());
            }
        }

        public GetResourcePropertyResponse getResourceProperty(QName arg0) throws RemoteException, InvalidResourcePropertyQNameFaultType, org.oasis.wsrf.properties.ResourceUnknownFaultType {
            return new GetResourcePropertyResponse();
        }

        public PauseSubscriptionResponse pauseSubscription(PauseSubscription arg0) throws RemoteException, PauseFailedFaultType, org.oasis.wsn.ResourceUnknownFaultType {
            try{
                if(topicList.size() != 0){
                	//ISNotifier notifier = GHNContext.getImplementation(ISNotifier.class);
                    //notifier.unregisterFromISNotification(topicList, sctx, sctx.getScope());
                	NotifierRequestQueue.getInstance().add(
        					new UnsubscribeFromNotificationRequest(
        							topicList, sctx,
        							sctx.getScope()
        					)
        			);
                }
                return new PauseSubscriptionResponse();
            }
            catch(Exception e){
                throw new RemoteException(e.getMessage());
            }
        }

        public ResumeSubscriptionResponse resumeSubscription(ResumeSubscription arg0) throws RemoteException, ResumeFailedFaultType, org.oasis.wsn.ResourceUnknownFaultType {
            try{
            	//ISNotifier notifier = GHNContext.getImplementation(ISNotifier.class);
                //notifier.registerToISNotification(topicList, consumer, sctx, sctx.getScope());
            	NotifierRequestQueue.getInstance().add(
        				new SubscribeToNotificationRequest(
        						topicList, consumer, sctx, sctx.getScope()
        				)
        		);
                return new ResumeSubscriptionResponse();
            }
            catch(Exception e){
                throw new RemoteException(e.getMessage());
            }
        }

        public SetTerminationTimeResponse setTerminationTime(SetTerminationTime arg0) throws RemoteException, UnableToSetTerminationTimeFaultType, ResourceUnknownFaultType, TerminationTimeChangeRejectedFaultType {
            return new SetTerminationTimeResponse();
        }
    }
}
