package org.gcube.indexmanagement.geo;

import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.channels.ReadableByteChannel;
import java.nio.charset.Charset;
import java.nio.charset.CharsetDecoder;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;

import org.gcube.common.core.utils.logging.GCUBELog;
import org.gcube.indexmanagement.common.IndexException;
import org.gcube.indexmanagement.common.IndexType;
import org.gcube.indexmanagement.geo.GeoIndexField.DataType;
import org.gcube.indexmanagement.geo.shape.Rectangle;
import org.geotools.index.Data;
import org.geotools.index.DataDefinition;

import com.vividsolutions.jts.geom.Envelope;

/**
 * A utility class used to read and administer the data, coordinates and id of a
 * GeoIndex entry. *
 */
public class DataWrapper {

	public static final String OFFSET_SEPARATOR = ":";

	public static final String DEFAULT_CHARSET = "UTF-8";

	/** logger */
	static GCUBELog logger = new GCUBELog(DataWrapper.class);
	
	/** A map containing the actual field names and values */
	private HashMap<String, String> fieldsMap = new HashMap<String, String>();

    /** The data part of the GeoTools R-Tree entry */
    private Data data;

    /** The data part of the GeoTools R-Tree entry */
    private String colID;

    /** The data part of the GeoTools R-Tree entry */
    private String colLang;

    
    /** The lowest X coordinate of the entry */
    private long x1;

    /** The highest X coordinate of the entry */
    private long x2;

    /** The lowest Y coordinate of the entry */
    private long y1;

    /** The highest Y coordinate of the entry */
    private long y2;

    /** The ID of the entry */
    private String id;

    /** The GeoIndexType of the entry */
    private GeoIndexType indexTypeObject;

    /**
     * A private constructor used to build a DataWrapper instance from a
     * GeoTools Data object
     * 
     * @param data -
     *            a GeoTools Data object containing the coordinates and id as
     *            the first five fields
     */
    private DataWrapper(GeoIndexType indexTypeObject, Data data) {
        this.data = data;
        this.x1 = (Long) data.getValue(0);
        this.x2 = (Long) data.getValue(1);
        this.y1 = (Long) data.getValue(2);
        this.y2 = (Long) data.getValue(3);
        this.id = (String) data.getValue(4);
        this.indexTypeObject = indexTypeObject;
    }
    
    /**
     * A private constructor used to build a DataWrapper instance from a
     * GeoTools Data object
     * 
     * @param data -
     *            a GeoTools Data object containing the coordinates and id as
     *            the first five fields
     */
    private DataWrapper(GeoIndexType indexTypeObject, Data data, RandomAccessFile rawDataFile) throws Exception{
        this.data = data;
        this.x1 = (Long) data.getValue(0);
        this.x2 = (Long) data.getValue(1);
        this.y1 = (Long) data.getValue(2);
        this.y2 = (Long) data.getValue(3);
        this.id = (String) data.getValue(4);
        String allFields = getFromFile((String)data.getValue(5), rawDataFile);
        this.fieldsMap = parseFieldString(allFields);
        this.indexTypeObject = indexTypeObject;
    }
    
    private DataWrapper(String colIdString, String colLangString,
			GeoIndexType indexTypeObject, Data geoData, HashMap<String, String> fieldsMap) {
    	this(indexTypeObject, geoData);
		this.colID = colIdString;
		this.colLang = colLangString;
		this.fieldsMap = fieldsMap;
	}

	/**
     * A getter method for the lowest X coordinate
     * 
     * @return the lowest X coordinate of the entry
     */
    public long getMinX() {
        return x1;
    }

    /**
     * A getter method for the highest X coordinate
     * 
     * @return the highest X coordinate of the entry
     */
    public long getMaxX() {
        return x2;
    }

    /**
     * A getter method for the lowest Y coordinate
     * 
     * @return the lowest Y coordinate of the entry
     */
    public long getMinY() {
        return y1;
    }

    /**
     * A getter method for the highest Y coordinate
     * 
     * @return the highest Y coordinate of the entry
     */
    public long getMaxY() {
        return y2;
    }

    /**
     * A getter method for the ID
     * 
     * @return the ID of the entry
     */
    public String getID() {
        return id;
    }

    /** 
     * A getter method for the Data 
     * @return the data type of data
     */
    public Data getData() {
        return data;
    }

    /**
     * A method to get the Minimum Bounding Rectangle of the entry
     * 
     * @return the Minimum Bounding Rectangle of the entry
     */
    public Envelope getMbr() {
        return new Envelope(x1, x2, y1, y2);
    }

    /**
     * A method to get the value of the IndexType field with the specified field
     * name
     * 
     * @param fieldName -
     *            the fields index/placement in the IndexType
     * @return the value of the field. NULL if the field isn't found.
     */
    public Object getValue(String fieldName) {
        return fieldsMap.get(fieldName);
    }

    /**
     * A method used to get a DataWrapper instance from a GeoTools Data object
     * 
     * @param indexTypeObject -
     *            a GeoTools Data object containing the coordinates and id as
     *            the first five fields
     * @param data - The data for  the GeoTools data object.
     * @return a DataWrapper representation of the Data object
     */
    public static DataWrapper getInstance(GeoIndexType indexTypeObject,
            Data data, RandomAccessFile rawDataFile) throws Exception{
        return new DataWrapper(indexTypeObject, data, rawDataFile);
    }

    /**
     * A method used to create a DataWrapper instance from an IndexType and a
     * DeltaFile input channel
     * 
     * @param indexTypeObject -
     *            The indexType of the entry
     * @param inChannel -
     *            An input channel reading from a DeltaFile
     * @return a DataWrapper representation of the next obect read from the
     *         DeltaFile. Null upon EOF.
     * @throws IndexException
     *             an error reading from the DeltaFile
     */
    public static DataWrapper getInstance(GeoIndexType indexTypeObject, ReadableByteChannel inChannel, RandomAccessFile rawDataFile) throws IndexException {
        try {
        	//first read the collection ID
        	ByteBuffer countBuffer = ByteBuffer.allocate(4);
            if (inChannel.read(countBuffer) == -1) {
                return null;
            }

            countBuffer.flip();
            int idSize = countBuffer.getInt();

            // get col id string
            ByteBuffer buffer = ByteBuffer.allocate(idSize);
            inChannel.read(buffer);
            buffer.flip();
            
            byte[] charArray = new byte[idSize];
            buffer.get(charArray);
            ByteBuffer chars = ByteBuffer.wrap(charArray);

            Charset western = Charset.forName("ISO-8859-1");
            CharsetDecoder isoDecoder = western.newDecoder();
            String colIdString = null;
            colIdString = new String(isoDecoder.decode(chars).array());

            logger.trace("read ColID: " + colIdString);
            
            //then read the collection language
        	countBuffer = ByteBuffer.allocate(4);
            if (inChannel.read(countBuffer) == -1) {
                return null;
            }

            countBuffer.flip();
            idSize = countBuffer.getInt();

            // get collection language string
            buffer = ByteBuffer.allocate(idSize);
            inChannel.read(buffer);
            buffer.flip();
            
            charArray = new byte[idSize];
            buffer.get(charArray);
            chars = ByteBuffer.wrap(charArray);

            String colLangString = null;
            colLangString = new String(isoDecoder.decode(chars).array());
            
            logger.trace("read ColLang: " + colLangString);
            
        	//then read the document ID
        	countBuffer = ByteBuffer.allocate(4);
            if (inChannel.read(countBuffer) == -1) {
                return null;
            }

            countBuffer.flip();
            idSize = countBuffer.getInt();

            // get document id string, 4 envelope points(long):
            buffer = ByteBuffer.allocate(idSize + 4*8);
            inChannel.read(buffer);
            buffer.flip();

            charArray = new byte[idSize];
            buffer.get(charArray);
            chars = ByteBuffer.wrap(charArray);

            String idString = null;
            idString = new String(isoDecoder.decode(chars).array());
            
            logger.trace("read ID: " + idString);
            
            Rectangle rect = new Rectangle(buffer.getLong(), buffer.getLong(),
                    buffer.getLong(), buffer.getLong());
            
            logger.trace("read rect: " + rect.toString());
            
            //read the fields' String and store it as it is
            countBuffer = ByteBuffer.allocate(4);
            if (inChannel.read(countBuffer) == -1) {
                return null;
            }

            countBuffer.flip();
            idSize = countBuffer.getInt();

            // get the fields' string
            buffer = ByteBuffer.allocate(idSize);
            inChannel.read(buffer);
            buffer.flip();
            
            charArray = new byte[idSize];
            buffer.get(charArray);
            chars = ByteBuffer.wrap(charArray);

            String allFields = null;
            allFields = new String(isoDecoder.decode(chars).array());
            
            logger.trace("read allFields: " + allFields);
            
            //create a HashMap with the field names and values of this entry
            HashMap<String, String> fields = parseFieldString(allFields);
            //check if all the fields are declared in the indextype
            Iterator<String> fieldsIter = fields.keySet().iterator();
            while(fieldsIter.hasNext()) {
            	String fieldName = fieldsIter.next();
            	if(indexTypeObject.getFieldPosition(fieldName) == null) {
            		logger.error("The geo index lookup received a field with name : " 
            				+ fieldName + ", that is not declared in the geoindexType." 
            				+ " All fields String: " + allFields);
            		throw new IndexException("The geo index lookup received a field with name : " 
            				+ fieldName + ", that is not declared in the geoindexType.");
            	}
            }
                                    
            Data geoData = new Data(createDefinition());
            geoData.addValue(rect.getMinX());
            geoData.addValue(rect.getMaxX());
            geoData.addValue(rect.getMinY());
            geoData.addValue(rect.getMaxY());
            geoData.addValue(idString);

            long offset;
            int length;
            synchronized(rawDataFile)
            {
            	//we will append to the file
                rawDataFile.seek(rawDataFile.length());
	            //append the allFields String and write the offset, length 
	            //in the geoData 
	            offset = rawDataFile.getFilePointer();
	            byte[] byteArray = allFields.getBytes(DEFAULT_CHARSET);
	            length = byteArray.length;
	            rawDataFile.write(byteArray);
            }
            String position = "" + offset + OFFSET_SEPARATOR + length;
            //add the position in the raw Data file
            geoData.addValue(position);            
            
            return new DataWrapper(colIdString, colLangString, indexTypeObject, geoData, fields);
        } catch (Exception e) {
            logger.error("Error while getting DataWrapper instance.", e);
            throw new IndexException(e);
        }
    }

    /**
     * A method to generate a GeoTools DataDefinition object from an IndexType
     * object
     * 
     * @param indexTypeObject -
     *            The IndexType to be translated into a DataDefinition
     * @return a DataDefinition representation of the IndexType
     */
    private static DataDefinition createDefinition(GeoIndexType indexTypeObject) {
        DataDefinition df = new DataDefinition("ISO-8859-1");
        df.addField(Long.class); // x1
        df.addField(Long.class); // x2
        df.addField(Long.class); // y1
        df.addField(Long.class); // y2
        df.addField(GeoIndexField.DataType.STRING.getDefaultSize()); // id

        for (GeoIndexField field : indexTypeObject.fields) {
            if (!field.dataType.equals(GeoIndexField.DataType.STRING)) {
                df.addField(field.dataType.getTypeClass());
            } else {
                df.addField(field.size);
            }
        }
        return df;
    }
    
    /**
     * A method to generate the default GeoTools DataDefinition object. 
     * The default definition contains the id, the 4 coordinates, and a
     * String variable field for all the fields of a document(this makes 
     * possible the different structure of the fields for each document)  
     * 
     * @return a DataDefinition representation of the IndexType
     */
    public static DataDefinition createDefinition() {
        DataDefinition df = new DataDefinition("ISO-8859-1");
        df.addField(Long.class); // x1
        df.addField(Long.class); // x2
        df.addField(Long.class); // y1
        df.addField(Long.class); // y2
        df.addField(GeoIndexField.DataType.STRING.getDefaultSize()); // id

        df.addField(GeoIndexField.DataType.STRING.getDefaultSize()); //document fields
        
        return df;
    }
    
    /**
     * gets a String value from a raw data file
     * @param value (<offset>:<length>)
     * @return String in the file
     * @throws Exception
     */
    private static String getFromFile(String value, RandomAccessFile rawData) throws Exception{
    	//read the length and the offset
		int separator = value.indexOf(DataWrapper.OFFSET_SEPARATOR);
		if(separator == -1)
		{
			throw new Exception("in all fields we have stored: " + value);
		}
		long offset = Long.parseLong(value.substring(0, separator));
		int length = Integer.parseInt(value.substring(separator+1));
		
		int off = 0;
		// Create the byte array to hold the data
	    byte[] bytes = new byte[(int)length];
	    synchronized(rawData)
		{
			//read from the offset position, length bytes
			rawData.seek(offset);
			// Read in the bytes
		    int numRead = 0;
		    while (off < bytes.length
		           && (numRead=rawData.read(bytes, off, bytes.length-off)) >= 0) {
		        off += numRead;
		    }
		}

	    // Ensure all the bytes have been read in
	    if (off < bytes.length) {
	    	throw new Exception("Could not completely read raw data file. Read: "
	        		+ off + ", Length: " + bytes.length);
	    }
	    
	    String fieldData = new String(bytes, DEFAULT_CHARSET);
	    //TODO: remove this msg after testing
	    logger.debug("Variable field. " + " Data stored in Rtree: " 
	    		+ value + ", Offset: " + offset + ", Length: " + length
	    		+ ", fieldData: " + fieldData);
	    
	    return fieldData;
    }
    
    /**
     * parses a String containing all the fields of a document
     * @param value - The String value
     * @return fields - a HashMap containing the field names and data 
     */
    private static HashMap<String,String> parseFieldString(String allFields) throws Exception{
    	String currentString = allFields;
    	HashMap<String, String> ret = new HashMap<String, String>();
    	while(currentString.length() > 0) {
    		//get the fieldName
    		int separator = currentString.indexOf(IndexType.SEPERATOR_FIELD_INFO);
    		if(separator == -1)
    		{
    			throw new Exception("the all fields String we have stored: " + allFields);
    		}
    		int size = Integer.parseInt(currentString.substring(0, separator));
    		String fieldName = currentString.substring(separator+1, separator+1+size);
    		currentString = currentString.substring(separator+1+size);
    		//get the fieldData
    		separator = currentString.indexOf(IndexType.SEPERATOR_FIELD_INFO);
    		if(separator == -1)
    		{
    			throw new Exception("the all fields String we have stored: " + allFields);
    		}
    		size = Integer.parseInt(currentString.substring(0, separator));
    		String fieldData = currentString.substring(separator+1, separator+1+size);
    		String oldContent = ret.get(fieldName);
    		if(oldContent != null)
    		{
    			ret.put(fieldName, oldContent + fieldData);
    		}else{
    			ret.put(fieldName, fieldData);
    		}
    		currentString = currentString.substring(separator+1+size);    		
    	}
    	return ret;
    }

	public String getColID() {
		return colID;
	}

	public String getColLang() {
		return colLang;
	}


}
