package org.gcube.indexmanagement.featureindexlibrary.fullscan;

import java.io.*;

import org.apache.log4j.Logger;
import org.gcube.indexmanagement.featureindexlibrary.commons.*;

/**
 * Writer of the Full scan file
 * 
 * @author UoA
 */
public class FullScanWriter {
	/**
	 * The Logger used by this class
	 */
	private static Logger log = Logger.getLogger(FullScanWriter.class);
	/**
	 * The head parameters
	 */
	private FullScanParams charact =null;
	/**
	 * The output stream
	 */
	private FileOutputStream fos=null;
	/**
	 * Buffered stream
	 */
	private BufferedOutputStream bos=null;
	/**
	 * Data Stream
	 */
	private DataOutputStream ds=null;
	
	/**
	 * Creates a new instance
	 * 
	 * @param charact The head parameters
	 */
	public FullScanWriter(FullScanParams charact){
		this.charact=charact;
	}
	
	/**
	 * Reads a string
	 * 
	 * @param ds The pointer
	 * @return the string
	 */
	private String readString(RandomAccessFile ds){
		try{
			int count=ds.readInt();
			char []buf=new char[count];
			for(int i=0;i<count;i+=1){
				buf[i]=ds.readChar();
			}
			return new String(buf);
		}catch(EOFException e){
			return null;
		}catch(Exception e){
			log.debug("Could not read string. Assuming eof",e);
			return null;
		}
	}
	
	/**
	 * Reads a float vector
	 * 
	 * @param ds The pointer
	 * @return The vector
	 */
	private float []readVector(RandomAccessFile ds){
		try{
			int count=ds.readInt();
			float []buf=new float[count];
			for(int i=0;i<count;i+=1){
				buf[i]=ds.readFloat();
			}
			return buf;
		}catch(EOFException e){
			return null;
		}catch(Exception e){
			log.debug("Could not read string. Assuming eof",e);
			return null;
		}
	}
	
	/**
	 * Writes a string
	 * 
	 * @param toWrite The string to write
	 * @param ds The pointer
	 * @throws Exception An error
	 */
	private void writeString(String toWrite,DataOutputStream ds) throws Exception{
		char []toWriteA=toWrite.toCharArray();
		ds.writeInt(toWriteA.length);
		for(int i=0;i<toWriteA.length;i+=1){
			ds.writeChar(toWriteA[i]);
		}
	}
	
	/**
	 * Writes a string
	 * 
	 * @param toWrite The string to write
	 * @param ds the pointer
	 * @throws Exception An error
	 */
	private void writeString(String toWrite,RandomAccessFile ds) throws Exception{
		char []toWriteA=toWrite.toCharArray();
		ds.writeInt(toWriteA.length);
		for(int i=0;i<toWriteA.length;i+=1){
			ds.writeChar(toWriteA[i]);
		}
	}
	
	/**
	 * Writes  a vector
	 * 
	 * @param vector The vector
	 * @param ds the pointer
	 * @throws Exception An error
	 */
	private void writeVector(float []vector,DataOutputStream ds) throws Exception{
		ds.writeInt(vector.length);
		for(int i=0;i<vector.length;i+=1){
			ds.writeFloat(vector[i]);
		}
	}
	
	/**
	 * Writes the head 
	 * 
	 * @param file The file 
	 * @throws Exception An error
	 */
	public void writeHead(String file) throws Exception{
		RandomAccessFile rand=null;
		try{
			rand=new RandomAccessFile(file,"rw");
			rand.seek(0);
			rand.writeLong(this.charact.getElementCount());
			this.writeString(this.charact.getDistanceMeasure().toString(),rand);
			this.writeString(this.charact.getFeatureName().toString(),rand);
			this.writeString(this.charact.getIndexID(),rand);
			this.writeString(this.charact.getIndexName(),rand);
			this.writeString(this.charact.getStorage(),rand);
			rand.close();
		}catch(Exception e){
			if(rand!=null) rand.close();
			log.error("Could not write header. thrpwing exception",e);
			throw new Exception("Could not write header");
		}
	}
	
	/**
	 * Removes the entry
	 * 
	 * @param id the id
	 * @param file The file
	 * @throws Exception An error
	 */
	public void removeEntry(String id,String file) throws Exception{
		RandomAccessFile rand=null;;
		try{
			rand=new RandomAccessFile(file,"rw");
			rand.seek(0);
			rand.readLong();
			this.readString(rand);
			this.readString(rand);
			this.readString(rand);
			this.readString(rand);
			this.readString(rand);
			while(true){
				long offset=rand.getFilePointer();
				String idc=this.readString(rand);
				if(idc==null) break;
				float []vector=this.readVector(rand);
				if(vector==null) break;
				long newOffset=rand.getFilePointer();
				if(idc.compareTo(id)==0){
					rand.seek(offset);
					rand.write('-');
				}
				rand.seek(newOffset);
			}
			rand.close();
		}catch(Exception e){
			if(rand!=null)rand.close();
			log.error("Could not remove entry. thrpwing exception",e);
			throw new Exception("Could not remove entry");
		}
	}
	
	/**
	 * Writes  an entry
	 * 
	 * @param elem The element to write
	 * @throws Exception An error
	 */
	public void writeEntry(FeatureVectorElement elem) throws Exception{
		try{
			this.writeString(elem.getId(),this.ds);
			this.writeVector(elem.getVector(),this.ds);
		}catch(Exception e){
			log.error("Could not write entry. Throwing Exception",e);
			throw new Exception("Could not write entry");
		}
	}
	
	/**
	 * Opens the file for updating
	 * 
	 * @param file The file
	 * @throws Exception An error
	 */
	public void openForUpdate(String file) throws Exception{
		try{
			fos=new FileOutputStream(file,true);
			bos=new BufferedOutputStream(fos);
			ds=new DataOutputStream(bos);
		}catch(Exception e){
			log.error("Could not open writer. Throwing Exception",e);
			throw new Exception("Could not open writer");
		}
	}
	
	/**
	 * Closes the streams
	 * 
	 * @throws Exception An error
	 */
	public void close() throws Exception{
		try{
		    ds.close();
		    bos.close();
		    fos.close();
		}catch(Exception e){
			log.error("Could not close writer. Throwing Exception",e);
			throw new Exception("Could not close writer");
		}
	}
}
