/*
 * State.java
 *
 * $Author: tsakas $
 * $Date: 2007/12/20 14:37:39 $
 * $Id: FullTextIndexType.java,v 1.1 2007/12/20 14:37:39 tsakas Exp $
 *
 * <pre>
 *             Copyright (c) : 2006 Fast Search & Transfer ASA
 *                             ALL RIGHTS RESERVED
 * </pre>
 */

package org.gcube.indexmanagement.common;

import java.util.Vector;

import org.gcube.common.core.scope.GCUBEScope;
import org.gcube.common.core.utils.logging.GCUBELog;

/**
 * Representation of an index'es IndexType and IndexFormat
 */
public class FullTextIndexType extends IndexType {

	/** logger */
	static GCUBELog logger = new GCUBELog(FullTextIndexType.class);
	
    /** Then number of fields this index type contains */
    private int noOfFields;

    /** An array of the fields contained in the IndexType */
    private IndexField[] fields;

    /** The parser used to parse the xml representation of the IndexType */
    private XMLProfileParser parser = null;

    /**
     * The CSTR of the index type
     * 
     * @param indexTypeName -
     *            The index type name.
     */
    public FullTextIndexType(String indexTypeName, String scopeStr) {
    	super(indexTypeName);
    	
    	GCUBEScope scope = GCUBEScope.getScope(scopeStr);
    	
        try {
        	String indexType = retrieveIndexTypeGenericResource(scope);
            if (parser == null) {
                parser = new XMLProfileParser();
            }

            parser.readString(indexType, null);
            readIndexType();
        } catch (Exception ex) {
        	logger.error("Error initializing FullTextIndexType.", ex);
        }
    }

    /**
     * Get the number of fields in the index type.
     * 
     * @return - the number of fields in the index type.
     */
    public int getNumberOfFields() {
        return noOfFields;
    }

    /**
     * Get the specified array of fields.
     * 
     * @return - the arrary of fields in the index type.
     */
    public IndexField[] getFields() {
        return fields;
    }

    /**
     * Parses the indexType from the constructor into an
     * <code>IndexField Array</code>
     */
    private void readIndexType() throws Exception {
        int position = 0;
        try {
            // Extract the fields
            parser.setRootNode("field-list");
            noOfFields = parser.countDescendants("field");
            fields = new IndexField[noOfFields];
            Vector parents = new Vector();
            while (true) {
                if (parser.goChildElement("field")) {
                    fillField(position);
                    if (position != 0) {
                        parents.add(fields[position - 1]);
                        ((IndexField) parents.get(parents.size() - 1))
                                .addChildField(fields[position]);
                    }
                    position++;
                } else if (parser.goSiblingElement("field")) {
                    fillField(position);
                    if (!parents.isEmpty())
                        ((IndexField) parents.get(parents.size() - 1))
                                .addChildField(fields[position]);
                    position++;
                } else {
                    boolean foundAnotherNode = false;
                    while (parser.goParentElement("field")) {
                        parents.removeElementAt(parents.size() - 1);
                        if (parser.goSiblingElement("field")) {
                            fillField(position);
                            if (!parents.isEmpty())
                                ((IndexField) parents.get(parents.size() - 1))
                                        .addChildField(fields[position]);
                            position++;
                            foundAnotherNode = true;
                            break;
                        }
                    }
                    if (!foundAnotherNode)
                        return;
                }
            }

        } catch (Exception ex) {
            logger.error("Error while reading indexType, position: " + position, ex);
            throw ex;
        }
    }

    /**
     * Reads the current field into the current position of the
     * <code>IndexField Array</code>
     */
    private void fillField(int position) throws Exception {
        fields[position] = new IndexField();

        fields[position].name = parser.getFieldByValue("name");

        fields[position].index = parser.getFieldByValue("index")
                .equalsIgnoreCase("no") ? false : true;

        fields[position].tokenize = parser.getFieldByValue("tokenize")
                .equalsIgnoreCase("no") ? false : true;

        fields[position].store = parser.getFieldByValue("store")
                .equalsIgnoreCase("no") ? false : true;

        fields[position].returned = parser.getFieldByValue("return")
                .equalsIgnoreCase("no") ? false : true;

        fields[position].sort = parser.getFieldByValue("sort")
                .equalsIgnoreCase("no") ? false : true;

        fields[position].boost = (new Float(parser.getFieldByValue("boost")))
                .floatValue();
    }

    /**
     * {@inheritDoc}
     */
    public String toString() {
        StringBuffer ret = new StringBuffer("\nID=" + indexTypeID
                + "\n\nFields:\n");
        for (int i = 0; i < noOfFields; i++) {
            ret.append(fields[i] + "\n");
        }
        return ret.toString();
    }
}
