package org.gcube.indexmanagement.featureindexlibrary.fullscan;

import java.util.ArrayList;
import java.util.Calendar;
import org.apache.log4j.Logger;
import org.gcube.indexmanagement.featureindexlibrary.commons.*;

/**
 * The Full scan index
 * 
 * @author UoA
 */
public class FullVectorScan implements FeatureIndex{
	/**
	 * Used for synchronization
	 */
	private static final Object lockMe=new Object();
	/**
	 * The logger used by the class
	 */
	private static Logger logger = Logger.getLogger(FullVectorScan.class);
	/**
	 * The index head params
	 */
	private FullScanParams charact=null;
	
	/**
	 * Creates a new index
	 * 
	 * @param charact the head params
	 * @throws Exception An error
	 */
	public FullVectorScan(FullScanParams charact) throws Exception{
		try{
			this.charact=charact;
			if(this.charact.getDistanceMeasure().compareTo(FIEnums.DistanceTypes.Default)==0){
				logger.error("Cannot initialize index with default distance measure. throwing exception");
				throw new Exception("Cannot initialize index with default distance measure");
			}
			if(FileHelper.existsFullScanFile(charact.getStorage(),charact.getIndexID())){
				FullScanReader reader=new FullScanReader(FileHelper.getFullScanFile(charact.getStorage(),charact.getIndexID()));
				this.charact=reader.getCharacteristics();
				reader.close();
			}
		}catch(Exception e){
			logger.error("Could not initialize FullVectorSan. Throwinbg Exception",e);
			throw new Exception("Could not initialize FullVectorSan");
		}
	}
	
	/**
	 * @see org.gcube.indexmanagement.featureindexlibrary.commons.FeatureIndex#getNumberOfElements()
	 * @return the number of elements
	 */
	public long getNumberOfElements(){
		return this.charact.getElementCount();
	}
	
	/**
	 * @see org.gcube.indexmanagement.featureindexlibrary.commons.FeatureIndex#addFeatureVector(org.gcube.indexmanagement.featureindexlibrary.commons.FeatureVectorElement)
	 * @param elem The element to add
	 * @throws Exception an error
	 */
	public void addFeatureVector(FeatureVectorElement elem) throws Exception {
		synchronized (lockMe) {
			FullScanWriter writer =new FullScanWriter(this.charact);
			try{
				writer.openForUpdate(FileHelper.getFullScanBufferFile(charact.getStorage(),charact.getIndexID()));
				writer.writeEntry(elem);
				writer.close();
			}catch(Exception e){
				if(writer!=null) writer.close();
				logger.error("Could not add vector. Throwinbg Exception",e);
				throw new Exception("Could not add vector");
			}
		}
	}

	/**
	 * @see org.gcube.indexmanagement.featureindexlibrary.commons.FeatureIndex#addFeatureVector(org.gcube.indexmanagement.featureindexlibrary.commons.FeatureVectorElement[])
	 * @param elem the elements to add
	 * @throws Exception an error
	 */
	public void addFeatureVector(FeatureVectorElement []elem) throws Exception {
		synchronized (lockMe) {
			FullScanWriter writer =new FullScanWriter(this.charact);
			try{
				writer.openForUpdate(FileHelper.getFullScanBufferFile(charact.getStorage(),charact.getIndexID()));
				for(int i=0;i<elem.length;i+=1){
					writer.writeEntry(elem[i]);
				}
			}catch(Exception e){
				if(writer!=null) writer.close();
				logger.error("Could not add vector. Throwinbg Exception",e);
				throw new Exception("Could not add vector");
			}
		}
	}
	
	/**
	 * @see org.gcube.indexmanagement.featureindexlibrary.commons.FeatureIndex#commit()
	 * @throws Exception an error
	 */
	public void commit() throws Exception{
		synchronized (lockMe) {
			if(!FileHelper.existsFullScanBufferFile(charact.getStorage(),charact.getIndexID())){
				return;
			}
			FullScanWriter writer =null;
			FullScanReader reader=null;
			try{
				writer =new FullScanWriter(this.charact);
				reader=new FullScanReader(FileHelper.getFullScanBufferFile(charact.getStorage(),charact.getIndexID()));
				if(!FileHelper.existsFullScanFile(charact.getStorage(),charact.getIndexID())){
					writer.writeHead(FileHelper.getFullScanFile(charact.getStorage(),charact.getIndexID()));
				}
				writer.openForUpdate(FileHelper.getFullScanFile(charact.getStorage(),charact.getIndexID()));
				long count=this.charact.getElementCount();
				while(true){
					FeatureVectorElement elem=reader.getElement();
					if(elem==null) break;
					writer.writeEntry(elem);
					count+=1;
				}
				reader.close();
				writer.close();
				this.charact.setElementCount(count);
				writer.writeHead(FileHelper.getFullScanFile(charact.getStorage(),charact.getIndexID()));
				FileHelper.removeFullScanBufferFile(charact.getStorage(),charact.getIndexID());
			}catch(Exception e){
				if(reader!=null) reader.close();
				if(writer!=null) writer.close();
				FileHelper.removeFullScanBufferFile(charact.getStorage(),charact.getIndexID());
				logger.error("Could not commit vector. Throwinbg Exception",e);
				throw new Exception("Could not commit vector");
			}
		}
	}
	
	/**
	 * @see org.gcube.indexmanagement.featureindexlibrary.commons.FeatureIndex#lookup(float[], int)
	 * @param vector the vectror to search for
	 * @param k the number of results
	 * @return the results
	 * @throws Exception an error
	 */
	public ArrayList<RankedResultElement> lookup(float []vector,int k) throws Exception{
		return this.lookup(vector,k,new LookupParams(FIEnums.AlgoType.Default,FIEnums.DistanceTypes.Default,false));
	}
	
	/**
	 * @see org.gcube.indexmanagement.featureindexlibrary.commons.FeatureIndex#lookup(float[], int, org.gcube.indexmanagement.featureindexlibrary.commons.LookupParams)
	 * @param vector the vectror to search for
	 * @param k the number of results
	 * @param params lookup parameters
	 * @return the results
	 * @throws Exception an error
	 */
	public ArrayList<RankedResultElement> lookup(float []vector,int k,LookupParams params) throws Exception{
		if(!FileHelper.existsFullScanFile(charact.getStorage(),charact.getIndexID())){
			return new ArrayList<RankedResultElement>();
		}
		params.setWeighted(false);
		FullScanReader reader=null;
		try{
			long now=Calendar.getInstance().getTimeInMillis();
			long dtotal=0;
			long btotal=0;
			
			LookupBuffer buffer=new LookupBuffer(k);
			reader=new FullScanReader(FileHelper.getFullScanFile(charact.getStorage(),charact.getIndexID()));
			reader.getCharacteristics();
			FeatureVectorElement elem=null;
			float threshold=Float.MAX_VALUE;
			while(true){
				elem=reader.getElement();
				if(elem==null) break;
				long dnow=Calendar.getInstance().getTimeInMillis();
				FIEnums.DistanceTypes distMeasure=this.charact.getDistanceMeasure();
				if(params.getDist().compareTo(FIEnums.DistanceTypes.Default)!=0) distMeasure=params.getDist();
				float distance=DistanceCalculation.distance(vector,elem.getVector(),threshold,distMeasure);
				dtotal+=Calendar.getInstance().getTimeInMillis()-dnow;
				long bnow=Calendar.getInstance().getTimeInMillis();
				if(distance!=threshold) threshold=buffer.process(new RankedResultElement(elem.getId(),distance));
				btotal+=Calendar.getInstance().getTimeInMillis()-bnow;
			}
			logger.info("distance calculations took in total "+dtotal+" miliseconds");
			logger.info("buffer calculations took in total "+btotal+" miliseconds");
			logger.info("lookup took in total "+(Calendar.getInstance().getTimeInMillis()-now)+" miliseconds");
			return buffer.getResults(this.charact.getIndexName());
		}catch(Exception e){
			if(reader!=null) reader.close();
			logger.error("Could not perform lookup. throwing Exception",e);
			throw new Exception("Could not perform lookup");
		}
	}
	
	/**
	 * @see org.gcube.indexmanagement.featureindexlibrary.commons.FeatureIndex#removeEntry(java.lang.String)
	 * @param id the id to remove
	 * @throws Exception an error
	 */
	public void removeEntry(String id) throws Exception{
		synchronized (lockMe) {
			try{
				FullScanWriter writer=new FullScanWriter(this.charact);
				writer.removeEntry(id,FileHelper.getFullScanFile(this.charact.getStorage(),this.charact.getIndexID()));
			}catch(Exception e){
				logger.error("Could not remove entry. throwing Exception",e);
				throw new Exception("Could not remove entry");
			}
		}
	}

	/**
	 * Returns the index id of the index instance
	 * 
	 * @return The index ID
	 * @throws Exception An unrecoverable for the operation error occured
	 * @see org.gcube.indexmanagement.featureindexlibrary.commons.FeatureIndex#getIndexID()
	 */
	public String getIndexID() throws Exception {
		return this.charact.getIndexID();
	}
}
