/*
 * IndexField.java
 *
 * $Author: tsakas $
 * $Date: 2007/12/20 14:37:40 $
 * $Id: GeoIndexField.java,v 1.1 2007/12/20 14:37:40 tsakas Exp $
 *
 * <pre>
 *             Copyright (c) : 2006 Fast Search & Transfer ASA
 *                             ALL RIGHTS RESERVED
 * </pre>
 */

package org.gcube.indexmanagement.geo;

/**
 * Represent the characteristics of a single field in a RowSet.
 */
public class GeoIndexField {

    /** The name of the field */
    public String name;

    /**
     * An indicator of whether the size of this field is variable, 
     * and we can NOT store it directly to the R-tree of geo-tools
     */
    public boolean isVariable;
    
    /**
     * An indicator of whether the field is returnable (to be returned from a
     * query)
     */
    public boolean isReturnable;

    /** The byte size of the field */
    public int size;
    
    /** The data type of the field */
    public DataType dataType;
    
    /**
     * An enumeration of accepted data types, along with their respective Class
     * objects and default byte size
     */
    public enum DataType {
        /** 
         * The short data type.
         */
        SHORT(Short.class, 2), 
        /** 
         * The INT data type
         */
        INT(Integer.class, 4), 
        /** 
         * The LONG data type.
         */
        LONG(Long.class, 8), 
        /** 
         * The DATE data type.
         */
        DATE (Long.class, 8), 
        /** 
         * The FLOAT data type.
         */
        FLOAT(Float.class, 4), 
        /** 
         * The DOUBLE data type.
         */
        DOUBLE(Double.class, 8), 
        /** 
         * The STRING data type.
         */
        //TODO: fullpayload case 
        STRING(String.class, 100);
        
        /** 
         *The Class of this DataType instance 
         */
        private final Class typeClass;
        
        /** The default size */
        private final int defaultSize;
        
        /** Private constructor, used to create the accepted data types */
        private DataType(Class typeClass, int defaultSize) {
            this.typeClass = typeClass;
            this.defaultSize = defaultSize;
        }
        
        /** A case insensitive version of valueOf(String).
         * @param name - the string value to gte the type for
         * @return DataType of the input string name.
         * @throws IndexException - in case of failure.
         */
        public static DataType getByName(String name) throws Exception {
            for (DataType type : DataType.values()) {
                if (name.equalsIgnoreCase(type.toString())) {
                    return type;
                }
            }
            throw new Exception(
                    "specified DataTypeName does not match any DataTypes: \""
                            + name + "\"");
        }

        /**
         * Getter method for the Class object of this data type
         * 
         * @return the Class object of this data type
         */
        public Class getTypeClass() {
            return typeClass;
        }

        /**
         * Getter method for the default byte size of this data type
         * 
         * @return the default byte size of this data type
         */
        public int getDefaultSize() {
            return defaultSize;
        }

    }

    /**
     * Empty constructor. The field attributes must be set in subsequent calls.
     */
    public GeoIndexField() {
    }

    /**
     * Constructor setting all field attributes.
     * 
     * @param name -
     *            The name of the field to be created
     * @param isReturnable -
     *            An indicator of whether the field to be created is returnable
     *            (form a query)
     * @param dataType -
     *            The data type of the field to be created
     */
    public GeoIndexField(String name, boolean isReturnable, DataType dataType) {
        this.name = name;
        this.isReturnable = isReturnable;
        this.dataType = dataType;
        this.size = dataType.getDefaultSize();
    }

    /**
     * Constructor setting all field attributes. Sets the data type to
     * DataType.STRING with the specified byte size.
     * 
     * @param name -
     *            The name of the field to be created
     * @param isReturnable -
     *            An indicator of whether the field to be created is returnable
     *            (form a query)
     * @param stringDataTypeSize -
     *            The byte size of the field to be created
     */
    public GeoIndexField(String name, boolean isReturnable,
            int stringDataTypeSize) {
        this.name = name;
        this.isReturnable = isReturnable;
        this.dataType = DataType.STRING;
        this.size = stringDataTypeSize;
    }

    /**
     * {@inheritDoc}
     */
    public String toString() {
        return name + "( isReturnable=" + isReturnable + " dataType="
                + dataType + " class=" + dataType.getTypeClass().getName()
                + " size=" + size + " )";
    }
}
