package org.gcube.indexmanagement.common;

import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.util.ArrayList;

import javax.xml.namespace.QName;

import org.gcube.common.core.informationsystem.notifier.ISNotifier.NotificationEvent;
import org.gcube.common.core.scope.GCUBEScope;
import org.gcube.indexmanagement.common.notifications.NotifierRequestQueue;
import org.gcube.indexmanagement.common.notifications.SubscribeToNotificationRequest;
import org.gcube.indexmanagement.common.notifications.UnsubscribeFromNotificationRequest;
import org.globus.wsrf.Topic;
import org.w3c.dom.Element;

/**
 * @author Spyros Boutsis, NKUA
 *
 */
public abstract class IndexLookupWSResource extends IndexWSResource {

	/** The list of standard resource properties defined by every lookup index resource */
    public static final String RP_CONNECTION_ID = "ConnectionID";
    public static final String RP_DOCCOUNT = "DocumentCount";
    public static final String RP_SUPPORTED_RELATIONS = "SupportedRelations";
    
	/** The name of the state change notification topic */
	private static final String STATE_CHANGE_TOPIC_NAME = "SharedStateChange";
	
	/** The name of the index update notification topic */
	private static final String INDEX_UPDATE_TOPIC_NAME = "UpdateNotification";
	 
	/** The state change notification topic QName*/
	protected QName sharedStateChangeTopicName;
	
	/** The index update notification topic */
	protected Topic indexChangeTopic;

	/** The namespace of the topic consumed by the updater */
	private String managementServiceNamespace;
	
	/** The directory where index data is kept in the local filesystem */
	private String dataDir;
	
	/**
	 * Initializes the index management resource.
	 * 
	 * @param namespace the namespace of the service that this resource belongs to
	 * @param managementServiceNamespace the namespace of the corresponding index management service
	 * @param indexID the indexID of this resource
	 * @param indexTypeName the index type name of this resource
	 * @param collectionID the list of collection IDs of this resource
	 * @param dataDir the local directory where index data is stored
	 * @throws Exception an error occured
	 */
	public void initialise(String namespace, String managementServiceNamespace, String indexID, String indexTypeName, String collectionID[], String dataDir, String supportedRelations[]) throws Exception {
		super.initialise(namespace, indexID, indexTypeName, collectionID);
		
		this.managementServiceNamespace = managementServiceNamespace;
		this.dataDir = dataDir;
		
		/* Create and initialize the common lookup resource properties */
		createProperty(RP_CONNECTION_ID);
		
		createProperty(RP_DOCCOUNT);
		this.getResourcePropertySet().get(RP_DOCCOUNT).clear();
    	this.getResourcePropertySet().get(RP_DOCCOUNT).add(new Integer(0));
    	
    	createProperty(RP_SUPPORTED_RELATIONS);
		int size = supportedRelations.length;
    	this.getResourcePropertySet().get(RP_SUPPORTED_RELATIONS).clear();
    	for (int i=0; i<size; i++)
    		this.getResourcePropertySet().get(RP_SUPPORTED_RELATIONS).add(supportedRelations[i]);
		
		/* Register as a consumer of the state change notification topic
		 * exposed by the management service
		 */
        this.sharedStateChangeTopicName = new QName(managementServiceNamespace + "/" + indexID, STATE_CHANGE_TOPIC_NAME);
        ArrayList<QName> topicSubscriptions = new ArrayList<QName>();
        topicSubscriptions.add(sharedStateChangeTopicName);
		GCUBEScope scope = this.getServiceContext().getScope();
		//ISNotifier notifier = GHNContext.getImplementation(ISNotifier.class);
		//notifier.registerToISNotification(topicSubscriptions, new ConsumerNotification(scope), this.getServiceContext(), scope);
		NotifierRequestQueue.getInstance().add(
				new SubscribeToNotificationRequest(
						topicSubscriptions, new ConsumerNotification(scope), this.getServiceContext(), scope
				)
		);
		logger.debug("Consumer subscribed for notification on topic: " + sharedStateChangeTopicName);

        /* Register the index update topic to the IS */
        this.indexChangeTopic = this.createTopic(INDEX_UPDATE_TOPIC_NAME, namespace + "/" + indexID);
	}
	
	/**
	 * Invoked when a resource is being created from a serialized, previously saved state.
	 * 
	 * @param ois the input stream through which the state can be read
	 * @param indicates if the resource is being loaded for the first time (hard load) or not (soft load)
	 * @throws Exception an error occured during resource deserialization
	 */
	protected void onLoad(ObjectInputStream ois, boolean firstLoad) throws Exception {
		super.onLoad(ois, firstLoad);
		
		/* Create and initialize the common lookup resource properties */
		createProperty(RP_CONNECTION_ID);
		int size = ois.readInt();
    	for (int i=0; i<size; i++)
    		this.getResourcePropertySet().get(RP_CONNECTION_ID).add((Integer) ois.readInt());
    	
    	Integer docCount = ois.readInt();
		createProperty(RP_DOCCOUNT);
		this.getResourcePropertySet().get(RP_DOCCOUNT).clear();
    	this.getResourcePropertySet().get(RP_DOCCOUNT).add(docCount);

    	this.managementServiceNamespace = (String) ois.readObject();

    	this.dataDir = (String) ois.readObject();
    	
    	size = ois.readInt();
		createProperty(RP_SUPPORTED_RELATIONS);
    	this.getResourcePropertySet().get(RP_SUPPORTED_RELATIONS).clear();
    	for (int i=0; i<size; i++)
    		this.getResourcePropertySet().get(RP_SUPPORTED_RELATIONS).add((String) ois.readObject());
    	
		/* Register as a consumer of the state change notification topic
		 * exposed by the management service
		 */
    	if (firstLoad) {
	        this.sharedStateChangeTopicName = new QName(managementServiceNamespace + "/" + this.getIndexID(), STATE_CHANGE_TOPIC_NAME);
	        ArrayList<QName> topicSubscriptions = new ArrayList<QName>();
	        topicSubscriptions.add(sharedStateChangeTopicName);
	        GCUBEScope scope = GCUBEScope.getScope(this.getResourcePropertySet().getScope().get(0));
	        //ISNotifier notifier = GHNContext.getImplementation(ISNotifier.class);
			//notifier.registerToISNotification(topicSubscriptions, new ConsumerNotification(scope), this.getServiceContext(), scope);
	        NotifierRequestQueue.getInstance().add(
					new SubscribeToNotificationRequest(
							topicSubscriptions, new ConsumerNotification(scope), this.getServiceContext(), scope
					)
			);
			logger.debug("Consumer subscribed for notification on topic: " + sharedStateChangeTopicName);
    	}
    	
        /* Register the index update topic to the IS */
        this.indexChangeTopic = this.createTopic(INDEX_UPDATE_TOPIC_NAME, namespace + "/" + this.getIndexID());
	}
	
	/**
	 * Invoked when the state of the resource must be saved (resource serialization)
	 * 
	 * @param oos the output stream to write the resource state to
	 * @throws Exception an error occured during resource serialization
	 */
	protected void onStore(ObjectOutputStream oos) throws Exception {
		super.onStore(oos);
		
		int size = this.getResourcePropertySet().get(RP_CONNECTION_ID).size();
		oos.writeInt(size);
		for (int i=0; i<size; i++)
			oos.writeInt((Integer) this.getResourcePropertySet().get(RP_CONNECTION_ID).get(i));
		
		oos.writeInt((Integer) this.getResourcePropertySet().get(RP_DOCCOUNT).get(0));
		
		oos.writeObject(managementServiceNamespace);
		
		oos.writeObject(dataDir);
		
		size = this.getResourcePropertySet().get(RP_SUPPORTED_RELATIONS).size();
		oos.writeInt(size);
		for (int i=0; i<size; i++)
			oos.writeObject((String) this.getResourcePropertySet().get(RP_SUPPORTED_RELATIONS).get(i));
	}
	
	/**
	 * Sends a state change notification message
	 * @param message the notification message to send
	 */
	public void sendIndexChangeNotification(Object message) {
		try {
			indexChangeTopic.notify(message);   
			logger.debug("Index update notification sent on topic: " + indexChangeTopic.getName());
		} catch(Exception e) {
			logger.error("Exception while trying to send index update notification.", e);
		}
	}

    /**
     * Getter method for the ConnectionID Resource Property
     * @return <code>int</code> the requested connection number
     */
    public int getConnectionID() {
        return (Integer) this.getResourcePropertySet().get(RP_CONNECTION_ID).get(0);
    }

    /**
     * Setter method for the ConnectionID Resource Property.
     * @param connectionID - the id of the connection
     */
    public void setConnectionID(int connectionID) {
    	this.getResourcePropertySet().get(RP_CONNECTION_ID).clear();
    	this.getResourcePropertySet().get(RP_CONNECTION_ID).add(connectionID);
    }
    
    /**
     * Adds a connection ID.
     * @param connectionID
     */
    public void addConnectionID(Integer connectionID) {
    	this.getResourcePropertySet().get(RP_CONNECTION_ID).add(connectionID);
    }
    
    /**
     * Setter method for the DocumentCount Resource Property
     * @param documentCount <code>int</code> the new document count
     */
    public void setDocumentCount(int documentCount) {
    	this.getResourcePropertySet().get(RP_DOCCOUNT).clear();
    	this.getResourcePropertySet().get(RP_DOCCOUNT).add(new Integer(documentCount));
    }

    /**
     * Getter method for the DocumentCount Resource Property
     * @return - the document count <code>int</code>.
     */
    public int getDocumentCount() {
        return (Integer) this.getResourcePropertySet().get(RP_DOCCOUNT).get(0);
    }
    
    /**
     * Getter method for the SupportedRelations list Resource Property
     * @return <code>String[]</code> the requested SupportedRelations
     */
    public String[] getSupportedRelations() {
    	int size = this.getResourcePropertySet().get(RP_SUPPORTED_RELATIONS).size();
    	String[] relations = new String[size];
    	for (int i=0; i<size; i++)
    		relations[i] = (String) this.getResourcePropertySet().get(RP_SUPPORTED_RELATIONS).get(i);
    	return relations;
    }
    
    /**
     * Setter method for the SupportedRelations list Resource Property
     * @param <code>String[]</code> the new SupportedRelations
     */
    public void setSupportedRelations(String[] relations) throws Exception {
    	int size = relations.length;
    	this.getResourcePropertySet().get(RP_SUPPORTED_RELATIONS).clear();
    	for (int i=0; i<size; i++)
    		this.getResourcePropertySet().get(RP_SUPPORTED_RELATIONS).add(relations[i]);
    }

    /**
     * Modifies the DocumentCount Resource Property
     * @return - the document count <code>int</code>.
     */
    public void changeDocumentCount(int change) {
        setDocumentCount(getDocumentCount() + change);
    }

    /**
     * Returns the local directory where index data is stored
     * @return the local index data directory
     */
    public String getIndexDataDirectory() {
    	return dataDir;
    }
    
    /**
     * Returns the namespace of the management resource associated with this lookup resource
     * @return the namespace
     */
    public String getManagementResourceNamespace() {
    	return this.managementServiceNamespace;
    }
    
    /*
     * (non-Javadoc)
     * @see org.gcube.indexmanagement.common.IndexWSResource#onResourceRemoval()
     */
    public void onResourceRemoval() {
		super.onResourceRemoval();
		
		try{			
	        ArrayList<QName> topicSubscriptions = new ArrayList<QName>();
	        topicSubscriptions.add(sharedStateChangeTopicName);
//			ISNotifier notifier = GHNContext.getImplementation(ISNotifier.class);
//			notifier.unregisterFromISNotification(topicSubscriptions, this.getServiceContext(),
//					GCUBEScope.getScope(this.getResourcePropertySet().getScope().get(0)));
	        NotifierRequestQueue.getInstance().add(
					new UnsubscribeFromNotificationRequest(
							topicSubscriptions, this.getServiceContext(), 
							GCUBEScope.getScope(this.getResourcePropertySet().getScope().get(0))
					)
			);
			logger.debug("Consumer unsubscribed from notification on topic: " + sharedStateChangeTopicName + " for index: " + this.getIndexID());
		} catch(Exception e){
            logger.debug("Failed to unregister notification topics from the IS.", e);
		}
	}
	
	/**
	 * Invoked whenever a message is received for this consumer.
	 * @param message the received message
 	 */
	public abstract void onLookupNotificationReceived(Element message);
	
	/**
	 * Class that handles the consuming of received notifications
	 * @author Spyros Boutsis, NKUA
	 */
	public class ConsumerNotification extends IndexNotificationConsumer {
		public ConsumerNotification(GCUBEScope scope) { super(IndexLookupWSResource.this, scope); }
		
		protected void onNewNotification(NotificationEvent event){ 
			logger.debug("Index notification consumer received notification message.");
			Element message = null;
			try {
				message = (Element) event.getPayload().getMessageObject();
				onLookupNotificationReceived(message);
			}  catch (Exception e) {
				logger.debug("Error in onNotificationReceived.", e);
			}
		}
	}
}
