package org.gcube.indexmanagement.featureindexlibrary.vafile.algo;

import java.util.ArrayList;
import java.util.Collections;

import org.apache.log4j.Logger;
import org.gcube.indexmanagement.featureindexlibrary.commons.FeatureVectorElement;
import org.gcube.indexmanagement.featureindexlibrary.vafile.VAFileParams;
import org.gcube.indexmanagement.featureindexlibrary.vafile.io.FileBufferReader;

/**
 * Computes the partitoning points in each dimention
 * 
 * @author UoA
 */
public class PointPartitioning {
	/**
	 * The Logger used by this class
	 */
	private static Logger log = Logger.getLogger(PointPartitioning.class);
	/**
	 * The file holding the vectors
	 */
	private String file=null;
	/**
	 * The VAFile header params
	 */
	private VAFileParams params=null;
	/**
	 * The partitioning points
	 */
	private float [][]partitionPoints=null;
	
	
	/**
	 * Creates a new instance
	 * 
	 * @param file The file holding the vectores
	 * @param params The VAFile header params
	 */
	public PointPartitioning(String file, VAFileParams params){
		this.file=file;
		this.params=params;
	}
	
	/**
	 * Retrieves teh backets in each partition
	 * 
	 * @return Teh buckets
	 */
	public int getPartitionBuckets(){
		int buckets=(int)Math.pow(2,params.getBits());
		return buckets+1;
	}
	
	/**
	 * Retrieves teh partitioning points
	 * 
	 * @return The points
	 */
	public float [][]getPartitionPoints(){
		return this.partitionPoints;
	}
	
	/**
	 * Computes the partitioning points
	 * 
	 * @throws Exception An error
	 */
	public void computePartitionPoints() throws Exception{
		try{
			this.partitionPoints=new float [params.getVectorLength()][];
			for(int i=0;i<params.getVectorLength();i+=1){
				this.partitionPoints[i]=this.computePartitionPoints(i);
			}
		}catch(Exception e){
			log.error("Could not compute partition points. Throwing Exception",e);
			throw new Exception("Could not compute partition points");
		}
	}
	
	/**
	 * Computes partitioning points in each dimention
	 * 
	 * @param dim The dimention
	 * @return The points
	 * @throws Exception An error
	 */
	private float []computePartitionPoints(int dim) throws Exception{
		try{
			int buckets=(int)Math.pow(2,params.getBits());
			float []points=new float[buckets+1];
			ArrayList<Float> vectordim=new ArrayList<Float>();
			FileBufferReader reader=new FileBufferReader(file,params);
			reader.openForReading();
			FeatureVectorElement elem=null;
			while(true){
				elem=reader.readElement();
				if(elem==null) break;
				vectordim.add(new Float(elem.getVector()[dim]));
			}
			reader.close();
			Collections.sort(vectordim);
			int count=(int)(vectordim.size()/buckets);
			points[0]=(float)vectordim.get(0);
			points[points.length-1]=(float)vectordim.get(vectordim.size()-1);
			int []indexes=new int[buckets-1];
			for(int i=0;i<indexes.length;i+=1){
				indexes[i]=(i+1)*count;
			}
			int addition=0;
			for(int i=0;i<vectordim.size()-(buckets*count);i+=1){
				indexes[i]+=i+1;
				addition=i+1;
			}
			for(int i=vectordim.size()-(buckets*count);i<indexes.length;i+=1){
				indexes[i]+=addition;
			}
			for(int i=1;i<buckets;i+=1){
				if(indexes[i-1]>=vectordim.size()) indexes[i-1]=vectordim.size()-1;
				points[i]=(float)vectordim.get(indexes[i-1]);
				//logger.info("index = "+indexes[i-1]);
				//logger.info("point = "+(float)vectordim.get(indexes[i-1]));
			}
			//logger.info("\n------------------------------------------\n");
			return points;
		}catch(Exception e){
			log.error("Could not compute partition points for dimention "+dim+". Throwing exception",e);
			throw new Exception("Could not compute partition points for dimention "+dim);
		}
	}
}
