/**
 * JdbmWrapper.java
 *
 * $Author: spiros $
 * $Date: 2008/03/13 10:10:59 $
 * $Id: JdbmWrapper.java,v 1.9 2008/03/13 10:10:59 spiros Exp $
 *
 * <pre>
 *             Copyright (c) : 2007 Fast Search & Transfer ASA
 *                             ALL RIGHTS RESERVED
 * </pre>
 */

package org.gcube.indexmanagement.jdbmwrapper;

import java.io.File;
import java.io.FileInputStream;
import java.io.FilenameFilter;
import java.io.StringReader;
import java.nio.ByteBuffer;
import java.nio.channels.ReadableByteChannel;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Comparator;
import java.util.Date;
import java.util.LinkedList;
import java.util.ListIterator;
import java.util.Properties;

import jdbm.RecordManager;
import jdbm.RecordManagerFactory;
import jdbm.btree.BTree;
import jdbm.helper.IntegerComparator;
import jdbm.helper.StringComparator;
import jdbm.helper.Tuple;
import jdbm.helper.TupleBrowser;

import org.gcube.common.core.utils.logging.GCUBELog;
import org.gcube.indexmanagement.common.IndexException;
import org.gcube.indexmanagement.common.PropertyElementForwardIndex;
import org.gcube.indexmanagement.common.XMLProfileParser;
import org.gcube.indexmanagement.common.XMLTokenReplacer;
import org.gcube.common.searchservice.searchlibrary.resultset.elements.PropertyElementBase;
import org.gcube.common.searchservice.searchlibrary.resultset.elements.ResultElementGeneric;
import org.gcube.common.searchservice.searchlibrary.rsclient.elements.RSLocator;
import org.gcube.common.searchservice.searchlibrary.rsclient.elements.RSResourceWSRFType;
import org.gcube.common.searchservice.searchlibrary.rswriter.RSXMLWriter;
import org.xml.sax.Attributes;
import org.xml.sax.InputSource;
import org.xml.sax.SAXParseException;
import org.xml.sax.XMLReader;
import org.xml.sax.helpers.DefaultHandler;
import org.xml.sax.helpers.XMLReaderFactory;


enum LookupType { EQUAL, NOTEQUAL, LT, LE, GT, GE, GTANDLT, GTANDLE, GEANDLT, GEANDLE };

/**
 *  Class to store key and value pairs in the jdbm database
 */
public class JdbmWrapper {
    /**
     * @param dbPathAndName   Full path and name of storage file for the b-tree.
     */    
    private String dbPathAndName;
    
    /**
     * @param treeName   Name of the tree to store within the storage file.
     */
    private String treeName;
    
    /**
     * @param treeDir   Path to storage.
     */
    private String treeDir;
    
    /**
     * @param props   Properties for the JDBM
     */
    private Properties props;

    /**
     * @param tree   The Integer-based B-tree.
     */
    private BTree tree;

    /**
     * @param recman   The record manager, handling the underlying file storage
     */
    private RecordManager recman;

    /**
     * @param recid   Identification of the record manager.
     */
    private long recid;

    /* Log4j */
    static GCUBELog logger = new GCUBELog(JdbmWrapper.class);

    /**
     * @param comparator that is used to compare the keys.
     */
    private Comparator comparator;
    
    /**
     * The enum forward index types datebase type.
     * @param INT_TYPE contains an int, string
     * @param FLOAT_TYPE contains and float, string
     * @param STRING_TYPE contains a string, string
     * @param DATE_TYPE contians a date,string
     */
    public enum DbTypeEnum {INT_TYPE,FLOAT_TYPE,STRING_TYPE,DATE_TYPE};
    
    private DbTypeEnum dbType; 

    /**
     * Provides the ability to store simple key-value pairs. 
     * @param treeDir    The base directory
     * @param dbName     The nameof the database
     * @param treeName   The name of the tree
     * @param dbT        The type of DB (int,string or float,string or string,string
     *                   date,string).
     * @throws IndexException - in case of failure.
     */
    public JdbmWrapper(String treeDir,String dbName, String treeName, DbTypeEnum dbT)
        throws IndexException {
        // Logger
        
        this.dbType = dbT;
        logger.debug(" >>> JdbmWrapper \n"  
                     + "treeDir: " + treeDir 
                     + "dbName: " + dbName
                     + "treeName: " + treeName
                     + "dbT " + dbT);
        this.treeName = treeName;
        this.props = new Properties();
        this.treeDir = treeDir;
        this.dbPathAndName = this.treeDir + dbName;
        try {
            this.recman = RecordManagerFactory.createRecordManager(this.dbPathAndName,
								   props);
            this.recid = recman.getNamedObject(this.treeName);
            // either reload tree, or create a new one
            if (recid != 0) {
                logger.debug("   JdbmWrapper reload tree name " + dbPathAndName);
                this.tree = BTree.load(recman, recid);
                if (this.dbType == DbTypeEnum.INT_TYPE) {
                    logger.debug("    Int database loaded");
                    comparator = new IntegerComparator();
                }
                else if (this.dbType == DbTypeEnum.FLOAT_TYPE) {
                    logger.debug("    Float database loaded");
                    comparator = new FloatComparator();
                }
                else if (this.dbType == DbTypeEnum.STRING_TYPE) {
                    logger.debug("    String database loaded");
                    comparator = new StringComparator();
                }
                else if (this.dbType == DbTypeEnum.DATE_TYPE) {
                    logger.debug("    Date database loaded");
                    comparator = new DateComparator();
                } 
                else {
                    // This can never be reached
                    logger.error("ForwardIndex Can not create database");
                }
            } 
            else {
                if (this.dbType == DbTypeEnum.INT_TYPE) {
                    this.tree = BTree.createInstance(recman,
                                                     new IntegerComparator());
                    recman.setNamedObject(this.treeName, tree.getRecid());
                    comparator = new IntegerComparator();
                    logger.debug("    New int datbase created ");
                }
                else if (this.dbType == DbTypeEnum.FLOAT_TYPE) {
                    this.tree = BTree.createInstance(recman,
                                                     new FloatComparator());
                    recman.setNamedObject(this.treeName, tree.getRecid());
                    comparator = new FloatComparator();
                    logger.debug("    New float datbase created ");
                }
                else if (this.dbType == DbTypeEnum.STRING_TYPE) {
                    this.tree = BTree.createInstance(recman,
                                                     new StringComparator());
                    recman.setNamedObject(this.treeName, tree.getRecid());
                    comparator = new StringComparator();
                    logger.debug("    New string datbase created ");
                }
                else if (this.dbType == DbTypeEnum.DATE_TYPE) {
                    this.tree = BTree.createInstance(recman,
                                                     new DateComparator());
                    recman.setNamedObject(this.treeName, tree.getRecid());
                    comparator = new DateComparator();
                    logger.debug("    New date datbase created ");
                } 
                else {
                    // This can never be reached
                    logger.error(" *** ForwardIndex Can not create database");
                }
            }
            logger.debug(" <<< JdbmWrapper");  
        } 
        catch (Exception e) {
            logger.error("db create failed ", e);
            // Must be handled
        }
    }

    /**
     * Closes this jdbmWrapper by deleting the whole index
     * @throws Exception
     */
    public void close() throws Exception {
    	/* Wrap the index data directory in a File object */
    	File dataDir = new File(this.treeDir);
    	
    	/* Construct a filename filter to find all the files belonging to the specific index 
    	 * (indexName-based) */
    	FilenameFilter filter = new FilenameFilter() {
    		public boolean accept(File dir, String name) {
    			return name.startsWith(treeName);
    		}
    	};
    	
    	/* Find the index data files and delete them */
    	for (File f : dataDir.listFiles(filter))
    		f.delete();
    }

    
    /**
     * Insert a key and value pair in the the tree.
     * 
     * @param key -   the tuple key.
     * @param value - the tuple value.
     * @throws IndexException - when IOException is received.
     */
    public void insertPairInDb(Object key, Object value) throws IndexException {
        logger.debug(" >>> insertPairInDb key/value:  "+ key + " / " + value);
        LinkedList valueList;      
        try {
            // If key exists, get the list, if not create a new list
            if(this.tree.find(key) == null) {
                valueList = new LinkedList();
            } else {
                valueList = (LinkedList)this.tree.find(key);
            }
            // add value to list (doesn't matter if list is old or new)
            valueList.add(value);
            this.tree.insert(key,valueList,true);
            // Warning, commiting will use a great deal of time
            this.recman.commit();
            logger.debug(" <<< insertPairInDb");
        } 
        catch (Exception e) {
            logger.error(" *** insertPairInDb error: ", e);
            throw new IndexException(
                                     " *** Error *** insertPair got IOException: "
                                     + " key/value: " + key + "/" + value + ":"
                                     + e.getMessage());
        }
    }

    /**
     * Deletes a key and value pair in the the tree.
     * 
     * @param key -   the tuple key.
     * @param value - the tuple value.
     * @throws IndexException - when IOException is received.
     */
    public void deletePairInDb(Object key) throws IndexException {
        logger.debug(" >>> deletePairInDb: "+ key);
        try {   
            this.tree.remove(key);
            logger.debug(" <<< deletePairInDb");
        } 
        catch (Exception e) {
            logger.error(" *** deletePairInDb error for key: " + key + " message: " + e.getMessage());
            throw new IndexException(" *** Error *** deletePair got IOException: "
                                     + " key: " + key +":"
                                     + e.getMessage());
        }    
    }

    /**
     *  Inserts tuples in a rowset into the tree, parsing the values from a String.
     *  Deletes tuples in a rowset from the tree, parsing the values from a String.
     * @param rowset - the rowset containing the tuples
     * @throws IndexException - when parsing goes bonk.
     */
    public void insertRowSet(String rowset) throws IndexException {
        logger.debug(" >>> insertRowSet:\n" + rowset);
        XMLProfileParser XMLparser = new XMLProfileParser();
        String[][] fieldData = null;
        try {
            XMLparser.readString(rowset, null);
            XMLparser.setRootNode("ROWSET");
            
            fieldData = XMLparser.getSubFields();
            
            if ((fieldData[0][0].equalsIgnoreCase("insert")) ||
                (fieldData[0][1].equalsIgnoreCase("insert"))) {
                XMLparser.setRootNode("tuple");                
                while (XMLparser.setNextField()) {
                    fieldData = XMLparser.getSubFields();
                    if (!fieldData[0][0].equals("key")) {
                        logger.error(" *** insertRowSet error, insert error, key not defined");
                        throw new IndexException(" *** insertRowSet error ***: " +
                                                 "*** Error 'key' is not first field in rowset");
                    }
                    if (!fieldData[0][1].equals("value")) {
                        logger.error(" *** insertRowSet error, insert error, value not defined");
                        throw new IndexException(" *** insertRowSet error ***: " +
                                                 "*** Error 'value' is not second field in rowset");
                    }
                    Object key = fieldData[1][0];
                    Object value = fieldData[1][1];
                    
                    // NB! null values are converted to "null" by soap
                    // if( value.equals("null") || value == null)
                    // logger.info( "- Deleting "+key );
                    // Convert the key from String to the correct type
                    if (this.dbType == DbTypeEnum.INT_TYPE) {
                        insertPairInDb(convertToInt(key),value);
                    }
                    else if (this.dbType == DbTypeEnum.FLOAT_TYPE) {
                        insertPairInDb(convertToFloat(key),value);
                    }
                    else if (this.dbType == DbTypeEnum.STRING_TYPE) {
                        insertPairInDb(convertToString(key),value);
                    }
                    else if (this.dbType == DbTypeEnum.DATE_TYPE) {
                        insertPairInDb(convertToDate(key),value);
                    }
                }
            }
            if ((fieldData[0][0].equalsIgnoreCase("delete")) ||
                (fieldData[0][1].equalsIgnoreCase("delete"))) {
                XMLparser.setRootNode("tuple");                
                while (XMLparser.setNextField()) {
                    fieldData = XMLparser.getSubFields();
                    if (!fieldData[0][0].equals("key")) {
                        logger.error(" *** insertRowSet error, delete error, key not defined");
                        throw new IndexException(" *** deleteRowSet error ***: " +
                                                 "*** Error 'key' is not first field in rowset");
                    }
                    Object key = fieldData[1][0];                    
                    // NB! null values are converted to "null" by soap
                    // if( value.equals("null") || value == null)
                    // logger.info( "- Deleting "+key );
                    // Convert the key from String to the correct type
                    if (this.dbType == DbTypeEnum.INT_TYPE) {
                        deletePairInDb(convertToInt(key));
                    }
                    else if (this.dbType == DbTypeEnum.FLOAT_TYPE) {
                        deletePairInDb(convertToFloat(key));
                    }
                    else if (this.dbType == DbTypeEnum.STRING_TYPE) {
                        deletePairInDb(convertToString(key));
                    }
                    else if (this.dbType == DbTypeEnum.DATE_TYPE) {
                        deletePairInDb(convertToDate(key));
                    }
                }
            }
        } 
        catch (Exception e) {
            logger.error(" *** insertRowSet error " + e.getMessage());
            throw new IndexException(" *** insertRowSet error ***: "
                                     + e.getMessage());
        }
        logger.debug(" <<< insertRowSet");
    }

    /**
     * Adds a tuple to a resultset writer.
     * 
     * @param writer the RSXMLWriter to add results to
     * @param key the tuple's key
     * @param value the tuple's value
     */
    private void addToResults(RSXMLWriter writer, Object key, Object value) throws Exception {
    	String xmlResult = null;

		ResultElementGeneric element = new ResultElementGeneric();
		String serializedRSElem = XMLTokenReplacer.XMLResolve((String) value);
		element.RS_fromXML(serializedRSElem);
		
		xmlResult = "<tuple>";
		xmlResult += "<key>";
		xmlResult += key;
		xmlResult += "</key>";
		xmlResult += "<value>";
		xmlResult += element.getPayload();
		xmlResult += "</value>";
		xmlResult += "</tuple>";

		ResultElementGeneric tmp= new ResultElementGeneric("1","1",xmlResult);
		tmp.setRecordAttributes(element.getRecordAttributes());
		writer.addResults(tmp);

    	logger.debug("=================================");
    	logger.debug("jdbmWrapper added to lookup results: " + xmlResult);
    	logger.debug("=================================");
    }
    
    /**
     * Adds a key to a resultset writer.
     * 
     * @param writer the RSXMLWriter to add results to
     * @param key the tuple's key
     */
    private void addToResults(RSXMLWriter writer, Object key) throws Exception {
    	String xmlResult = "<key>" + key.toString() + "</key>";
		writer.addResults(new ResultElementGeneric("1", "1", xmlResult));

    	logger.debug("=================================");
    	logger.debug("jdbmWrapper added to lookup results: " + xmlResult);
    	logger.debug("=================================");
    }
    
    /** 
     * Gets the values where the key equals the input key.
     * @param key - The equality key for the search
     * @param bKeysOnly return only keys, or full tuples?
     * @return String - The EPR of the result set containing the result.
     * @throws IndexException - when IOException is received.
     */
    public String getEQ(final Object key, final boolean bKeysOnly) throws IndexException {
    	RSLocator locator = null;

    	try {
	    	int maxHits = 1000;
		    PropertyElementBase[] properties = {new PropertyElementForwardIndex(maxHits)};
	 	    final RSXMLWriter rsWriter =  RSXMLWriter.getRSXMLWriter(properties);
	 	    locator = rsWriter.getRSLocator(new RSResourceWSRFType());
	 	    
	 	    (new Thread() {
	 	    	public void run() {
	 	    		logger.debug(" >>> getEQ for key: "+ key);
	 	    		LinkedList valueList = null;
	 	    		try {
	 	    			valueList = (LinkedList) tree.find(key);
	 	    			if(valueList != null) {
	 	    				if (bKeysOnly)
	 	    					addToResults(rsWriter, key);
	 	    				else {
		 	    				ListIterator it = valueList.listIterator();
		 	    				while (it.hasNext()) {
		 	    					Object value = it.next();
		 	    					addToResults(rsWriter, key, value);
		 	    				}
	 	    				}
	 	    			}
	 	    			rsWriter.close();
	 	    		}
	 	    		catch(Exception e){
	 	    			logger.error(" *** getEQ error key: " + key + " message: " + e.getMessage());
	 	    			try {
    						rsWriter.close();
    					}
    					catch (Exception e1) { }
	 	    		}
	 	    		if (valueList == null) {
	 	    			logger.debug("getEQ no result for key: " + key);
	 	    		}
	 	    		logger.debug(" <<< getEQ");
	 	    	}
	 	    }).start();
 	    } catch (Exception e) {
 	    	logger.error(e);
 	    }
 	    return locator.getLocator();
    }
    
    /** 
     * Gets values >= keyGE.
     * @param keyGE - the key
     * @param bKeysOnly return only keys, or full tuples?
     * @return String - The EPR of the result set containing the result.
     * @throws IndexException - when IOException is received.
     */
    public String getGE(final Object keyGE, final boolean bKeysOnly) throws IndexException {
    	RSLocator locator = null;

    	try {
    		int maxHits = 1000;
    		PropertyElementBase[] properties = {new PropertyElementForwardIndex(maxHits)};
    		final RSXMLWriter rsWriter =  RSXMLWriter.getRSXMLWriter(properties);
    		locator = rsWriter.getRSLocator(new RSResourceWSRFType());

    		(new Thread() {
    			public void run() {
    				logger.debug(" >>> getGE for key: "+ keyGE);
    				LinkedList valueList = null;
    				Tuple tuple = new Tuple(null, null);
    				TupleBrowser browser;
    				Object nextKey;
    				try {
    					browser = tree.browse(keyGE);
    					while (browser.getNext(tuple)) {
    						nextKey = tuple.getKey();
    						if (bKeysOnly)
    							addToResults(rsWriter, nextKey);
    						else {
	    						valueList = (LinkedList)tuple.getValue();
	    						ListIterator it = valueList.listIterator();
	    						while (it.hasNext()) {
	    							Object value = it.next();
	    							addToResults(rsWriter, nextKey, value);
	    						}
    						}
    					}
    					rsWriter.close();
    				} 
    				catch(Exception e){
    					logger.error(" *** getGE error for key: "+ keyGE + " message: " + e.getMessage());
    					try {
    						rsWriter.close();
    					}
    					catch (Exception e1) { }
    				}
    				logger.debug(" <<< getGE");
    			}
    		}).start();
    	} catch (Exception e) {
    		logger.error(e);
    	}
    	return locator.getLocator();
    }
    
    /** 
     * Gets keyGT < values 
     * @param keyGT - The lower exclusive bound key
     * @param bKeysOnly return only keys, or full tuples?
     * @return String - The EPR of the result set containing the result.
     * @throws IndexException - when IOException is received.
     */
    public String getGT(final Object keyGT, final boolean bKeysOnly) throws IndexException {
    	RSLocator locator = null;

    	try {
    		int maxHits = 1000;
    		PropertyElementBase[] properties = {new PropertyElementForwardIndex(maxHits)};
    		final RSXMLWriter rsWriter =  RSXMLWriter.getRSXMLWriter(properties);
    		locator = rsWriter.getRSLocator(new RSResourceWSRFType());

    		(new Thread() {
    			public void run() {
    				logger.debug(" >>> getGT for key: "+ keyGT);
    				LinkedList valueList = null;
    				Tuple tuple = new Tuple(null, null);
    				TupleBrowser browser;
    				Object nextKey;
    				try {
    					browser = tree.browse(keyGT);
    					while (browser.getNext(tuple)) {
    						if (comparator.compare(tuple.getKey(), keyGT) <=  0) {
    		                    continue;
    		                }
    						nextKey = tuple.getKey();
    						if (bKeysOnly)
    							addToResults(rsWriter, nextKey);
    						else {
	    						valueList = (LinkedList)tuple.getValue();
	    						ListIterator it = valueList.listIterator();
	    						while (it.hasNext()) {
	    							Object value = it.next();
	    							addToResults(rsWriter, nextKey, value);
	    						}
    						}
    					}
    					rsWriter.close();
    				} 
    				catch(Exception e){
    					logger.error(" *** getGT error for key: "+ keyGT + " message: " + e.getMessage());
    					try {
    						rsWriter.close();
    					}
    					catch (Exception e1) { }
    				}
    				logger.debug(" <<< getGT");
    			}
    		}).start();
    	} catch (Exception e) {
    		logger.error(e);
    	}
    	return locator.getLocator();
    }

    /** 
     * Gets values <= keyLE 
     * @param keyLE - The upper inclusive bound key
     * @param bKeysOnly return only keys, or full tuples?
     * @return String - The EPR of the result set containing the result.
     * @throws IndexException - when IOException is received.
     */
    public String getLE(final Object keyLE, final boolean bKeysOnly) throws IndexException {
    	RSLocator locator = null;

    	try {
    		int maxHits = 1000;
    		PropertyElementBase[] properties = {new PropertyElementForwardIndex(maxHits)};
    		final RSXMLWriter rsWriter =  RSXMLWriter.getRSXMLWriter(properties);
    		locator = rsWriter.getRSLocator(new RSResourceWSRFType());

    		(new Thread() {
    			public void run() {
    				logger.debug(" >>> getLE for key: "+ keyLE);
    				LinkedList valueList = null;
    				Tuple tuple = new Tuple(null, null);
    				TupleBrowser browser;
    				Object nextKey;
    				try {
    					browser = tree.browse();
    					while (browser.getNext(tuple)) {
    						if (comparator.compare(tuple.getKey(), keyLE) >  0) {
    							break;
    			            }
    						nextKey = tuple.getKey();
    						if (bKeysOnly)
    							addToResults(rsWriter, nextKey);
    						else {
	    						valueList = (LinkedList)tuple.getValue();
	    						ListIterator it = valueList.listIterator();
	    						while (it.hasNext()) {
	    							Object value = it.next();
	    							addToResults(rsWriter, nextKey, value);
	    						}
    						}
    					}
    					rsWriter.close();
    				} 
    				catch(Exception e){
    					logger.error(" *** getLE error for key: "+ keyLE + " message: " + e.getMessage());
    					try {
    						rsWriter.close();
    					}
    					catch (Exception e1) { }
    				}
    				logger.debug(" <<< getLE");
    			}
    		}).start();
    	} catch (Exception e) {
    		logger.error(e);
    	}
    	return locator.getLocator();
    }
    
    /** 
     * Gets values < keyLT 
     * @param keyLT - The upper exclusive bound key
     * @param bKeysOnly return only keys, or full tuples?
     * @return String - The EPR of the result set containing the result.
     * @throws IndexException - when IOException is received.
     */
    public String getLT(final Object keyLT, final boolean bKeysOnly) throws IndexException {
    	RSLocator locator = null;

    	try {
    		int maxHits = 1000;
    		PropertyElementBase[] properties = {new PropertyElementForwardIndex(maxHits)};
    		final RSXMLWriter rsWriter =  RSXMLWriter.getRSXMLWriter(properties);
    		locator = rsWriter.getRSLocator(new RSResourceWSRFType());

    		(new Thread() {
    			public void run() {
    				logger.debug(" >>> getLT for key: "+ keyLT);
    				LinkedList valueList = null;
    				Tuple tuple = new Tuple(null, null);
    				TupleBrowser browser;
    				Object nextKey;
    				try {
    					browser = tree.browse();
    					while (browser.getNext(tuple)) {
    						if (comparator.compare(tuple.getKey(), keyLT) >=  0) {
    							break;
    			            }
    						nextKey = tuple.getKey();
    						if (bKeysOnly)
    							addToResults(rsWriter, nextKey);
    						else {
	    						valueList = (LinkedList)tuple.getValue();
	    						ListIterator it = valueList.listIterator();
	    						while (it.hasNext()) {
	    							Object value = it.next();
	    							addToResults(rsWriter, nextKey, value);
	    						}
    						}
    					}
    					rsWriter.close();
    				} 
    				catch(Exception e){
    					logger.error(" *** getLT error for key: "+ keyLT + " message: " + e.getMessage());
    					try {
    						rsWriter.close();
    					}
    					catch (Exception e1) { }
    				}
    				logger.debug(" <<< getLT");
    			}
    		}).start();
    	} catch (Exception e) {
    		logger.error(e);
    	}
    	return locator.getLocator();
    }
    
    /** 
     * Gets keyGE <= values <= keyLE 
     * @param keyGE - The lower inclusive bound
     * @param keyLE - The upper inclusive bound
     * @param bKeysOnly return only keys, or full tuples?
     * @return String - The EPR of the result set containing the result.
     * @throws IndexException - when IOException is received.
     */
    public String getGEandLE(final Object keyGE, final Object keyLE, final boolean bKeysOnly) throws IndexException {
    	RSLocator locator = null;

    	try {
    		int maxHits = 1000;
    		PropertyElementBase[] properties = {new PropertyElementForwardIndex(maxHits)};
    		final RSXMLWriter rsWriter =  RSXMLWriter.getRSXMLWriter(properties);
    		locator = rsWriter.getRSLocator(new RSResourceWSRFType());

    		(new Thread() {
    			public void run() {
    				logger.debug(" >>> getGEandLE for keyGE "+ keyGE + " keyLE " + keyLE);
    				LinkedList valueList = null;
    				Tuple tuple = new Tuple(null, null);
    				TupleBrowser browser;
    				Object nextKey;
    				try {
    					browser = tree.browse(keyGE);
    					while (browser.getNext(tuple)) {
    						if (comparator.compare(tuple.getKey(), keyLE) <= 0) {
    							nextKey = tuple.getKey();
    							if (bKeysOnly)
    								addToResults(rsWriter, nextKey);
    							else {
	    		                    valueList = (LinkedList)tuple.getValue();
	    		                    ListIterator it = valueList.listIterator();
	    		                    while (it.hasNext()) {
	    		                        Object value = it.next();
	    		                        addToResults(rsWriter, nextKey, value);
	    		                    }
    							}
    		                }
    		                else {
    		                    break;
    		                }
    					}
    					rsWriter.close();
    				} 
    				catch(Exception e){
    					logger.error(" *** getGEandLE for keyGE and keyLE: "+ keyGE 
    	                         + " / " + keyLE + " message: " + e.getMessage());
    					try {
    						rsWriter.close();
    					}
    					catch (Exception e1) { }
    				}
    				logger.debug(" <<< getGEandLE");
    			}
    		}).start();
    	} catch (Exception e) {
    		logger.error(e);
    	}
    	return locator.getLocator();
    }
    
    /** 
     * Gets keyGE <= values < keyLT 
     * @param keyGE - The lower inclusive bound key
     * @param keyLT - The upper exclusive bound key
     * @param bKeysOnly return only keys, or full tuples?
     * @return String - The EPR of the result set containing the result.
     * @throws IndexException - when IOException is received.
     */
    public String getGEandLT(final Object keyGE, final Object keyLT, final boolean bKeysOnly) throws IndexException {
    	RSLocator locator = null;

    	try {
    		int maxHits = 1000;
    		PropertyElementBase[] properties = {new PropertyElementForwardIndex(maxHits)};
    		final RSXMLWriter rsWriter =  RSXMLWriter.getRSXMLWriter(properties);
    		locator = rsWriter.getRSLocator(new RSResourceWSRFType());

    		(new Thread() {
    			public void run() {
    				logger.debug(" >>> getGEandLT for keyGE "+ keyGE + " keyLT " + keyLT);
    				LinkedList valueList = null;
    				Tuple tuple = new Tuple(null, null);
    				TupleBrowser browser;
    				Object nextKey;
    				try {
    					browser = tree.browse(keyGE);
    					while (browser.getNext(tuple)) {
    						if (comparator.compare(tuple.getKey(), keyLT) < 0) {
    							nextKey = tuple.getKey();
    							if (bKeysOnly)
    								addToResults(rsWriter, nextKey);
    							else {
	    		                    valueList = (LinkedList)tuple.getValue();
	    		                    ListIterator it = valueList.listIterator();
	    		                    while (it.hasNext()) {
	    		                        Object value = it.next();
	    		                        addToResults(rsWriter, nextKey, value);
	    		                    }
    							}
    		                }
    		                else {
    		                    break;
    		                }
    					}
    					rsWriter.close();
    				} 
    				catch(Exception e){
    					logger.error(" *** getGEandLT for keyGE and keyLT: "
    	                         + keyGE + " / " + keyLT + " message: " + e.getMessage());
    					try {
    						rsWriter.close();
    					}
    					catch (Exception e1) { }
    				}
    				logger.debug(" <<< getGEandLT");
    			}
    		}).start();
    	} catch (Exception e) {
    		logger.error(e);
    	}
    	return locator.getLocator();
    }
    
    /** 
     * Gets keyGT < values <= keyLE 
     * @param keyGT - The lower exclusive bound
     * @param keyLE - The upper inclusive bound
     * @param bKeysOnly return only keys, or full tuples?
     * @return String - The EPR of the result set containing the result.
     * @throws IndexException - when IOException is received.
     */
    public String getGTandLE(final Object keyGT, final Object keyLE, final boolean bKeysOnly) throws IndexException {
    	RSLocator locator = null;

    	try {
    		int maxHits = 1000;
    		PropertyElementBase[] properties = {new PropertyElementForwardIndex(maxHits)};
    		final RSXMLWriter rsWriter =  RSXMLWriter.getRSXMLWriter(properties);
    		locator = rsWriter.getRSLocator(new RSResourceWSRFType());

    		(new Thread() {
    			public void run() {
    				logger.debug(" >>> getGTandLE for keyGT "+ keyGT + " keyLE " + keyLE);
    				LinkedList valueList = null;
    				Tuple tuple = new Tuple(null, null);
    				TupleBrowser browser;
    				Object nextKey;
    				try {
    					browser = tree.browse(keyGT);
    					while (browser.getNext(tuple)) {
    						if (comparator.compare(tuple.getKey(), keyGT) <= 0) {
    		                    continue;
    		                }
    		                if (comparator.compare(tuple.getKey(), keyLE) <= 0) {
    		                    nextKey = tuple.getKey();
    		                    if (bKeysOnly)
    		                    	addToResults(rsWriter, nextKey);
    		                    else {
	    		                    valueList = (LinkedList)tuple.getValue();
	    		                    ListIterator it = valueList.listIterator();
	    		                    while (it.hasNext()) {
	    		                        Object value = it.next();
	    		                        addToResults(rsWriter, nextKey, value);
	    		                    }
    		                    }
    		                } else {
    		                    break;
    		                }
    					}
    					rsWriter.close();
    				} 
    				catch(Exception e){
    					logger.error(" *** getGTandLE for keyGT and keyLE: "
    	                         + keyGT + " / " + keyLE + " message: " + e.getMessage());
    					try {
    						rsWriter.close();
    					}
    					catch (Exception e1) { }
    				}
    				logger.debug(" <<< getGTandLE");
    			}
    		}).start();
    	} catch (Exception e) {
    		logger.error(e);
    	}
    	return locator.getLocator();
    }
    
    /** 
     * Gets keyGT < values < keyLT 
     * @param keyGT - The lower exclusive bound key
     * @param keyLT - The upper exclusive bound key
     * @param bKeysOnly return only keys, or full tuples?
     * @return String - The EPR of the result set containing the result.
     * @throws IndexException - when IOException is received.
     */
    public String getGTandLT(final Object keyGT, final Object keyLT, final boolean bKeysOnly) throws IndexException {
    	RSLocator locator = null;

    	try {
    		int maxHits = 1000;
    		PropertyElementBase[] properties = {new PropertyElementForwardIndex(maxHits)};
    		final RSXMLWriter rsWriter =  RSXMLWriter.getRSXMLWriter(properties);
    		locator = rsWriter.getRSLocator(new RSResourceWSRFType());

    		(new Thread() {
    			public void run() {
    				logger.debug(" >>> getGTandLT for keyGT "+ keyGT + " keyLT " + keyLT);
    				LinkedList valueList = null;
    				Tuple tuple = new Tuple(null, null);
    				TupleBrowser browser;
    				Object nextKey;
    				try {
    					browser = tree.browse(keyGT);
    					while (browser.getNext(tuple)) {
    						if (comparator.compare(tuple.getKey(), keyGT) <= 0) {
    		                    continue;
    		                }
    		                if (comparator.compare(tuple.getKey(), keyLT) < 0) {
    		                    nextKey = tuple.getKey();
    		                    if (bKeysOnly)
    		                    	addToResults(rsWriter, nextKey);
    		                    else {
    		                    	valueList = (LinkedList)tuple.getValue();
	    		                    ListIterator it = valueList.listIterator();
	    		                    while (it.hasNext()) {
	    		                        Object value = it.next();
	    		                        addToResults(rsWriter, nextKey, value);
	    		                    }
    		                    }
    		                } else {
    		                    break;
    		                }
    					}
    					rsWriter.close();
    				} 
    				catch(Exception e){
    					logger.error(" *** getGTandLT for keyGT and keyLT: "
    	                         + keyGT + " / " + keyLT + " message: " + e.getMessage());
    					try {
    						rsWriter.close();
    					}
    					catch (Exception e1) { }
    				}
    				logger.debug(" >>> getGTandLT");
    			}
    		}).start();
    	} catch (Exception e) {
    		logger.error(e);
    	}
    	return locator.getLocator();
    }
    
    /** 
     * Gets all.
     * @param bKeysOnly return only keys, or full tuples?
     * @return String - The EPR of the result set containing the result.
     * @throws IndexException - when IOException is received.
     */
    public String getAllDec(final boolean bKeysOnly) throws IndexException {
    	RSLocator locator = null;

    	try {
    		int maxHits = 1000;
    		PropertyElementBase[] properties = {new PropertyElementForwardIndex(maxHits)};
    		final RSXMLWriter rsWriter =  RSXMLWriter.getRSXMLWriter(properties);
    		locator = rsWriter.getRSLocator(new RSResourceWSRFType());

    		(new Thread() {
    			public void run() {
    				logger.debug(" >>> getAllDec");
    				LinkedList valueList = null;
    				Tuple tuple = new Tuple(null, null);
    				TupleBrowser browser;
    				Object nextKey;
    				try {
    					browser = tree.browse(null);
    					while (browser.getPrevious(tuple)) {
		                    nextKey = tuple.getKey();
		                    if (bKeysOnly)
		                    	addToResults(rsWriter, nextKey);
		                    else {
		                    	valueList = (LinkedList)tuple.getValue();
			                    ListIterator it = valueList.listIterator();
			                    while (it.hasNext()) {
			                        Object value = it.next();
			                        addToResults(rsWriter, nextKey, value);
			                    }
		                    }
    					}
    					rsWriter.close();
    				} 
    				catch(Exception e){
    					logger.error(" *** getAllDec  message: " + e.getMessage());
    					try {
    						rsWriter.close();
    					}
    					catch (Exception e1) { }
    				}
    				logger.debug(" >>> getAllDec");
    			}
    		}).start();
    	} catch (Exception e) {
    		logger.error(e);
    	}
    	return locator.getLocator();
    }
    
    /** 
     * Gets all.
     * @param bKeysOnly return only keys, or full tuples?
     * @return String - The EPR of the result set containing the result.
     * @throws IndexException - when IOException is received.
     */
    public String getAllInc(final boolean bKeysOnly) throws IndexException {
    	RSLocator locator = null;

    	try {
    		int maxHits = 1000;
    		PropertyElementBase[] properties = {new PropertyElementForwardIndex(maxHits)};
    		final RSXMLWriter rsWriter =  RSXMLWriter.getRSXMLWriter(properties);
    		locator = rsWriter.getRSLocator(new RSResourceWSRFType());

    		(new Thread() {
    			public void run() {
    				logger.debug(" >>> getAllInc");
    				LinkedList valueList = null;
    				Tuple tuple = new Tuple(null, null);
    				TupleBrowser browser;
    				Object nextKey;
    				try {
    					browser = tree.browse();
    					while (browser.getNext(tuple)) {
		                    nextKey = tuple.getKey();
		                    if (bKeysOnly)
		                    	addToResults(rsWriter, nextKey);
		                    else {
		                    	valueList = (LinkedList)tuple.getValue();
			                    ListIterator it = valueList.listIterator();
			                    while (it.hasNext()) {
			                        Object value = it.next();
			                        addToResults(rsWriter, nextKey, value);
			                    }
		                    }
    					}
    					rsWriter.close();
    				}
    				catch(Exception e){
    					logger.error(" *** getAllInc  message: " + e.getMessage());
    					try {
    						rsWriter.close();
    					}
    					catch (Exception e1) { }
    				}
    				logger.debug(" >>> getAllInc");
    			}
    		}).start();
    	} catch (Exception e) {
    		logger.error(e);
    	}
    	return locator.getLocator();
    }
    
    /**
     * THIS METHOD IS NEVER CALLED IN FORWARD INDEX.
     * Merge deletion is used to update the index with documents to be deleted.
     * @param deletionFile - the file with the content to delete .
     * @throws IndexException - in case of failure.
     *
     */
    public synchronized int mergeDeletion(File deletionFile) throws IndexException {
        try {
            logger.debug(" >>> mergeDeletion");
            logger.debug(" <<< mergeDeletion");
            return 0;
        }
        catch (Exception ex) {
            logger.error(" *** mergeDeletion error  message: " + ex.getMessage());
            return 1;
        }
    }

    /**
     * Receives the file with rowsets that is to be merged into the index.
     * @param inputIndex - The file with rowsets that is merged into the index.
     * @return - number of added documents.
     * @throws IndexException - in case of error.
     */
    public synchronized int mergeAddition(File inputIndex)
        throws IndexException {
        logger.debug(" >>> mergeAddition file:" + inputIndex + "DBT " + dbType);
        int index = 0;
        try {
            FileInputStream inputStream = new FileInputStream(inputIndex);
            int available = inputStream.available();
            ReadableByteChannel deltaChannel = inputStream.getChannel();
            if (available > 0) {
                ByteBuffer buffer = ByteBuffer.allocate(available);
                int bytesRead = deltaChannel.read(buffer);
                buffer.flip();
                byte[] c = new byte[buffer.capacity()];
                buffer.get(c);
                String rowSet = "<ROWSET>" + new String(c, "UTF-8") + "</ROWSET>";
                logger.debug("    rowset\n" 
                             + rowSet);

                // Parse
                UpdateXMLParser parser = new UpdateXMLParser();
                int numAddedDocs = parser.parse(rowSet);
                
                logger.debug(" <<< mergeAddition");
                return numAddedDocs;
            }
            else {
                logger.error(" *** mergeAddition empty file, available bytes: " + available);
                throw new IndexException(" Empty insert file received");
            }
        }
        catch(Exception ex) {
            logger.error(" *** mergeAddition error  message: " + ex.getMessage());
            return 0;
        }
    }

    /**
     * Adds a tuple to the index.
     * 
     * @param key the tuple's key
     * @param value the tuple's value
     */
    private void addTuple(String key, String value) {
    	try {
    		
    		logger.debug("=================================");
            logger.debug("jdbmWrapper.addTuple merging current index with: " + value);
            logger.debug("=================================");
            
	    	 // Convert the key from String to the correct type
	        if (this.dbType == DbTypeEnum.INT_TYPE) {
	            insertPairInDb(convertToInt(key), value);
	        }
	        else if (this.dbType == DbTypeEnum.FLOAT_TYPE) {
	            insertPairInDb(convertToFloat(key), value);
	        }
	        else if (this.dbType == DbTypeEnum.STRING_TYPE) {
	            insertPairInDb(convertToString(key), value);
	        }
	        else if (this.dbType == DbTypeEnum.DATE_TYPE) {
	            insertPairInDb(convertToDate(key), value);
	        }
    	} catch (Exception e) {
    		logger.error(" *** addTuple error  message: " + e.getMessage());
    	}
    }
    
    /**
     * Deletes a tuple from the index
     * 
     * @param key the tuple's key
     */
    private void deleteTuple(String key) {
    	try {
	    	if (this.dbType == DbTypeEnum.INT_TYPE) {
	            deletePairInDb(convertToInt(key));
	        }
	        else if (this.dbType == DbTypeEnum.FLOAT_TYPE) {
	            deletePairInDb(convertToFloat(key));
	        }
	        else if (this.dbType == DbTypeEnum.STRING_TYPE) {
	            deletePairInDb(convertToString(key));
	        }
	        else if (this.dbType == DbTypeEnum.DATE_TYPE) {
	            deletePairInDb(convertToDate(key));
	        }
    	} catch (Exception e) {
    		logger.error(" *** deleteTuple error  message: " + e.getMessage());
    	}
    }
    
    /** 
     * Converts the String to an int value
     * @param in - the object to convert to int
     * @return - the int value of the String input
     * @throws -IndexException in case of failure
     */
    private final int convertToInt(Object in) throws IndexException {
        logger.debug(" >>> convertToInt input value: " + in);
        try {
            // The input is an String 
            int res = new Integer(((String)(in))).intValue();
            // Integer res = new Integer(((String)(in)));
            logger.debug(" <<< convertToInt value: " + res);
            return res;
        }
        catch (Exception ex) {
            logger.error(" *** convertToInt error  message: " + ex.getMessage());
            throw new IndexException(" *** convertToInt error", ex);
        }
    }

    /** 
     * Converts the String to a float value
     * @param in - the object to convert to float
     * @return - the float value of the String input
     * @throws -IndexException in case of failure
     */
    private final float convertToFloat(Object in) throws IndexException {
        logger.debug(" >>> convertToFloat input value: " + in);
        try {
            // The input is an String 
            float res = new Float(((String)(in))).floatValue();
            logger.debug(" <<< convertToFloat float: " + res);
            return res;
        }
        catch (Exception ex) {
            logger.error(" *** convertToFloat error  message: " + ex.getMessage());
            throw new IndexException(" *** convertToFloat error", ex);
        }
    }

    /** 
     * Converts the String to a date value
     * @param in - the object to convert to date
     * @return - the date value of the String input
     * @throws -IndexException in case of failure
     */
    private final Date convertToDate(Object in) throws IndexException {
        logger.debug(" >>> convertToDate input value: " + in);
        try {
            // The input is an String 
        	DateFormat df = new SimpleDateFormat("yyyy-MM-dd");            
            CustomDate res = new CustomDate((String) in, df);
            logger.debug(" <<< convertToDate date: " + res);
            return res;
        }
        catch (Exception ex) {
            logger.error(" *** convertToDate error  message: " + ex.getMessage());
            throw new IndexException(" *** convertToDate error", ex);
        }
    }

    /** 
     * Converts the Object to a string value
     * @param in - the object to convert to date
     * @return - the date value of the String input
     * @throws -IndexException in case of failure
     */
    private final String convertToString(Object in) throws IndexException {
        logger.debug(" >>> convertToString input value: " + in);
        try {
            // The input is an String
            // May be a too complicated conversion since it is already a String
            // but this method will at least throw exception if it fails
            String  res = new String((String)(in));
            logger.debug(" <<< convertToString string: " + res);
            return res;
        }
        catch (Exception ex) {
            logger.error(" *** convertToString error  message: " + ex.getMessage());
            throw new IndexException(" *** convertToString error", ex);
        }
    }

    /**
     * Updates the index.
     * NOT USED IN THE FORWARDINDEX
     * @throws IndexException - in case of error
     */
    public synchronized void updateIndex()
        throws IndexException {
        logger.debug(" >>> updateIndex");
        logger.debug(" <<< updateIndex");
        try {
            ;
        }
        catch(Exception ex) {
            logger.error(" *** updateIndex error  message: " + ex.getMessage());
            ;
        }
    }

    /**
     * Prints out the values in the iterator.
     * @param result - the resultSet to print.
     */
    private void printRs(String result) {
        // The result in the vector is a repetition of "key value"
        logger.info("Result set: " + result);
    }

    /**
     * Prints out the values in the iterator.
     * @param result - the resultSet to print.
     */
    private void printList(LinkedList result) {
        // The result in the vector is a repetition of "key value"
        int hits = 0;
        String xmlResult = "<forwardIndexResult>";
        ListIterator<Tuple> it = result.listIterator();
        while (it.hasNext()) {
            Tuple tuple = it.next();
            hits++;
            xmlResult += "<tuple>";
            xmlResult += "<key>";
            xmlResult += tuple.getKey();
            xmlResult += "</key>";
            xmlResult += "<value>";
            xmlResult += tuple.getValue();
            xmlResult += "</value>";
            xmlResult += "</tuple>";
        }
        xmlResult = xmlResult + "</forwardIndexResult>";
        logger.info(xmlResult);
    }

    /**
     * Test main.
     * @param args - the array of arguments.
     */
    public static void main(String[] args) {
	String res;
        boolean printRes=true;
        //        WriteResults writeResult = new WriteResults();
        DbTypeEnum dbtAll[] =DbTypeEnum.values();
        try {
            int startTime = (int) Calendar.getInstance().getTimeInMillis();
            logger.info(" Create Object");
            DbTypeEnum dbt = DbTypeEnum.INT_TYPE;
            JdbmWrapper db = new JdbmWrapper(".","namedb4","nametree4",dbt);
            db.insertPairInDb(40001, 21000);
            logger.info("Insert second 40001/21101");
            db.insertPairInDb(40001, 21101);
            logger.info("GetEQ");
            res=db.getEQ(40001, false);
            //            writeResult.write(res,"ForwardIndexLookup",true);
            db.printRs(res);;
            db.insertPairInDb(40001, 21101);
            res=db.getEQ(40001, false);
            db.printRs(res);
            //            db.deletePairInDb(40000);
            db.deletePairInDb(40001);
            db.insertPairInDb(1,1000);
            db.insertPairInDb(100001,1000);
            int key1=0;
            for (int i = 10;i < 10000; i++) {
                db.insertPairInDb(i, i+1000);
                res=db.getEQ(i, false);
                if (printRes)
                    db.printRs(res);
                res=db.getGE(i, false);
                if (printRes)
                    db.printRs(res);
                res=db.getGT(i, false);
                if (printRes)
                    db.printRs(res);
                res=db.getLE(i, false);
                if (printRes)
                    db.printRs(res);
                res=db.getLT(i, false);
                if (printRes)
                    db.printRs(res);
                res=db.getGEandLE(key1,i, false);
                if (printRes)
                    db.printRs(res);
                res=db.getGEandLT(key1,i, false);
                if (printRes)
                    db.printRs(res);
                res=db.getGTandLE(key1,i, false);
                if (printRes)
                    db.printRs(res);
                res=db.getGTandLT(key1,i, false);
                if (printRes)
                    db.printRs(res);
            }

           long usedTime = (long) Calendar.getInstance().getTimeInMillis()-startTime;
           logger.info(" Used time: millis: " + usedTime);
        }
        catch (Exception e) {
            logger.error(e);
        }
    }
    
    
    
    
    private enum ParserState { IDLE, IN_INSERT, IN_DELETE, IN_TUPLE, IN_KEY, IN_VALUE };
 
    /**
     * Class that handles the SAX parsing of updates to the index.
     * 
     * @author Spyros Boutsis, UoA
     */
    class UpdateXMLParser extends DefaultHandler {
    	
    	private static final String INSERT_ELEMENT = "INSERT";
    	private static final String DELETE_ELEMENT = "DELETE";
    	private static final String TUPLE_ELEMENT = "TUPLE";
    	private static final String KEY_ELEMENT = "KEY";
    	private static final String VALUE_ELEMENT = "VALUE";
    	
    	private ParserState currState;
    	private String currKey;
    	private String currValue;
    	private boolean bDeletion;
    	private Exception e;
    	private int numDocsAdded;
    	
    	public UpdateXMLParser() {
    		currState = ParserState.IDLE;
    		currKey = null;
    		currValue = null;
    		bDeletion = false;
    		numDocsAdded = 0;
    		e = null;
    	}
    	
    	public int parse(String updateData) throws Exception {
    		XMLReader xr = XMLReaderFactory.createXMLReader();
    		xr.setContentHandler(this);
    		xr.setErrorHandler(this);
    		xr.parse(new InputSource(new StringReader(updateData)));
    		if (e != null)
    			throw e;
    		return numDocsAdded;
    	}
    	
        public void startElement (String uri, String name, String qName, Attributes atts) {
        	boolean bError = false;
        	
        	if (qName.equalsIgnoreCase(INSERT_ELEMENT)) {
        		if (currState.equals(ParserState.IDLE)) {
        			currState = ParserState.IN_INSERT;
        			bDeletion = false;
        		}
        		else
        			bError = true;
        	}
        	else if (qName.equalsIgnoreCase(DELETE_ELEMENT)) {
        		if (currState.equals(ParserState.IDLE)) {
        			currState = ParserState.IN_DELETE;
        			bDeletion = true;
        		}
        		else
        			bError = true;
        	}
        	else if (qName.equalsIgnoreCase(TUPLE_ELEMENT)) {
        		if (currState.equals(ParserState.IN_INSERT) || currState.equals(ParserState.IN_DELETE))
        			currState = ParserState.IN_TUPLE;
        		else
        			bError = true;
        	}
        	else if (qName.equalsIgnoreCase(KEY_ELEMENT)) {
        		if (currState.equals(ParserState.IN_TUPLE))
        			currState = ParserState.IN_KEY;
        		else
        			bError = true;
        	}
        	else if (qName.equalsIgnoreCase(VALUE_ELEMENT)) {
        		if (currState.equals(ParserState.IN_TUPLE))
        			currState = ParserState.IN_VALUE;
        		else
        			bError = true;
        	}
        	
        	if (bError)
        		e = new Exception("Invalid XML in rowset data");
        }

        public void endElement (String uri, String name, String qName) {
        	boolean bError = false;
        	
        	if (qName.equalsIgnoreCase(INSERT_ELEMENT)) {
        		if (currState.equals(ParserState.IN_INSERT)) {
        			currState = ParserState.IDLE;
        		}
        		else
        			bError = true;
        	}
        	else if (qName.equalsIgnoreCase(DELETE_ELEMENT)) {
        		if (currState.equals(ParserState.IN_DELETE)) {
        			currState = ParserState.IDLE;
        		}
        		else
        			bError = true;
        	}
        	else if (qName.equalsIgnoreCase(TUPLE_ELEMENT)) {
        		if (currState.equals(ParserState.IN_TUPLE)) {
        			if (bDeletion) {	/* Deletion */
        				if (currKey != null) {
        					deleteTuple(currKey);
        					numDocsAdded--;
        				}
        				currState = ParserState.IN_DELETE;
        			}
        			else {	/* Insertion */
        				if (currKey!=null && currValue!=null) {
        					addTuple(currKey, currValue);
        					numDocsAdded++;
        				}
        				currState = ParserState.IN_INSERT;
        			}
        			
        			currKey = null;
        			currValue = null;
        		}
        		else
        			bError = true;
        	}
        	else if (qName.equalsIgnoreCase(KEY_ELEMENT)) {
        		if (currState.equals(ParserState.IN_KEY))
        			currState = ParserState.IN_TUPLE;
        		else
        			bError = true;
        	}
        	else if (qName.equalsIgnoreCase(VALUE_ELEMENT)) {
        		if (currState.equals(ParserState.IN_VALUE))
        			currState = ParserState.IN_TUPLE;
        		else
        			bError = true;
        	}
        	
        	if (bError)
        		e = new Exception("Invalid XML in rowset data");
        }
        
        public void characters (char ch[], int start, int length) {
        	if (currState == ParserState.IN_KEY) {
        		String s = new String(ch, start, length);
        		if (currKey == null)
        			currKey = s;
        		else
        			currKey = currKey + s; 
        	}
        	else if (currState == ParserState.IN_VALUE) {
        		String s = new String(ch, start, length);
        		if (currValue == null)
        			currValue = s;
        		else
        			currValue = currValue + s; 
        	}
        }
        
        public void error(SAXParseException e1) {
        	e = e1;
        }
        
        public void fatalError(SAXParseException e1) {
        	e = e1;
        }
    }
}
