package org.gcube.indexmanagement.featureindexlibrary.commons;

import org.apache.log4j.Logger;

/**
 * Distance computations
 * 
 * @author UoA
 */
public class DistanceCalculation {
	/**
	 * The Logger used by the class
	 */
	private static Logger log = Logger.getLogger(DistanceCalculation.class);
	
	/**
	 * Distance function
	 * 
	 * @param vector1 the first vector
	 * @param vector2 the second vector
	 * @param threshold The threshold
	 * @param distanceMeasure The distance measure to use
	 * @return The distance measure
	 * @throws Exception An error
	 */
	public static float distance(float []vector1,float []vector2,float threshold,FIEnums.DistanceTypes distanceMeasure) throws Exception{
//		try{
			if(distanceMeasure.compareTo(FIEnums.DistanceTypes.Eukledian)==0){
				return distanceEukledian(vector1,vector2,threshold);
			}
			else if(distanceMeasure.compareTo(FIEnums.DistanceTypes.Manhatan)==0){
				return distanceManhattan(vector1,vector2,threshold);
			}
			else{
				log.error("Could not map enum to distance algo "+distanceMeasure.toString()+". Throwing Exception");
				throw new Exception("Could not map enum to distance algo "+distanceMeasure.toString());
			}
//		}catch(Exception e){
//			return Float.MAX_VALUE;
//		}
	}
	
	/**
	 * Distance function
	 * 
	 * @param vector1 the first vector
	 * @param vector2 the second vector
	 * @param threshold The threshold
	 * @param distanceMeasure The distance measure to use
	 * @param weights weight per dimention
	 * @return The distance measure
	 * @throws Exception An error
	 */
	public static float distance(float []vector1,float []vector2,float threshold,FIEnums.DistanceTypes distanceMeasure,float []weights) throws Exception{
//		try{
			if(distanceMeasure.compareTo(FIEnums.DistanceTypes.Eukledian)==0){
				return distanceEukledian(vector1,vector2,threshold,weights);
			}
			else if(distanceMeasure.compareTo(FIEnums.DistanceTypes.Manhatan)==0){
				return distanceManhattan(vector1,vector2,threshold,weights);
			}
			else{
				log.error("Could not map enum to distance algo "+distanceMeasure.toString()+". Throwing Exception");
				throw new Exception("Could not map enum to distance algo "+distanceMeasure.toString());
			}
//		}catch(Exception e){
//			return Float.MAX_VALUE;
//		}
	}
	
	/**
	 * Eukledian distance
	 * 
	 * @param vector1 the first vector
	 * @param vector2 the second vector
	 * @param threshold The threshold
	 * @return The distance measure
	 */
	private static float distanceEukledian(float []vector1,float []vector2,float threshold){
		try{
			float ret=0;
			for(int i=0;i<vector1.length;i+=1){
				float point=(vector1[i]-vector2[i]);
				ret+=(point*point);
				if(ret>threshold) return Float.MAX_VALUE; 
			}
			return ret;
		}catch(Exception e){
			log.error("Could not compute Eukledian distance on vector with lengths "+vector1.length+" and "+vector2.length+" returning max value");
			return Float.MAX_VALUE;
		}
	}

	/**
	 * Eukledian distance
	 * 
	 * @param vector1 the first vector
	 * @param vector2 the second vector
	 * @param threshold The threshold
	 * @param weights The weights per dimention
	 * @return The distance measure
	 */
	private static float distanceEukledian(float []vector1,float []vector2,float threshold,float []weights){
		try{
			float ret=0;
			for(int i=0;i<vector1.length;i+=1){
				float point=(vector1[i]-vector2[i]);
				ret+=(weights[i]*point*point);
				if(ret>threshold) return Float.MAX_VALUE; 
			}
			return ret;
		}catch(Exception e){
			log.error("Could not compute Eukledian distance on vector with lengths "+vector1.length+" and "+vector2.length+" returning max value");
			return Float.MAX_VALUE;
		}
	}

	/**
	 * Manhattan distance
	 * 
	 * @param vector1 the first vector
	 * @param vector2 the second vector
	 * @param threshold The threshold
	 * @return The distance measure
	 */
	private static float distanceManhattan(float []vector1,float []vector2,float threshold){
		try{
			float ret=0;
			for(int i=0;i<vector1.length;i+=1){
				float point=Math.abs(vector1[i]-vector2[i]);
				ret+=point;
				if(ret>threshold) return Float.MAX_VALUE; 
			}
			return ret;
		}catch(Exception e){
			log.error("Could not compute Manhattan distance on vector with lengths "+vector1.length+" and "+vector2.length+" returning max value");
			return Float.MAX_VALUE;
		}
	}

	/**
	 * Manhattan distance
	 * 
	 * @param vector1 the first vector
	 * @param vector2 the second vector
	 * @param threshold The threshold
	 * @param weights The weights per dimention
	 * @return The distance measure
	 */
	private static float distanceManhattan(float []vector1,float []vector2,float threshold,float []weights){
		try{
			float ret=0;
			for(int i=0;i<vector1.length;i+=1){
				float point=Math.abs(vector1[i]-vector2[i]);
				ret+=(weights[i]*point);
				if(ret>threshold) return Float.MAX_VALUE; 
			}
			return ret;
		}catch(Exception e){
			log.error("Could not compute Manhattan distance on vector with lengths "+vector1.length+" and "+vector2.length+" returning max value");
			return Float.MAX_VALUE;
		}
	}
}
