package org.gcube.indexmanagement.bdbwrapper;

import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.Vector;

import org.gcube.common.core.utils.logging.GCUBELog;
import org.gcube.indexmanagement.bdbwrapper.BDBGcqlQueryContainer.SingleTerm;
import org.gcube.indexmanagement.common.IndexType;
import org.gcube.indexmanagement.gcqlwrapper.GcqlProcessor;
import org.gcube.indexmanagement.gcqlwrapper.GcqlQueryContainer;
import org.gcube.indexmanagement.resourceregistry.RRadaptor;

import search.library.util.cql.query.tree.GCQLAndNode;
import search.library.util.cql.query.tree.GCQLNode;
import search.library.util.cql.query.tree.GCQLNotNode;
import search.library.util.cql.query.tree.GCQLOrNode;
import search.library.util.cql.query.tree.GCQLProjectNode;
import search.library.util.cql.query.tree.GCQLQueryTreeManager;
import search.library.util.cql.query.tree.GCQLTermNode;
import search.library.util.cql.query.tree.ModifierSet;

public class BDBGcqlProcessor extends GcqlProcessor{
	
	private static final String DISTINCT = "distinct";

	private LinkedHashMap<String, String> projectedFields = new LinkedHashMap<String, String>();
	
	/**
	 * indicates if this query has a project distinct
	 */
	private boolean distinct = false;
	
	static GCUBELog logger = new GCUBELog(BDBGcqlProcessor.class);
	
	public static void main(String[] args) {
		
		ArrayList<String> presentable = new ArrayList<String>();
		ArrayList<String> searchable = new ArrayList<String>();
		presentable.add("title");
		presentable.add("author");
		presentable.add("year");
		presentable.add("code");
		
		searchable.add("title");
		searchable.add("author");
		searchable.add("year");
		searchable.add("description");
		searchable.add("anotations");
		searchable.add("code");
		searchable.add("gDocCollectionID");
		searchable.add("gDocCollectionLang");
		
		String query = " (((((title within \"A F\") or (author within \"A F\")) not ((title within \"B D\") and (author within \"E Q\"))) and ((code exact \"124adb323456\") or (code exact \"68dc3245abe231\")) ) and ((gDocCollectionID == A) and (gDocCollectionLang == fr)))  project title author code";
		try{
			BDBGcqlQueryContainer newContainer = (BDBGcqlQueryContainer)(new BDBGcqlProcessor().processQuery(presentable, searchable, query, new RRadaptor("dummy"))); 
 			
 			for(ArrayList<BDBGcqlQueryContainer.SingleTerm> q : newContainer.getBdbQueries()) {
 				BDBQueryExecutor.parseQueryTestLocally(q, new LinkedHashMap<String, ArrayList<String>>(), new ArrayList<String>());
 			}
			
		}catch (Exception e) {
			e.printStackTrace();
		}

	}
	
	public GcqlQueryContainer processQuery(ArrayList<String> presentableFields, ArrayList<String> searchableFields, String gCQLQuery, RRadaptor adaptor) throws Exception {
		//store the presentable and searchable fields
		this.presentableFields = presentableFields;
		this.searchableFields =  searchableFields;
		//store the Resource Registry adaptor
		this.adaptor = adaptor;
		
		//use the gCQL parser to get a tree from the String
		GCQLNode head = GCQLQueryTreeManager.parseGCQLString(gCQLQuery);
		//the processNode will run recursively, will retrieve 
		//the projected fields and will generate an ArrayList of BDB queries
		ArrayList<ArrayList<BDBGcqlQueryContainer.SingleTerm>> bdbQueries = processNode(head, false);
		return new BDBGcqlQueryContainer(bdbQueries, this.projectedFields, this.distinct);
	}
	
	private ArrayList<ArrayList<BDBGcqlQueryContainer.SingleTerm>> processNode(GCQLNode node, boolean not) throws Exception{
		
		//cases for the possible node types
		if(node instanceof GCQLProjectNode)
			return processNode((GCQLProjectNode)node, not);
		if(node instanceof GCQLAndNode)
			return processNode((GCQLAndNode)node, not);
		if(node instanceof GCQLNotNode)
			return processNode((GCQLNotNode)node, not);
		if(node instanceof GCQLOrNode)
			return processNode((GCQLOrNode)node, not);
		if(node instanceof GCQLTermNode)
			return processNode((GCQLTermNode)node, not);
		
		throw new Exception("This node class is not supported: " + node.getClass().toString());
	}
	
	private ArrayList<ArrayList<BDBGcqlQueryContainer.SingleTerm>> processNode(GCQLProjectNode node, boolean not) throws Exception{
		//add all the projections in the projected fields
		Vector<ModifierSet> projections = node.getProjectIndexes();
		for(ModifierSet projection : projections)
		{
			//check if there is a distinct indication
			//Note that it doesn't matter which is the field 
			//where the indication is attached(and it is the 
			//only modifier allowed)
			if(projection.getModifiers().size() > 0)
				if(projection.getModifiers().get(0).getType().equalsIgnoreCase(DISTINCT))
					distinct = true;
			
			//check if this projection is the wildcard
			if(projection.getBase().equals(IndexType.WILDCARD)) {
				distinct = false;
				projectedFields.clear();
				return processNode(node.subtree, not);
			}
			
			//get the field label for this field id
			String fieldLabel = adaptor.getFieldNameById(projection.getBase());
			String projField = findPresentable(fieldLabel);
			if(projField == null)
			{
				logger.error("Not in presentable fields: " + fieldLabel + ", " + projection.getBase());
				continue;
			}
			projectedFields.put(projection.getBase(), projField);
		}
		//return the lucene query of the subtree
		return processNode(node.subtree, not);
	}
	
	private ArrayList<ArrayList<BDBGcqlQueryContainer.SingleTerm>> processNode(GCQLTermNode node, boolean not) throws Exception{
		
		ArrayList<BDBGcqlQueryContainer.SingleTerm> query1 = new ArrayList<BDBGcqlQueryContainer.SingleTerm>();
		ArrayList<BDBGcqlQueryContainer.SingleTerm> query2 = null;
		
		//examine the index
		boolean found = false;
		String index = null;
		
		//if the field is the collection or the language then we won't be provided the fieldId
		//also change the relation
		if(node.getIndex().equals(IndexType.COLLECTION_FIELD) || node.getIndex().equals(IndexType.LANGUAGE_FIELD)) {
			
			node.getRelation().setBase(BDBWrapper.EXACT.toString());
			index = node.getIndex();
			found = true;
			
		} else {
			String fieldLabel = adaptor.getFieldNameById(node.getIndex());
			for(String field: searchableFields)
			{
				//we found a searchable field for the specified index
				if(fieldLabel.equalsIgnoreCase(field))
				{
					index = field;
					found = true;
					break;
				}
			}
		}
		
		if(!found)
		{
			logger.error("This field is not detected in the searchable fields: " + node.getIndex());
			throw new Exception("This field is not detected in the searchable fields: " + node.getIndex());
		}
		
		//find the type of the relation
		found = false;
		for(String relation : BDBWrapper.SupportedRelations) {
			if(node.getRelation().getBase().equalsIgnoreCase(relation.toString())) {
				if(relation.equals(BDBWrapper.EXACT)) {
					
					String value = removeQuotes(node.getTerm());
					//if the query must NOT be reverted
					if(!not) {
						query1.add(new SingleTerm(index, BDBQueryExecutor.eq, value));
						logger.debug("Term node - exact: " + index + BDBQueryExecutor.eq + value);
					
					//if the query must be reverted	
					} else {
						//if we wanted all the values, the complement of this query 
						//will be the empty set
						if(value.equals(BDBQueryExecutor.wild)) {
							logger.debug("Term node - exact+not+wild = empty");
							return new ArrayList<ArrayList<SingleTerm>>();
						}
						//else add two new terms for not including the specified value
						query1.add(new SingleTerm(index, BDBQueryExecutor.gt, value));
						logger.debug("Term node - exact+not: " + index + BDBQueryExecutor.gt + value);
						query1.add(new SingleTerm(index, BDBQueryExecutor.lt, value));
						logger.debug("Term node - exact+not: " + index + BDBQueryExecutor.lt + value);
						
					}
					
				} else if(relation.equals(BDBWrapper.WITHIN)) {	
					//We will try to split on whitespaces. First term specifies the lower bound,
					//and is followed by the upper bound
					String[] values = splitTerms(node.getTerm());
					if(values.length != 2)
					{
						logger.error("The argument of relation within is not valid: " + node.getTerm());
						throw new Exception("The argument of relation within is not valid: " + node.getTerm());
					}
					
					//if the query must NOT be reverted
					if(!not) {
						//wild char means that there is no bound
						if(!values[0].equals(BDBQueryExecutor.wild)) {
							query1.add(new SingleTerm(index, BDBQueryExecutor.gt+BDBQueryExecutor.eq, values[0]));
							logger.debug("Term node - within: " + index + BDBQueryExecutor.gt+BDBQueryExecutor.eq + values[0]);
						}
						//wild char means that there is no bound
						if(!values[1].equals(BDBQueryExecutor.wild)) {
							query1.add(new SingleTerm(index, BDBQueryExecutor.lt+BDBQueryExecutor.eq, values[1]));
							logger.debug("Term node - within: " + index + BDBQueryExecutor.lt+BDBQueryExecutor.eq + values[1]);
						}
						//if both are wild then search for everything
						if(values[0].equals(BDBQueryExecutor.wild) 
								&& values[1].equals(BDBQueryExecutor.wild)) {
							query1.add(new SingleTerm(index, BDBQueryExecutor.eq, BDBQueryExecutor.wild));
							logger.debug("Term node - within: " + index + BDBQueryExecutor.eq + BDBQueryExecutor.wild);
						}
					
					//if the query must be reverted	
					} else {
						//IMPORTANT: when we have two bounds that are connected with AND, and we revert them,
						//we end up with two bounds that are connected with OR! e.g. 3 =< x =< 8 becomes 
						//x < 3 OR 8 < x
						//wild char means that there is no bound
						if(!values[0].equals(BDBQueryExecutor.wild)) {
							query1.add(new SingleTerm(index, BDBQueryExecutor.lt, values[0]));
							logger.debug("Term node - within+not: " + index + BDBQueryExecutor.lt + values[0]);
						}
						//wild char means that there is no bound
						if(!values[1].equals(BDBQueryExecutor.wild)) {
							query2 = new ArrayList<BDBGcqlQueryContainer.SingleTerm>();
							query2.add(new SingleTerm(index, BDBQueryExecutor.gt, values[1]));
							logger.debug("OR Term node - within+not: " + index + BDBQueryExecutor.gt + values[1]);
						}
						//if both are wild
						if(values[0].equals(BDBQueryExecutor.wild) 
								&& values[1].equals(BDBQueryExecutor.wild)) {
							//if we wanted all the values, the complement of this query 
							//will be the empty set
							logger.debug("Term node - within+not+wild= empty");
							return new ArrayList<ArrayList<SingleTerm>>();
						}
					}
					
				} else {
					logger.error("Bug! Should not reach this point. This relation seems not to be supported: " + relation);
					throw new Exception("Bug! Should not reach this point. This relation seems not to be supported: " + relation);
				}
				
				//we found the relation
				found = true;
				break;
			}
		}
		
		if(!found)
		{
			logger.error("This relation is not supported by this type of Index: " + node.getRelation().getBase());
			throw new Exception("This relation is not supported by this type of Index: " + node.getRelation().getBase());
		}
		
		ArrayList<ArrayList<SingleTerm>> result =  new ArrayList<ArrayList<SingleTerm>>();
		result.add(query1);
		if(query2 != null) 
			result.add(query2);
		
		return result;
		
	}
	
	private ArrayList<ArrayList<BDBGcqlQueryContainer.SingleTerm>> processNode(GCQLNotNode node, boolean not) throws Exception{
		
		ArrayList<ArrayList<BDBGcqlQueryContainer.SingleTerm>> leftQueries = processNode(node.left, not);		
		//revert the not flag for the right subtree
		ArrayList<ArrayList<BDBGcqlQueryContainer.SingleTerm>> rightQueries = processNode(node.right, !not);
		
		logger.debug("Not node - not: " + not);
		logNumberOfConditions(leftQueries);
		logNumberOfConditions(rightQueries);
		
		//if not flag is raised then the AND(-NOT) becomes OR
		if(not) {
			return concatenateQueries(leftQueries, rightQueries);
		//else the queries must be connected with AND	
		} else {
			return mergeQueries(leftQueries, rightQueries);
		}		
	}
	
	private ArrayList<ArrayList<BDBGcqlQueryContainer.SingleTerm>> processNode(GCQLAndNode node, boolean not) throws Exception{ 
		
		ArrayList<ArrayList<BDBGcqlQueryContainer.SingleTerm>> leftQueries = processNode(node.left, not);
		
		
		ArrayList<ArrayList<BDBGcqlQueryContainer.SingleTerm>> rightQueries = processNode(node.right, not);
		
		logger.debug("And node - not: " + not);
		logNumberOfConditions(leftQueries);
		logNumberOfConditions(rightQueries);
		
		//if not flag is raised then the AND becomes OR
		if(not) {
			return concatenateQueries(leftQueries, rightQueries);
		//else the queries must be connected with AND	
		} else {
			return mergeQueries(leftQueries, rightQueries);
		}
	}
	
	private ArrayList<ArrayList<BDBGcqlQueryContainer.SingleTerm>> processNode(GCQLOrNode node, boolean not) throws Exception{ 
		
		ArrayList<ArrayList<BDBGcqlQueryContainer.SingleTerm>> leftQueries = processNode(node.left, not);		
		ArrayList<ArrayList<BDBGcqlQueryContainer.SingleTerm>> rightQueries = processNode(node.right, not);
		
		logger.debug("Or node - not: " + not);
		logNumberOfConditions(leftQueries);
		logNumberOfConditions(rightQueries);
		
		//if not flag is raised then the OR becomes AND
		if(not) {
			return mergeQueries(leftQueries, rightQueries);
		//else the queries must be connected with OR	
		} else {
			return concatenateQueries(leftQueries, rightQueries);
		}
	}
	
	private ArrayList<ArrayList<SingleTerm>> mergeQueries(ArrayList<ArrayList<SingleTerm>> leftQueries,
			ArrayList<ArrayList<SingleTerm>> rightQueries) {
		
		//When the queries are connected with AND they are merged
		//This means that we must create a new query for each pair 
		//of queries
		ArrayList<ArrayList<SingleTerm>> result =  new ArrayList<ArrayList<SingleTerm>>();
		for(ArrayList<SingleTerm> leftQuery : leftQueries) {
			for(ArrayList<SingleTerm> rightQuery : rightQueries) {
				ArrayList<SingleTerm> newQuery = new ArrayList<BDBGcqlQueryContainer.SingleTerm>();
				
				//add all the conditions for the matched queries in a new query
				newQuery.addAll(leftQuery);
				newQuery.addAll(rightQuery);
				
				//add the new query to the result
				result.add(newQuery);
			}
		}
		
		return result;
		
	}

	private ArrayList<ArrayList<SingleTerm>> concatenateQueries(
			ArrayList<ArrayList<SingleTerm>> leftQueries,
			ArrayList<ArrayList<SingleTerm>> rightQueries) {
		
		//When the queries are connected with OR they are concatenated
		//This means that we must create a new list containing all the 
		//queries from the two subtrees
		ArrayList<ArrayList<SingleTerm>> result = leftQueries;
		result.addAll(rightQueries);
		
		return result;
	}
	
	private void logNumberOfConditions(ArrayList<ArrayList<BDBGcqlQueryContainer.SingleTerm>> conditions) {
		logger.debug("Output is an OR of " + conditions.size() + " conditions");
		for(ArrayList<SingleTerm> andConditions : conditions) {
			logger.debug("This condition has " + andConditions.size() + " terms");
		}
	}

}
