/**
 * 
 */
package org.gcube.indexmanagement.common;

import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.util.ArrayList;

import javax.xml.namespace.QName;

import org.gcube.common.core.informationsystem.notifier.ISNotifier.NotificationEvent;
import org.gcube.common.core.scope.GCUBEScope;
import org.gcube.indexmanagement.common.notifications.NotifierRequestQueue;
import org.gcube.indexmanagement.common.notifications.SubscribeToNotificationRequest;
import org.gcube.indexmanagement.common.notifications.UnsubscribeFromNotificationRequest;
import org.w3c.dom.Element;

/**
 * @author Spyros Boutsis, NKUA
 *
 */
public abstract class IndexUpdaterWSResource extends IndexWSResource {

	/** The list of standard resource properties defined by every index updater resource */
    public static final String RP_CONNECTION_ID = "ConnectionID";
    public static final String RP_INDEX_STATUS = "IndexStatus";
    public static final String RP_IS_UPDATED = "IsUpdated";
    public static final String RP_DELTA_FILE_SIZE = "DeltaFileSize";

    /** The list of possible updater status values */
    public static final String UPDATER_STATUS_EMPTY = "EMPTY";
    public static final String UPDATER_STATUS_UPDATING = "UPDATING";
    public static final String UPDATER_STATUS_FINISHED = "FINISHED";
    
    /** The name of the state change notification topic */
	private static final String STATE_CHANGE_TOPIC_NAME = "SharedStateChange";
	
	/** The state change notification topic QName*/
	protected QName sharedStateChangeTopicName;
	
	/** The namespace of the topic consumed by the updater */
	private String managementServiceNamespace;
	
	/**
	 * Initializes the index management resource.
	 * 
	 * @param namespace the namespace of the service that this resource belongs to
	 * @param managementServiceNamespace the namespace of the corresponding index management service
	 * @param indexID the indexID of this resource
	 * @param indexTypeName the index type name of this resource
	 * @param collectionID the list of collection IDs of this resource
	 * @throws Exception an error occured
	 */
    public void initialize(String namespace, String managementServiceNamespace, String indexID, String indexTypeName, String collectionID[]) throws Exception {
    	super.initialise(namespace, indexID, indexTypeName, collectionID);

    	this.managementServiceNamespace = managementServiceNamespace;
    	
    	/* Create and initialize the common updater resource properties */
    	createProperty(RP_CONNECTION_ID);
    	
    	createProperty(RP_INDEX_STATUS);
    	this.getResourcePropertySet().get(RP_INDEX_STATUS).clear();
    	this.getResourcePropertySet().get(RP_INDEX_STATUS).add(UPDATER_STATUS_EMPTY);
    	
    	createProperty(RP_IS_UPDATED);
    	this.getResourcePropertySet().get(RP_IS_UPDATED).clear();
    	this.getResourcePropertySet().get(RP_IS_UPDATED).add(new Boolean(false));
    	
    	createProperty(RP_DELTA_FILE_SIZE);
    	this.getResourcePropertySet().get(RP_DELTA_FILE_SIZE).clear();
    	this.getResourcePropertySet().get(RP_DELTA_FILE_SIZE).add(new Long(1500000l));
        
        /* Register as a consumer of the state change notification topic
		 * exposed by the management service
		 */
        this.sharedStateChangeTopicName = new QName(managementServiceNamespace + "/" + indexID, STATE_CHANGE_TOPIC_NAME);
        ArrayList<QName> topicSubscriptions = new ArrayList<QName>();
        topicSubscriptions.add(sharedStateChangeTopicName);
		GCUBEScope scope = this.getServiceContext().getScope();
		//ISNotifier notifier = GHNContext.getImplementation(ISNotifier.class);
		//notifier.registerToISNotification(topicSubscriptions, new ConsumerNotification(scope), this.getServiceContext(), scope);
		NotifierRequestQueue.getInstance().add(
				new SubscribeToNotificationRequest(
						topicSubscriptions, new ConsumerNotification(scope), this.getServiceContext(), scope
				)
		);
		logger.debug("Consumer subscribed for notification on topic: " + sharedStateChangeTopicName);
    }
    
    /**
	 * Invoked when a resource is being created from a serialized, previously saved state.
	 * 
	 * @param ois the input stream through which the state can be read
	 * @throws Exception an error occured during resource deserialization
	 */
	protected void onLoad(ObjectInputStream ois, boolean firstLoad) throws Exception {
		super.onLoad(ois, firstLoad);
		
		/* Create and initialize the common updater resource properties from the given input stream */
    	createProperty(RP_CONNECTION_ID);
    	int size = ois.readInt();
    	for (int i=0; i<size; i++)
    		this.getResourcePropertySet().get(RP_CONNECTION_ID).add((Integer) ois.readInt());
    	
    	String status = (String) ois.readObject();
    	createProperty(RP_INDEX_STATUS);
    	this.getResourcePropertySet().get(RP_INDEX_STATUS).clear();
    	this.getResourcePropertySet().get(RP_INDEX_STATUS).add(status);
    	
    	Boolean isUpdated = ois.readBoolean();
    	createProperty(RP_IS_UPDATED);
    	this.getResourcePropertySet().get(RP_IS_UPDATED).clear();
    	this.getResourcePropertySet().get(RP_IS_UPDATED).add(isUpdated);
    	
    	Long deltaFileSize = ois.readLong();
    	createProperty(RP_DELTA_FILE_SIZE);
    	this.getResourcePropertySet().get(RP_DELTA_FILE_SIZE).clear();
    	this.getResourcePropertySet().get(RP_DELTA_FILE_SIZE).add(deltaFileSize);
        
    	this.managementServiceNamespace = (String) ois.readObject();
    	
        /* Register as a consumer of the state change notification topic
		 * exposed by the management service
		 */
    	if (firstLoad) {
	        this.sharedStateChangeTopicName = new QName(this.managementServiceNamespace + "/" + this.getIndexID(), STATE_CHANGE_TOPIC_NAME);
	        ArrayList<QName> topicSubscriptions = new ArrayList<QName>();
	        topicSubscriptions.add(sharedStateChangeTopicName);
			GCUBEScope scope = GCUBEScope.getScope(this.getResourcePropertySet().getScope().get(0));
			NotifierRequestQueue.getInstance().add(
					new SubscribeToNotificationRequest(
							topicSubscriptions, new ConsumerNotification(scope), this.getServiceContext(), scope
					)
			);
			logger.debug("Consumer subscribed for notification on topic: " + sharedStateChangeTopicName);
    	}
	}
	
	/**
	 * Invoked when the state of the resource must be saved (resource serialization)
	 * 
	 * @param oos the output stream to write the resource state to
	 * @throws Exception an error occured during resource serialization
	 */
	protected void onStore(ObjectOutputStream oos) throws Exception {
		super.onStore(oos);
		
		int size = this.getResourcePropertySet().get(RP_CONNECTION_ID).size();
		oos.writeInt(size);
		for (int i=0; i<size; i++)
			oos.writeInt((Integer) this.getResourcePropertySet().get(RP_CONNECTION_ID).get(i));
		
		oos.writeObject((String) this.getResourcePropertySet().get(RP_INDEX_STATUS).get(0));
		
		oos.writeBoolean((Boolean) this.getResourcePropertySet().get(RP_IS_UPDATED).get(0));
		
		oos.writeLong((Long) this.getResourcePropertySet().get(RP_DELTA_FILE_SIZE).get(0));
		
		oos.writeObject(this.managementServiceNamespace);
	}
	
    /**
     * Getter method for the IndexStatus Resource Property
     * @return <code>String</code> the requested status indicator
     */
    public String getIndexStatus() {
        return (String) this.getResourcePropertySet().get(RP_INDEX_STATUS).get(0);
    }
    
    /**
     * Getter method for the DeltaFileSize Resource Property
     * @return <code>long</code> the requested DeltaFileSize value
     */
    public long getDeltaFileSize() {
        return ((Long) this.getResourcePropertySet().get(RP_DELTA_FILE_SIZE).get(0)).longValue();
    }

    /**
     * Getter method for the IsUpdated Resource Property
     * @return <code>boolean</code> the requested IsUpdated indicator
     */
    public boolean getIsUpdated() {
        return ((Boolean) this.getResourcePropertySet().get(RP_IS_UPDATED).get(0)).booleanValue();
    }
    
    /**
     * Adds a connection ID
     * @param connID - the connectionID to add
     */
    public void addConnectionID(int connID) {
    	this.getResourcePropertySet().get(RP_CONNECTION_ID).add(new Integer(connID));    	
    }

    /**
     * Getter method for the ConnectionID Resource Property
     * @return <code>String</code> the requested ConnectionID 
     */
    public String getConnectionID(){return (String)this.getResourcePropertySet().get(RP_CONNECTION_ID).get(0);}

    /**
     * Setter method for the ConnectionID Resource Property
     * @param ConnectionID <code>String[]</code> the new Connection identifier
     */
    public synchronized void setConnectionID(String connectionID) throws Exception { 
    	this.getResourcePropertySet().get(RP_CONNECTION_ID).clear();
    	this.getResourcePropertySet().get(RP_CONNECTION_ID).add(connectionID);
    }

    /**
     * Setter method for the DeltaFileSize Resource Property
     * @param mergeInterval <code>long</code> the new DeltaFileSize in bytes
     */
    public void setDeltaFileSize(long mergeInterval) {
    	this.getResourcePropertySet().get(RP_DELTA_FILE_SIZE).clear();
    	this.getResourcePropertySet().get(RP_DELTA_FILE_SIZE).add(new Long(mergeInterval));
    }
        
    /**
     * Setter method for the IsUpdated Resource Property
     * @param isUpdated <code>boolean</code> the updated flag
     */
    public void setIsUpdated(boolean isUpdated) {
    	this.getResourcePropertySet().get(RP_IS_UPDATED).clear();
    	this.getResourcePropertySet().get(RP_IS_UPDATED).add(new Boolean(isUpdated));
    }
    
    /**
     * Setter method for the IndexStatus Resource Property
     * @param indexStatus <code>String</code> the new status of the resource
     */
    public void setIndexStatus(String indexStatus) {
    	this.getResourcePropertySet().get(RP_INDEX_STATUS).clear();
    	this.getResourcePropertySet().get(RP_INDEX_STATUS).add(indexStatus);
    }
    
    /**
     * Returns the namespace of the management resource associated with this updater resource
     * @return the namespace
     */
    public String getManagementResourceNamespace() {
    	return this.managementServiceNamespace;
    }

    /*
     * (non-Javadoc)
     * @see org.gcube.indexmanagement.common.IndexWSResource#onResourceRemoval()
     */
    public void onResourceRemoval() {
		super.onResourceRemoval();
		
		try {
	        ArrayList<QName> topicSubscriptions = new ArrayList<QName>();
	        topicSubscriptions.add(sharedStateChangeTopicName);
	        //ISNotifier notifier = GHNContext.getImplementation(ISNotifier.class);
			//notifier.unregisterFromISNotification(topicSubscriptions, this.getServiceContext(),
			//		this.getServiceContext().getScope());
	        NotifierRequestQueue.getInstance().add(
					new UnsubscribeFromNotificationRequest(
							topicSubscriptions, this.getServiceContext(), 
							this.getServiceContext().getScope()
					)
			);
			logger.debug("Consumer unsubscribed from notification on topic: " + sharedStateChangeTopicName + " for index: " + this.getIndexID());			
		} catch(Exception e){
            logger.debug("Failed to unregister notification topics from the IS.", e);
		}
	}
	
	/**
	 * Invoked whenever a message is received for this consumer.
	 * @param message the received message
 	 */
	public abstract void onUpdaterNotificationReceived(Element message);
	
    /**
	 * Class that handles the consuming of received notifications
	 * @author Spyros Boutsis, NKUA
	 */
	public class ConsumerNotification extends IndexNotificationConsumer {
		public ConsumerNotification(GCUBEScope scope) { super(IndexUpdaterWSResource.this, scope); }
		
		protected void onNewNotification(NotificationEvent event){ 
			logger.debug("Index notification consumer received notification message.");
			Element message = null;
			try {
				message = (Element) event.getPayload().getMessageObject();
				onUpdaterNotificationReceived(message);
			}  catch (Exception e) {
				logger.debug("Error in onNotificationReceived.", e);
			}
		}
	}
}
