package org.gcube.indexmanagement.geo.shape;

import java.awt.geom.AffineTransform;
import java.awt.geom.Area;
import java.awt.geom.PathIterator;
import java.util.ArrayList;

import org.gcube.common.core.utils.logging.GCUBELog;

public class PolygonProcessing {
	
	/* Logger */
	static GCUBELog logger = new GCUBELog(PolygonProcessing.class);
	
	/**
	 * returns the intersection of two polygons a, b
	 * @param a - polygon a
	 * @param b - polygon b
	 * @return the intersection of a and b
	 */
	public static Polygon intersection(Polygon a, Polygon b) {
		Area areaA = new Area(transformPolygon2Polygon(a));
		Area areaB = new Area(transformPolygon2Polygon(b));
		areaA.intersect(areaB);
		return constructPolygonFromArea(areaA);
	}
	
	/**
	 * returns the union of two polygons a, b
	 * @param a - polygon a
	 * @param b - polygon b
	 * @return the union of a and b
	 */
	public static Polygon union(Polygon a, Polygon b) {
		Area areaA = new Area(transformPolygon2Polygon(a));
		Area areaB = new Area(transformPolygon2Polygon(b));
		areaA.add(areaB);
		return constructPolygonFromArea(areaA);
	}
	
	/**
	 * checks if polygon internal is contained inside the polygon external
	 * @param internal - the internal polygon
	 * @param external - the external polygon
	 * @return true if yes / false otherwise
	 */
	public static boolean isContained(Polygon internal, Polygon external) {
		Area areaExt = new Area(transformPolygon2Polygon(external));
		Area areaInt = new Area(transformPolygon2Polygon(internal));
		return isContained(areaInt, areaExt);
	}
	
	/**
	 * checks if a polygon a overlaps with a polygon b
	 * @param a - the a polygon
	 * @param b - the b polygon
	 * @return true if yes / false otherwise
	 */
	public static boolean overlaps(Polygon a, Polygon b) {
		if(intersection(a, b).getVertices().size() == 0)
			return false;
		else
			return true;		
	}
	
	private static java.awt.Polygon transformPolygon2Polygon(Polygon poly) {
		//construct a java.awt.Polygon 
		Point[] vertices = poly.getVertexArray();
		int npoints = vertices.length;
		int[] xpoints = new int[npoints];
		int[] ypoints = new int[npoints];
		for(int i=0; i<npoints; i++)
		{
			long x = vertices[i].getX();
			if(x > Integer.MAX_VALUE)
				logger.error("coordinate: " + x + ", is larger than the max int value..");
			xpoints[i] = (int) x;
			long y = vertices[i].getY();
			if(y > Integer.MAX_VALUE)
				logger.error("coordinate: " + y + ", is larger than the max int value..");
			ypoints[i] = (int) y;			
		}
		return new java.awt.Polygon(xpoints, ypoints, npoints);
	}
	
	private static Polygon constructPolygonFromArea(Area a) {
		PathIterator iter = a.getPathIterator(new AffineTransform());
		ArrayList<Point> vertices = new ArrayList<Point>();
		while(!iter.isDone())
		{
			float[] coords = new float[6];
			Point p = null;
			switch(iter.currentSegment(coords)) {
			//TODO: the polygons must be able to store double values
			case PathIterator.SEG_MOVETO:
				p = new Point((long)coords[0], (long)coords[1]);
				break;
			case PathIterator.SEG_LINETO:
				p = new Point((long)coords[0], (long)coords[1]);
				break;
			case PathIterator.SEG_CLOSE:
				break;
			default:
				logger.error("This case is not predicted here! This means that we have a bug!");	
			}
			if(p != null)
				vertices.add(p);
			iter.next();
		}
		if(vertices.size() == 0)
			return null;
		
		return new Polygon(vertices);
	}
	
	/**
	 * used only for testing - Std out will not be used by this class when called by the container
	 * @param args
	 */
	public static void main (String[] args) {
		Point[] verticesA = {new Point(2, 2), new Point(5, 4), 
				new Point(6, 4), new Point(10, 2), 
				new Point(6, 1), new Point(5, 1)};
		
		Point[] verticesB = {new Point(5, 5), new Point(6, 5), 
				new Point(6, 2), new Point(5, 2)};
		
		Point[] verticesC = {new Point(5, 5), new Point(6, 5), 
				new Point(6, 7), new Point(5, 7)};
		
		Polygon polyA = new Polygon(verticesA);
		Polygon polyB = new Polygon(verticesB);
		Polygon polyC = new Polygon(verticesC);
		
		Polygon result = intersection(polyA, polyB);
		System.out.println("intersection: " + result.toString());
		
		result = union(polyA, polyB);
		System.out.println("union: " + result.toString());
		
		if(isContained(polyB, polyA))
			System.out.println("Is contained");
		else
			System.out.println("Is NOT contained");
		
		if(isContained(polyA, polyB))
			System.out.println("Is contained");
		else
			System.out.println("Is NOT contained");
		
		if(overlaps(polyA, polyB))
			System.out.println("Overlaps True");
		else
			System.out.println("Overlaps False");
		
		if(overlaps(polyA, polyC))
			System.out.println("Overlaps True");
		else
			System.out.println("Overlaps False");
		
		if(overlaps(polyB, polyC))
			System.out.println("Overlaps True");
		else
			System.out.println("Overlaps False");
				
	}
	
	/**
	 * Tests if an area is contained to another area
	 * @param external the area that must contain the internal area
	 * @param internal the area that must be contained in the external area
	 * @return true if internal is contained in external / false otherwise
	 */
	private static boolean isContained(Area internal, Area external) {
		//first calculate the intersection of the two areas
		external.intersect(internal);
		//if the intersection is equal to the internal then 
		//the external contains the internal
		if(internal.equals(external))
			return true;
		else
			return false;
	}

}
