/*
 * State.java
 *
 * $Author: tsakas $
 * $Date: 2007/12/20 14:37:40 $
 * $Id: GeoIndexType.java,v 1.1 2007/12/20 14:37:40 tsakas Exp $
 *
 * <pre>
 *             Copyright (c) : 2006 Fast Search & Transfer ASA
 *                             ALL RIGHTS RESERVED
 * </pre>
 */

package org.gcube.indexmanagement.geo;

import java.util.HashMap;

import org.gcube.common.core.scope.GCUBEScope;
import org.gcube.common.core.utils.logging.GCUBELog;
import org.gcube.indexmanagement.common.IndexType;
import org.gcube.indexmanagement.common.XMLProfileParser;

/**
 * A Java object representation of the GeoIndexType
 */
public class GeoIndexType extends IndexType{

	/** logger */
	static GCUBELog logger = new GCUBELog(GeoIndexType.class);
	
    /** The fields of this GeoIndexType */
    public GeoIndexField[] fields;

    /** A map, which maps the names of fields to their index in the fieldsArray */
    private HashMap<String, Integer> fieldIndexMap;

    /** The number of fields of this GeoIndexType */
    public int nrOfFields;

    /** The accumulated size of all the fields of this GeoIndexType */
    public int size = 0;

    /** The parser used to parse the xml representation of the GeoIndexType */
    private XMLProfileParser parser = null;

    /**
     * Standard constructor which build a GeoIndexType object from a String
     * containing a GeoIndexType xml document
     * 
     * @param indexTypeName -
     *            The name to assign to the GeoIndexType
     * @param indexType -
     *            An xml representation of the indextype
     */
    public GeoIndexType(String indexTypeName, GCUBEScope scope) {
    	super(indexTypeName);
    	
    	try {
	    	String indexType = retrieveIndexTypeGenericResource(scope);
	        this.fieldIndexMap = new HashMap<String, Integer>();
	        if (parser == null) {
	            parser = new XMLProfileParser();
	        }
        
            parser.readString(indexType, null);
            readIndexType();
        } catch (Exception ex) {
            logger.error("Failed to initialize GeoIndexType.", ex);
        }
    }

    /**
     * The method which actually traverses the created DOM document in order to
     * fill the fields
     * 
     * @throws Exception
     */
    private void readIndexType() throws Exception {
        int position = 0;
        try {
            // Extract the fields
            parser.setRootNode("field-list");
            nrOfFields = parser.countDescendants("field");
            fields = new GeoIndexField[nrOfFields];
            while (parser.setNextField()) {
                fillField(position++);
            }

        } catch (Exception ex) {
            logger.error("Error while reading indexType, position: " + position, ex);
            throw ex;
        }
    }

    /**
     * The method which actually fills a field of this GeoIndexType object
     * 
     * @param position -
     *            The index (in the fields array) of the GeoIndexField to fill
     * @throws Exception -
     *             Unable to get the field information from the parser
     */
    private void fillField(int position) throws Exception {
        String sizeString;

        fields[position] = new GeoIndexField();
        // logger.info("name: " +
        // parser.getFieldByValue("name"));
        fields[position].name = parser.getFieldByValue("name");
        fields[position].dataType = GeoIndexField.DataType.getByName(parser
                .getFieldByValue("type"));
        if (fields[position].dataType.equals(GeoIndexField.DataType.STRING)
                && (sizeString = parser.getFieldByValue("size")).length() > 0) {
            fields[position].size = Integer.parseInt(sizeString);
        } else {
            fields[position].size = fields[position].dataType.getDefaultSize();
        }

        fields[position].isReturnable = parser.getFieldByValue("return")
                .equalsIgnoreCase("yes") ? true : false;

        //this flag denotes if the size of this field is variable, 
        //and we can NOT store it directly to the R-tree of geo-tools
        fields[position].isVariable = parser.getFieldByValue("variable")
        		.equalsIgnoreCase("yes") ? true : false;
        
        size += fields[position].size;
        fieldIndexMap.put(fields[position].name, position);

    }

    /**
     * A Method to check if this GeoIndexType object contains a certain field
     * 
     * @param fieldName -
     *            The name of the field to look for
     * @param fieldDataType -
     *            The DataType of the field to look for
     * @return - TRUE if this GeoIndexType object contains a field with the
     *         specified name and data type
     */
    public boolean containsField(String fieldName,
            GeoIndexField.DataType fieldDataType) {
        for (GeoIndexField field : fields) {
            if (field.name.equals(fieldName)
                    && field.dataType.equals(fieldDataType)) {
                return true;
            }
        }
        return false;
    }

    /**
     * A Method to get the position of a field from the field name
     * 
     * @param fieldName -
     *            The name of the field to get the position of
     * @return - the position of the specified field. NULL if not found.
     */
    public Integer getFieldPosition(String fieldName) {
        return fieldIndexMap.get(fieldName);
    }

    /**
     * {@inheritDoc}
     */
    public String toString() {
        String ret = ("IndexType: \nid=" + "tempID" + "\nnrOfFields="
                + nrOfFields + "\n");
        for (int i = 0; i < nrOfFields; i++) {
            ret += (fields[i] + "\n");
        }
        return ret;
    }

}
