package org.gcube.indexmanagement.featureindexlibrary.vafile;

import java.io.RandomAccessFile;
import java.util.ArrayList;
import java.util.Calendar;

import org.apache.log4j.Logger;
import org.gcube.indexmanagement.featureindexlibrary.commons.*;
import org.gcube.indexmanagement.featureindexlibrary.vafile.algo.ComputeBoundsL;
import org.gcube.indexmanagement.featureindexlibrary.vafile.algo.ComputeBoundsLU;
import org.gcube.indexmanagement.featureindexlibrary.vafile.algo.PointPartitioning;
import org.gcube.indexmanagement.featureindexlibrary.vafile.algo.PopulateIndex;
import org.gcube.indexmanagement.featureindexlibrary.vafile.elements.ApproximationFileEntry;
import org.gcube.indexmanagement.featureindexlibrary.vafile.elements.ApproximationFileEntryInfo;
import org.gcube.indexmanagement.featureindexlibrary.vafile.elements.VectorFileEntry;
import org.gcube.indexmanagement.featureindexlibrary.vafile.elements.VectorFileEntryInfo;
import org.gcube.indexmanagement.featureindexlibrary.vafile.io.ApproximationFileRandomAccess;
import org.gcube.indexmanagement.featureindexlibrary.vafile.io.ApproximationFileReader;
import org.gcube.indexmanagement.featureindexlibrary.vafile.io.FileBufferWriter;
import org.gcube.indexmanagement.featureindexlibrary.vafile.io.IOHelper;
import org.gcube.indexmanagement.featureindexlibrary.vafile.io.VectorFileRandomAccess;
import org.gcube.indexmanagement.featureindexlibrary.vafile.io.VectorFileReader;

/**
 * The VAFile index
 * 
 * @author UoA
 */
public class VAFile implements FeatureIndex{
	/**
	 * Used for synchronization
	 */
	private static final Object lockMe=new Object(); 
	/**
	 * The Logger used by the class
	 */
	private static Logger logger = Logger.getLogger(VAFile.class);
	/**
	 * The header parameters
	 */
	private VAFileParams params=null;
	
	/**
	 * Creates a new instance
	 * 
	 * @param params The index creation parameters 
	 * @throws Exception An error
	 */
	public VAFile(VAFileParams params) throws Exception{
		try{
			this.params=params;
			if(this.params.getDistanceMeasure().compareTo(FIEnums.DistanceTypes.Default)==0){
				logger.error("Cannot initialize index with default distance measure. throwing exception");
				throw new Exception("Cannot initialize index with default distance measure");
			}
			if(FileHelper.existsVAFVectorFile(params.getStorage(),params.getIndexID())){
				IOHelper helper=new IOHelper();
				RandomAccessFile rand=new RandomAccessFile(FileHelper.getVAFVectorFile(params.getStorage(),params.getIndexID()),"rw");
				this.params=helper.readHeader(rand);
				rand.close();
			}
		}catch(Exception e){
			logger.error("Could not initialize VAFile. throwing Exception",e);
			throw new Exception("Could not initialize VAFile");
		}
	}
	
	/**
	 * @see org.gcube.indexmanagement.featureindexlibrary.commons.FeatureIndex#addFeatureVector(org.gcube.indexmanagement.featureindexlibrary.commons.FeatureVectorElement)
	 * @param elem The element to add
	 * @throws Exception an error
	 */
	public void addFeatureVector(FeatureVectorElement elem) throws Exception{
		synchronized (lockMe) {
			FileBufferWriter writer=null;
			int count=0;
			try{
				writer=new FileBufferWriter(FileHelper.getVAFBufferFile(params.getStorage(),params.getIndexID()),params);
				writer.openForUpdate();
				if(!writer.addElement(elem)) count+=1;
				writer.close();
				if(count>0) logger.info("Could not add "+count+" record");
			}catch(Exception e){
				if(writer!=null) writer.close();
				logger.error("Could not add record. throwing Exception",e);
				throw new Exception("Could not add record");
			}
		}
	}

	/**
	 * @see org.gcube.indexmanagement.featureindexlibrary.commons.FeatureIndex#addFeatureVector(org.gcube.indexmanagement.featureindexlibrary.commons.FeatureVectorElement[])
	 * @param elem the elements to add
	 * @throws Exception an error
	 */
	public void addFeatureVector(FeatureVectorElement []elem) throws Exception{
		synchronized (lockMe) {
			FileBufferWriter writer=null;
			int count=0;
			try{
				writer=new FileBufferWriter(FileHelper.getVAFBufferFile(params.getStorage(),params.getIndexID()),params);
				writer.openForUpdate();
				for(int i=0;i<elem.length;i+=1){
					if(!writer.addElement(elem[i])) count+=1;
				}
				writer.close();
				if(count>0) logger.info("Could not add "+count+" record");
			}catch(Exception e){
				if(writer!=null) writer.close();
				logger.error("Could not add records. throwing Exception",e);
				throw new Exception("Could not add records");
			}
		}
	}
	
	/**
	 * @see org.gcube.indexmanagement.featureindexlibrary.commons.FeatureIndex#commit()
	 * @throws Exception an error
	 */
	public void commit() throws Exception{
		synchronized (lockMe) {
			PopulateIndex pop=null;
			try{
				if(!FileHelper.existsVAFBufferFile(params.getStorage(),params.getIndexID())) return;
				if(!FileHelper.existsVAFVectorFile(params.getStorage(),params.getIndexID())){
					PointPartitioning partitioning=new PointPartitioning(FileHelper.getVAFBufferFile(params.getStorage(),params.getIndexID()),params);
					partitioning.computePartitionPoints();
					params.setBuckets(partitioning.getPartitionBuckets());
					params.setPartitionPoints(partitioning.getPartitionPoints());
					pop=new PopulateIndex(params,true);
				}
				else{
					pop=new PopulateIndex(params,false);
				}
				pop.open();
				pop.populate();
				pop.close();
				FileHelper.removeVAFBufferFile(params.getStorage(),params.getIndexID());
			}catch(Exception e){
				FileHelper.removeVAFBufferFile(params.getStorage(),params.getIndexID());
				if(pop!=null) pop.close();
				logger.error("Could not perform commit. Throwing Exception",e);
				throw new Exception("Could not perform commit");
			}
		}
	}
	
	/**
	 * @see org.gcube.indexmanagement.featureindexlibrary.commons.FeatureIndex#getNumberOfElements()
	 * @return the number of elements
	 * @throws Exception an error
	 */
	public long getNumberOfElements() throws Exception{
		return this.params.getElementCount();
	}
	
	/**
	 * @see org.gcube.indexmanagement.featureindexlibrary.commons.FeatureIndex#lookup(float[], int)
	 * @param vector the vectror to search for
	 * @param k the number of results
	 * @return the results
	 * @throws Exception an error
	 */
	public ArrayList<RankedResultElement> lookup(float []vector,int k) throws Exception{
		try{
			return this.lookup(vector,k,new LookupParams(FIEnums.AlgoType.Default,FIEnums.DistanceTypes.Default,params.getWeighted()));
		}catch(Exception e){
			logger.error("Could not perform lookup operation. Throwing Exception",e);
			throw new Exception("Could not perform lookup operation");
		}
	}
	
	/**
	 * @see org.gcube.indexmanagement.featureindexlibrary.commons.FeatureIndex#lookup(float[], int, org.gcube.indexmanagement.featureindexlibrary.commons.LookupParams)
	 * @param vector the vectror to search for
	 * @param k the number of results
	 * @param lp lookup parameters
	 * @return the results
	 * @throws Exception an error
	 */
	public ArrayList<RankedResultElement> lookup(float []vector,int k,LookupParams lp) throws Exception{
		try{
			ArrayList<RankedResultElement> res=null;
			long start=Calendar.getInstance().getTimeInMillis();
			long bstart=0;
			long bstop=0;
			long lstart=0;
			long lstop=0;
			if(lp.getAlgo().compareTo(FIEnums.AlgoType.SSA)==0){
				bstart=Calendar.getInstance().getTimeInMillis();
				ComputeBoundsL bounds=new ComputeBoundsL(params.getPartitionPoints(),params.getBuckets());
				bounds.preCompute(vector);
				bstop=Calendar.getInstance().getTimeInMillis();
				lstart=Calendar.getInstance().getTimeInMillis();
				res=this.lookupSSA(vector,k,bounds,lp);
				lstop=Calendar.getInstance().getTimeInMillis();
			}
			if(lp.getAlgo().compareTo(FIEnums.AlgoType.Default)==0 || lp.getAlgo().compareTo(FIEnums.AlgoType.NOA)==0){
				bstart=Calendar.getInstance().getTimeInMillis();
				ComputeBoundsLU bounds=new ComputeBoundsLU(params.getPartitionPoints(),params.getBuckets());
				bounds.preCompute(vector);
				bstop=Calendar.getInstance().getTimeInMillis();
				lstart=Calendar.getInstance().getTimeInMillis();
				res=this.lookupNOA(vector,k,bounds,lp);
				lstop=Calendar.getInstance().getTimeInMillis();
			}
			long stop=Calendar.getInstance().getTimeInMillis();
			logger.info("bound computations took "+(bstop-bstart)+" millisecs");
			logger.info("lookup computations took "+(lstop-lstart)+" millisecs");
			logger.info("total lookup took "+(stop-start)+" millisecs");
			return res;
		}catch(Exception e){
			logger.error("Could not perform lookup operation. Throwing Exception",e);
			throw new Exception("Could not perform lookup operation");
		}
	}
	
	/**
	 * Simple Search Algorithm
	 * 
	 * @param vector The prototype vector
	 * @param k The number of results
	 * @param bounds The precomputed lower bounds
	 * @param lp The lookup parameteers
	 * @return The results 
	 * @throws Exception An error
	 */
	private ArrayList<RankedResultElement> lookupSSA(float []vector,int k,ComputeBoundsL bounds,LookupParams lp) throws Exception{
		try{
			ApproximationFileReader aReader=new ApproximationFileReader(params);
			VectorFileRandomAccess vReader=new VectorFileRandomAccess(params);
			LookupBuffer buf=new LookupBuffer(k);
			aReader.open();
			vReader.open();
			float threshold=Float.MAX_VALUE;
			while(true){
				ApproximationFileEntryInfo info=aReader.getNext();
				if(info==null) break;
				if(bounds.getLowerBound(info.getEntry().getBitString())<threshold){
					vReader.moveTo(info.getPosition());
					VectorFileEntry entry=vReader.read();
					if(entry==null) continue;
					FIEnums.DistanceTypes distMeasure=this.params.getDistanceMeasure();
					if(lp.getDist().compareTo(FIEnums.DistanceTypes.Default)!=0) distMeasure=lp.getDist();
					float distance=0;
					if(lp.isWeighted()) distance=DistanceCalculation.distance(vector,entry.getVector(),threshold,distMeasure,params.getWeights());
					else distance=DistanceCalculation.distance(vector,entry.getVector(),threshold,distMeasure);
					threshold=buf.process(new RankedResultElement(entry.getId(),distance));
				}
			}
			aReader.close();
			vReader.close();
			return buf.getResults(this.params.getIndexName());
		}catch(Exception e){
			logger.error("Could not perform lookup operation. Throwing Exception",e);
			throw new Exception("Could not perform lookup operation");
		}
	}

	/**
	 * Near Optimal Algorithm
	 * 
	 * @param vector The prototype vector
	 * @param k The number of results
	 * @param bounds The precomputed lower and upper bounds
	 * @param lp The lookup parameteers
	 * @return The results 
	 * @throws Exception An error
	 */
	private ArrayList<RankedResultElement> lookupNOA(float []vector,int k,ComputeBoundsLU bounds,LookupParams lp) throws Exception{
		try{
			ApproximationFileReader aReader=new ApproximationFileReader(params);
			LookupBuffer buf=new LookupBuffer(k);
			SortedHeap heap=new SortedHeap();
			aReader.open();
			float threshold=Float.MAX_VALUE;
			float lBound=0;
			while(true){
				ApproximationFileEntryInfo info=aReader.getNext();
				if(info==null) break;
				lBound=bounds.getLowerBound(info.getEntry().getBitString());
				if(lBound<threshold){
					threshold=buf.process(new RankedResultElement("",bounds.getUpperBound(info.getEntry().getBitString())));
					heap.add(new HeapElement(info.getPosition(),lBound));
				}
			}
			aReader.close();
			heap.sort();
			
			VectorFileRandomAccess vReader=new VectorFileRandomAccess(params);
			vReader.open();
			buf=new LookupBuffer(k);
			threshold=Float.MAX_VALUE;
			lBound=0;
			long recNum=0;
			int count=heap.getNumberOfElements();
			for(int i=0;i<count && lBound<threshold ;i+=1){
				lBound=heap.get(i).getLBound();
				recNum=heap.get(i).getRecNum();
				if(lBound<threshold){
					vReader.moveTo(recNum);
					VectorFileEntry entry=vReader.read();
					if(entry==null) continue;
					FIEnums.DistanceTypes distMeasure=this.params.getDistanceMeasure();
					if(lp.getDist().compareTo(FIEnums.DistanceTypes.Default)!=0) distMeasure=lp.getDist();
					float distance=0;
					if(lp.isWeighted()) distance=DistanceCalculation.distance(vector,entry.getVector(),threshold,distMeasure,params.getWeights());
					else distance=DistanceCalculation.distance(vector,entry.getVector(),threshold,distMeasure);
					threshold=buf.process(new RankedResultElement(entry.getId(),distance));
				}
			}
			vReader.close();
			return buf.getResults(this.params.getIndexName());
		}catch(Exception e){
			logger.error("Could not perform lookup operation. Throwing Exception",e);
			throw new Exception("Could not perform lookup operation");
		}
	}

	/**
	 * @see org.gcube.indexmanagement.featureindexlibrary.commons.FeatureIndex#removeEntry(java.lang.String)
	 * @param id the id to remove
	 * @throws Exception an error
	 */
	public void removeEntry(String id) throws Exception{
		synchronized (lockMe) {
			try{
				VectorFileReader reader=new VectorFileReader(params);
				long recNum=0;
				reader.open();
				while(true){
					VectorFileEntryInfo entry=reader.getNext();
					if(entry==null) break;
					if(entry.getEntry().getId().compareTo(id)==0){
						recNum=entry.getPosition();
						break;
					}
				}
				reader.close();
				
				ApproximationFileRandomAccess apReader=new ApproximationFileRandomAccess(params);
				apReader.open();
				apReader.moveTo(recNum);
				ApproximationFileEntry apEntry=apReader.read();
				apEntry.setActive(false);
				apReader.moveTo(recNum);
				apReader.write(apEntry);
				apReader.close();
				
				VectorFileRandomAccess vReader=new VectorFileRandomAccess(params);
				vReader.open();
				vReader.moveTo(recNum);
				VectorFileEntry vEntry=vReader.read();
				vEntry.setActive(false);
				vReader.moveTo(recNum);
				vReader.write(vEntry);
				vReader.close();
			}catch(Exception e){
				logger.error("Could not remove record. Throwing Exception",e);
				throw new Exception("Could not remove record");
			}
		}
	}
	
	/**
	 * Returns the index id of the index instance
	 * 
	 * @return The index ID
	 * @throws Exception An unrecoverable for the operation error occured
	 * @see org.gcube.indexmanagement.featureindexlibrary.commons.FeatureIndex#getIndexID()
	 */
	public String getIndexID() throws Exception {
		return this.params.getIndexID();
	}
}
