package org.gcube.indexmanagement.geo.shape;

import java.util.ArrayList;
import java.util.Collection;

/** A java representation of a Polygon */
public class Polygon {

    /** A list of all the vertices (corners) of the polygon */
    private ArrayList<Point> vertices;

    /** Constructor */
    public Polygon() {
        this.vertices = new ArrayList<Point>();
    }

    /**
     * Constructor which creates a polygon from a collection of points
     * representing the vertices
     * 
     * @param vertices -
     *            collection of points representing the vertices of the polygon
     *            to be created
     */
    public Polygon(Collection<Point> vertices) {
        this.vertices = new ArrayList<Point>();
        addVertices(vertices);
    }

    /**
     * Constructor which creates a polygon from an array of points representing
     * the vertices
     * 
     * @param vertices -
     *            an array of points representing the vertices of the polygon to
     *            be created
     */
    public Polygon(Point[] vertices) {
        this.vertices = new ArrayList<Point>();
        addVertices(vertices);
    }

    /**
     * A method to add a vertex to the polygon
     * 
     * @param vertex -
     *            The vertex to be added to the polygon
     */
    public void addVertex(Point vertex) {
        this.vertices.add(vertex);
    }

    /**
     * A method to add a collection of vertices to the polygon
     * 
     * @param vertices -
     *            a collection of vertices to be added to the polygon
     */
    public void addVertices(Collection<Point> vertices) {
        this.vertices.addAll(vertices);
    }

    /**
     * A method to add an array of vertices to the polygon
     * 
     * @param vertices -
     *            an array of vertices to be added to the polygon
     */
    public void addVertices(Point[] vertices) {
        for (int i = 0; i < vertices.length; i++) {
            this.vertices.add(vertices[i]);
        }
    }

    /**
     * A method to get an ArrayList of the vertices of the polygon
     * 
     * @return The vertices of the polygon
     */
    public ArrayList<Point> getVertices() {
        return vertices;
    }

    /**
     * A method to get an array of the vertices of the polygon
     * 
     * @return The vertices of the polygon
     */
    public Point[] getVertexArray() {
        return vertices.toArray(new Point[vertices.size()]);
    }

    /**
     * A method to get the Minimum Bounding Rectanngle of the polygon
     * 
     * @return The Minimum Bounding Rectanngle of the polygon
     */
    public Rectangle getBoundingBox() {
        ArrayList<Point> vertices = getVertices();
        long minX = vertices.get(0).getX(), maxX = vertices.get(0).getX();
        long minY = vertices.get(0).getY(), maxY = vertices.get(0).getY();

        for (Point vertex : vertices) {
            if (vertex.getX() < minX)
                minX = vertex.getX();
            else if (vertex.getX() > maxX)
                maxX = vertex.getX();
            if (vertex.getY() < minY)
                minY = vertex.getY();
            else if (vertex.getY() > maxY)
                maxY = vertex.getY();
        }
        return new Rectangle(minX, maxX, minY, maxY);
    }

    /**
     * A method to check the validity of the polygon. A valid polygon has three
     * or more vertices, and no crossing lines.
     * 
     * @return TRUE if the the polygon is valid.
     */
    public boolean isValid() {

        int length = vertices.size();
        if (length > 2) {
            Point lineStart = null;
            Point lineEnd = null;
            long curStartX, curEndX, curStartY, curEndY;
            for (int i = 1; i < length; i++) {
                curStartX = vertices.get(i - 1).getX();
                curEndX = vertices.get(i).getX();
                curStartY = vertices.get(i - 1).getY();
                curEndY = vertices.get(i).getY();
                for (int j = 2; j < length - 1; j++) {
                    lineStart = vertices.get((j + i - 1) % length);
                    lineEnd = vertices.get((j + i) % length);
                    if (doLinesCross(curStartX, curStartY, curEndX, curEndY,
                            lineStart.getX(), lineStart.getY(), lineEnd.getX(),
                            lineEnd.getY()))
                        return false;
                }
                // check the next to last line against the last point (checking
                // whole last line would always fail)
                lineStart = vertices.get((i + length - 2) % length);
                lineEnd = vertices.get((i + length - 1) % length);
                if (doesLastCross(lineStart.getX(), lineStart.getY(), lineEnd
                        .getX(), lineEnd.getY(), curEndX, curEndY))
                    return false;
            }
        } else
            return false;
        return true;
    }

    /*
     * ********************************************************************* An
     * implementation of the Sedgewick line-cross-detecting algorithm:
     */

    /**
     * A method to check if the "elbow" of line, vertex, line pair bends
     * clockwise of counter clockwise.
     * 
     * @param x1 -
     *            X coordinate of the first line's start.
     * @param y1 -
     *            Y coordinate of the first line's start.
     * @param x2 -
     *            X coordinate of the vertex (first line's end, second line's
     *            start).
     * @param y2 -
     *            Y coordinate of the vertex (first line's end, second line's
     *            start).
     * @param x3 -
     *            X coordinate of the second line's end.
     * @param y3 -
     *            Y coordinate of the second line's end.
     * @return 1 if elbow bends counter-clockwise, 0 if straight, -1 if it bends
     *         clockwise.
     */
    private int ccw(long x1, long y1, long x2, long y2, long x3, long y3) {
        long dx1 = x2 - x1;
        long dx2 = x3 - x1;
        long dy1 = y2 - y1;
        long dy2 = y3 - y1;

        if (dy1 * dx2 < dy2 * dx1) {
            return 1;
        }
        if (dy1 * dx2 > dy2 * dx1) {
            return -1;
        }
        if (dx1 * dx2 < 0 || dy1 * dy2 < 0) {
            return -1;
        }
        if ((dx1 * dx1 + dy1 * dy1) >= (dx2 * dx2 + dy2 * dy2))
            return 0;
        else
            return 1;
    }

    /**
     * A method to check if two lines cross
     * 
     * @param x1 -
     *            Starting X coordinate of the first line.
     * @param y1 -
     *            Starting Y coordinate of the first line.
     * @param x2 -
     *            Ending X coordinate of the first line.
     * @param y2 -
     *            Ending Y coordinate of the first line.
     * @param x3 -
     *            Starting X coordinate of the second line.
     * @param y3 -
     *            Starting Y coordinate of the second line.
     * @param x4 -
     *            Ending X coordinate of the second line.
     * @param y4 -
     *            Ending Y coordinate of the second line.
     * @return TRUE if the two lines cross
     */
    private boolean doLinesCross(long x1, long y1, long x2, long y2, long x3,
            long y3, long x4, long y4) {
        return ccw(x1, y1, x2, y2, x3, y3) * ccw(x1, y1, x2, y2, x4, y4) <= 0
                && ccw(x3, y3, x4, y4, x1, y1) * ccw(x3, y3, x4, y4, x2, y2) <= 0;
    }

    /**
     * Adaption for the next to last line of a polygon tested against the last
     * point (only checks if last point is on line)
     * 
     * Example: p1-------p3--------p2
     * 
     * @param x1 -
     *            Starting X coordinate of the line.
     * @param y1 -
     *            Starting Y coordinate of the line.
     * @param x2 -
     *            Ending X coordinate of the line.
     * @param y2 -
     *            Ending Y coordinate of the line.
     * @param pointX -
     *            X coordinate of the point.
     * @param pointY -
     *            Y coordinate of the point.
     * 
     * @return TRUE if the last point is on the next to last line
     */
    private boolean doesLastCross(long x1, long y1, long x2, long y2,
            long pointX, long pointY) {
        long dx1 = x2 - x1;
        long dx2 = pointX - x1;
        long dy1 = y2 - y1;
        long dy2 = pointY - y1;
        // if the slopes are equal and the p1 (linestart) is closer to p3 than
        // p2 (lineend)
        if (dy1 * dx2 == dy2 * dx1
                && (dx1 * dx1 + dy1 * dy1) >= (dx2 * dx2 + dy2 * dy2)) {
            return true;
        }
        return false;
    }

    /*
     * ************End of line-cross-detecting algorithm **********************
     */

    /**
     * {@inheritDoc}
     */
    public String toString() {
        StringBuffer buffer = new StringBuffer();
        boolean first = true;
        String separator = "";
        buffer.append("Poly[");
        for (Point vertex : vertices) {
            buffer.append(separator).append(vertex.getX()).append(", ").append(
                    vertex.getY());
            if (first) {
                separator = " ; ";
            }
        }
        buffer.append("]");
        return buffer.toString();
    }
}
