package org.gcube.indexmanagement.geo.shape;

import java.awt.geom.Rectangle2D;

/** A java representation of a rectangle */
public class Rectangle {

    /** The rectangle smallest X coordinate */
    private long minX;

    /** The rectangle biggest X coordinate */
    private long maxX;

    /** The rectangle smallest Y coordinate */
    private long minY;

    /** The rectangle biggest Y coordinate */
    private long maxY;

    /**
     * A constructor creating a rectangle from two X and two Y coordinates
     * 
     * @param x1 -
     *            one of two X coodinates
     * @param x2 -
     *            one of two X coodinates
     * @param y1 -
     *            one of two Y coodinates
     * @param y2 -
     *            one of two Y coodinates
     */
    public Rectangle(long x1, long x2, long y1, long y2) {
        minX = Math.min(x1, x2);
        maxX = Math.max(x1, x2);
        minY = Math.min(y1, y2);
        maxY = Math.max(y1, y2);
    }

    /**
     * A constructor creating a rectangle from a jts Envelope
     * 
     * @param e -
     *            The jts Envelope to represent as a rectangle
     */
    public Rectangle(com.vividsolutions.jts.geom.Envelope e) {
        minX = new Double(e.getMinX()).longValue();
        maxX = new Double(e.getMaxX()).longValue();
        minY = new Double(e.getMinY()).longValue();
        maxY = new Double(e.getMaxY()).longValue();
    }

    /** 
     * Getter method for the smallest X coordinate
     * @return the min X coordinate
     */
    public long getMinX() {
        return minX;
    }

    /** 
     * Getter method for the biggest X coordinate 
     * @return the max X coordinate
     */
    public long getMaxX() {
        return maxX;
    }

    /** 
     * Getter method for the smallest Y coordinate 
     * @return the min Y coordinate
     */
    public long getMinY() {
        return minY;
    }

    /** 
     * Getter method for the biggest Y coordinate 
     * @return the max Y coordinate
     */
    public long getMaxY() {
        return maxY;
    }

    /**
     * A method which decides if this rectangle contains the specified
     * rectangle.
     * 
     * @param r -
     *            The rectangle to test whether is contained by this rectangle.
     * @return TRUE if this rectangle contains the specified rectangle
     */
    public boolean contains(Rectangle r) {
        return maxX >= r.getMaxX() && maxY >= r.getMaxY()
                && minX <= r.getMinX() && minY <= r.getMinY();
    }

    /**
     * A method which decides if this rectangle contains the specified point.
     * 
     * @param x -
     *            The X coordinate of the point to test whether is contained by
     *            this rectangle.
     * @param y -
     *            The Y coordinate of the point to test whether is contained by
     *            this rectangle.
     * @return TRUE if this rectangle contains the specified point.
     */
    public boolean containsPoint(long x, long y) {
        return x >= minX && x <= maxX && y >= minY && y <= maxY;
    }

    /**
     * A method which decides if this rectangle overlaps with the specified
     * rectangle.
     * 
     * @param r -
     *            The rectangle to test whether overlaps with this rectangle.
     * @return TRUE if this rectangle overlaps with specified rectangle
     */
    public boolean overlaps(Rectangle r) {
        return containsPoint(r.getMinX(), r.getMinY())
                || containsPoint(r.getMinX(), r.getMaxY())
                || containsPoint(r.getMaxX(), r.getMinY())
                || containsPoint(r.getMaxX(), r.getMaxY()) || r.contains(this);
    }

    /**
     * A method used to get a Rectangle2D representation of this rectangle.
     * 
     * Remember that a polygon uses long precision, while Rectangle2D uses
     * double. Though it is a widening transformation, it is not always precise.
     * From Sun JVM book: "The l2d instruction performs a widening primitive
     * conversion (§2.6.2) that may lose precision because values of type double
     * have only 53 significand bits." The problem seems to occur after 2^53 = 9
     * 007 199 254 740 992.
     * 
     * @return The requested Rectangle2D representation of the rectangle.
     */
    private Rectangle2D getRectangle2D() {
        return new Rectangle2D.Double(minX, minY, maxX - minX, maxY - minY);
    }

    /**
     * {@inheritDoc}
     */
    public String toString() {
        return new StringBuffer().append("Rect[x1: ").append(minX).append(
                " x2: ").append(maxX).append(" | y1: ").append(minY).append(
                " y2: ").append(maxY).append("]").toString();
    }
}
