/*
 * LuceneGenerator.java
 *
 * $Author: tsakas $
 * $Date: 2007/12/20 14:37:39 $
 * $Id: LuceneGenerator.java,v 1.1 2007/12/20 14:37:39 tsakas Exp $
 *
 * <pre>
 *             Copyright (c) : 2006 Fast Search & Transfer ASA
 *                             ALL RIGHTS RESERVED
 * </pre>
 */

package org.gcube.indexmanagement.lucenewrapper;

import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.io.StringReader;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;

import javax.xml.parsers.SAXParser;
import javax.xml.parsers.SAXParserFactory;

import org.apache.lucene.analysis.Analyzer;
import org.apache.lucene.analysis.TokenStream;
import org.apache.lucene.analysis.standard.StandardAnalyzer;
import org.apache.lucene.document.Document;
import org.apache.lucene.document.Field;
import org.apache.lucene.index.IndexReader;
import org.apache.lucene.index.IndexWriter;
import org.apache.lucene.index.Term;
import org.apache.lucene.store.Directory;
import org.apache.lucene.store.FSDirectory;
import org.apache.lucene.store.RAMDirectory;
import org.gcube.common.core.utils.logging.GCUBELog;
import org.gcube.indexmanagement.common.FullTextIndexType;
import org.gcube.indexmanagement.common.IndexException;
import org.gcube.indexmanagement.common.IndexField;
import org.gcube.indexmanagement.common.IndexGenerator;
import org.gcube.indexmanagement.common.IndexType;
import org.gcube.indexmanagement.common.XMLProfileParser;
import org.xml.sax.Attributes;
import org.xml.sax.InputSource;
import org.xml.sax.XMLReader;
import org.xml.sax.helpers.DefaultHandler;

/**
 * An IndexGenerator implementation responsible for creating and updating a
 * Lucene index.
 * 
 */
public class LuceneGenerator implements IndexGenerator {

	/** logger */
	static GCUBELog logger = new GCUBELog(LuceneGenerator.class);
	
    /** The base directory where all Index directories are stored. */
    private final String baseIndexDir;

    /** The directory in which to store the deletion file. */
    private final String deletionDir;

    /** The writer to use for writing the Index to disk */
    private IndexWriter writer = null;

    /**
     * The writer to use for buffering in memory before writing the Index to
     * disk
     */
    private IndexWriter ramWriter = null;

    /** The IndexReader to use for reading from the Index */
    private IndexReader reader = null;

    /** The IndexReader to use for reading from the memory buffer */
    private IndexReader ramReader = null;

    /** The RAM directory to use for storing the buffer in memory */
    private RAMDirectory ramDir = null;

    /** The directory to store the index in */
    private String indexDir = null;

    /** The IndexType of the Index */
    private FullTextIndexType idxType = null;

    private Analyzer NoStopAnalyser = new StandardAnalyzer(new String[0]);
    /**
     * A constructor which sets the base directory for Index directories, and
     * the directory in which to store deletion files.
     * 
     * @param baseIndexDir -
     *            The base directory for Index directories.
     * @param deletionDir -
     *            The directory in which to store deletion files.
     */
    public LuceneGenerator(String baseIndexDir, String deletionDir) {
        this.baseIndexDir = baseIndexDir;
        this.deletionDir = deletionDir;
        File ddFile = new File(deletionDir);
        if (!ddFile.exists())
        	ddFile.mkdirs();
    }

    /**
     * {@inheritDoc}
     */
    public synchronized void createIndex(String indexName, FullTextIndexType idxType, boolean forceCreate) throws IndexException {
        try {
        	logger.info("creating index: " + indexName);
            writer = createLuceneIndexWriter(baseIndexDir + indexName,forceCreate);
            writer.close();
            indexDir = indexName;
            this.idxType = idxType;

        } catch (Exception e) {
            throw new IndexException(e);
        }
    }
    
    private IndexWriter createLuceneIndexWriter(String dir, boolean forceCreate) throws IOException
    {
    	GlobalSimilarity similarity = new GlobalSimilarity();
    	IndexWriter indexWriter = new IndexWriter(dir, NoStopAnalyser, forceCreate);
    	indexWriter.setSimilarity(similarity);
    	return indexWriter;
    }

    private IndexWriter createLuceneIndexWriter(Directory dir, boolean forceCreate) throws IOException
    {
    	GlobalSimilarity similarity = new GlobalSimilarity();
    	IndexWriter indexWriter = new IndexWriter(dir, NoStopAnalyser, forceCreate);
    	indexWriter.setSimilarity(similarity);
    	return indexWriter;
    }
    /**
     * {@inheritDoc}
     */
    public synchronized void openIndex(String indexName) throws IndexException {
        try {
            ramDir = new RAMDirectory();
            ramWriter = createLuceneIndexWriter(ramDir, true);
            indexDir = indexName;
            logger.info("opening index: " + indexDir);
        } catch (java.io.IOException e) {
            throw new IndexException(e);
        }
    }

    /**
     * {@inheritDoc}
     */
    public synchronized void setIndexType(FullTextIndexType idxType)
            throws IndexException {
        this.idxType = idxType;
    }

    /**
     * {@inheritDoc}
     */
    public synchronized long getIndexFileSize() throws IndexException {
        String directoryPath = baseIndexDir + indexDir;
        return getFileSize(new File(directoryPath));
    }

    /**
     * A method to get the size of a file or directory.
     * 
     * @param file -
     *            The file or directory of which to return the size.
     * @return The requested file/directory size.
     * @throws IndexException
     */
    private synchronized long getFileSize(File file) throws IndexException {
        if (file.canRead()) {
            if (file.isDirectory()) {
                long size = 0;
                String[] files = file.list();
                if (files != null) {
                    for (int i = 0; i < files.length; i++) {
                        size += getFileSize(new File(file, files[i]));
                    }
                }
                return size;
            } else
                return file.length();
        } else
            throw new IndexException("Unable to get the size of the Index");
    }

    /**
     * {@inheritDoc}
     */
    public synchronized void abort() throws IndexException {
        try {
            if (ramWriter != null) {
                ramWriter.close();
            }
            ramDir = new RAMDirectory();
            ramWriter = createLuceneIndexWriter(ramDir, true);
        } catch (java.io.IOException e) {
            throw new IndexException(e);
        }
    }

    /**
     * {@inheritDoc}
     */
    public synchronized void commit() throws IndexException {
        try {
            // merges ramwriter into main index
            ramWriter.close();
            Directory[] mergeDirs = { ramDir };
            writer = createLuceneIndexWriter(baseIndexDir + indexDir, false);
            writer.addIndexes(mergeDirs);
            writer.close();

            // resets the ramWriter
            ramDir = new RAMDirectory();
            ramWriter = createLuceneIndexWriter(ramDir, true);
        } catch (Exception e) {
            throw new IndexException(e);
        }
    }

    /**
     * {@inheritDoc}
     */
    public synchronized long getCommittedFileSize() throws IndexException {
        try {
            long size = 0;
            String[] files;
            Directory tempDir = writer.getDirectory();
            if ((files = tempDir.list()) != null) {
                for (String fileName : files) {
                    size += tempDir.fileLength(fileName);
                }
            }
            return size;
        } catch (Exception e) {
            throw new IndexException(e);
        }
    }

    /**
     * {@inheritDoc}
     */
    public synchronized long getUnCommittedFileSize() throws IndexException {
        try {
        	return ramDir.sizeInBytes();
        } catch (Exception e) {
            throw new IndexException(e);
        }
    }

    /**
     * A method used to create and initialize a deletion file.
     * 
     * @return The newly created deletion file.
     * @throws IndexException
     */
    private synchronized File createDeletionFile() throws IndexException {
        try {
            int deletionCount = 0;
            File deletionFile;
            do {
                deletionFile = new File(deletionDir + "deletion" + deletionCount + ".xml");
                deletionCount++;
            } while (deletionFile.exists());
            deletionFile.createNewFile();
            FileWriter writer = new FileWriter(deletionFile);
            writer.write("<?xml version=\"1.0\" encoding=\"iso-8859-1\"?><IDList/>");
            writer.close();

            return deletionFile;
        } catch (Exception e) {
            throw (new IndexException(e));
        }
    }

    /**
     * {@inheritDoc}
     */
    public synchronized int mergeIndex(File inputIndex) throws IndexException {
        try {
            IndexWriter mergeWriter = createLuceneIndexWriter(baseIndexDir + indexDir, false);
            int docCount = mergeWriter.docCount();
            Directory inputDirectory = FSDirectory.getDirectory(inputIndex.getAbsolutePath(), false);
            Directory[] mergeDirs = { inputDirectory };
            mergeWriter.addIndexes(mergeDirs);
            int change = mergeWriter.docCount() - docCount;
            mergeWriter.close();
            return change;

        } catch (Exception e) {
            logger.error("Error while merging index.", e);
            throw new IndexException(e);
        }

    }

    /**
     * {@inheritDoc}
     */
    public synchronized int mergeDeletionFile(File deletionFile)
            throws IndexException {
        try {
            IdDeletionHandler deletionHandler = new IdDeletionHandler();
            SAXParserFactory factory = SAXParserFactory.newInstance();
            SAXParser saxParser = factory.newSAXParser();
            XMLReader xmlReader = saxParser.getXMLReader();
            xmlReader.setContentHandler(deletionHandler);
            xmlReader.parse(new InputSource(deletionFile.toURI().toURL().toString()));

            return deletionHandler.getDeletionCount();

        } catch (Exception e) {
            throw new IndexException(e);
        }
    }

    /**
     * A SAX handler used to parse a deletion file.
     */
    private class IdDeletionHandler extends DefaultHandler {
        boolean isID;

        int count = 0;

        public IdDeletionHandler() throws Exception {
            reader = IndexReader.open(baseIndexDir + indexDir);
        };

        /**
         * {@inheritDoc}
         */
        public void startElement(String ns, String sname, String qName,
                Attributes atts) {
            isID = qName != null && qName.trim().equalsIgnoreCase("id");
        }

        /**
         * {@inheritDoc}
         */
        public void characters(char[] ch, int start, int length) {
            if (isID) {
                String id = new String(ch, start, length);
                id = id.trim();
                if (!id.equals("")) {
                    try {
                        Term idTerm = new Term("objectid", id.trim());
                        reader.deleteDocuments(idTerm);
                        count++;
                    } catch (Exception e) {
                    }
                }
            }
        }

        public int getDeletionCount() {
            return count;
        }
    }

    /**
     * {@inheritDoc}
     */
    public synchronized void closeIndex() throws IndexException {
        try {
        	if (ramWriter != null) {
	            ramWriter.close();
	            Directory[] mergeDirs = { ramDir };
	            writer = createLuceneIndexWriter(baseIndexDir + indexDir, false);
	            writer.addIndexes(mergeDirs);
	            writer.close();
        	}
        } catch (java.io.IOException e) {
        	logger.error("Failed to merge active RamWriter with the persistent index.", e);
            throw new IndexException(e);
        }
    }

    /**
     * {@inheritDoc}
     */
    public void close() {
        ramWriter = null;
        writer = null;
    }

    /**
     * {@inheritDoc}
     */
    public synchronized void deleteIndex() throws IndexException {
        closeIndex();
        
        /* Wrap the index data directory in a File object */
    	File dataDir = new File(baseIndexDir + indexDir);
    	logger.info("Going to delete all files in: " + dataDir.getAbsolutePath());
    	if (dataDir.exists()) {
	    	/* Delete all the index data files */
	    	for (File f : dataDir.listFiles()) {
	    		f.delete();
	    		logger.info("Deleted file: " + f.getAbsolutePath());
	    	}
	    	
	    	/* Delete the index data directory */
	    	dataDir.delete();
	    	logger.info("Deleted directory: " + dataDir.getAbsolutePath());
    	}
    }

    /**
     * {@inheritDoc}
     */
    public File deleteDocuments(String[] documentIDs) throws IndexException {
        File deletionFile = createDeletionFile();

        deleteDocuments(documentIDs, true, deletionFile);

        return deletionFile;
    }

    /**
     * {@inheritDoc}
     */
    public int deleteDocumentsLocally(String[] documentIDs)
            throws IndexException {
        return deleteDocuments(documentIDs, false, null);
    }

    /**
     * A method used to delete documents from both the local, and upon request
     * the remote, Index.
     * 
     * @param documentIDs -
     *            The IDs of the documents to delete.
     * @param deleteFromMainIndex -
     *            An indicator of whether to delete the documetns from the Main
     *            Index as well as the local one.
     * @param deletionFile -
     *            The current Deletion File.
     * @return The number of documents deleted locally.
     * @throws IndexException
     */
    private int deleteDocuments(String[] documentIDs,
            boolean deleteFromMainIndex, File deletionFile)
            throws IndexException {
        try {
            // delete from this ram and saved index and add to deletion set
            reader = IndexReader.open(baseIndexDir + indexDir);
            ramReader = IndexReader.open(ramDir);
            HashSet<String> deletionSet = new HashSet<String>();

            Term idTerm;
            int deletedDocs = 0;
            for (String documentID : documentIDs) {
                idTerm = new Term("objectid", documentID);
                if (reader != null) {
                    deletedDocs += reader.deleteDocuments(idTerm);
                }
                if (ramReader != null) {
                    deletedDocs += ramReader.deleteDocuments(idTerm);
                }
                if (deleteFromMainIndex) {
                    deletionSet.add(documentID);
                }
            }
            // adds to the deletion file
            if (deleteFromMainIndex) {
                addToDeletionFile(deletionFile, deletionSet);
            }

            return deletedDocs;
        } catch (Exception e) {
            throw new IndexException();
        }
    }

    /**
     * Adds a set of document IDs to a deletion file.
     * 
     * @param delFile -
     *            The deletion to which to add the document IDs.
     * @param deletionSet -
     *            The set of IDs to add to the deletion file.
     * @throws Exception -
     *             An error occurred while writing to the Deletion File.
     */
    private void addToDeletionFile(File delFile, Set<String> deletionSet)
            throws Exception {
        if (deletionSet.size() == 0) {
            return;
        }
        RandomAccessFile deletionFile = new RandomAccessFile(delFile, "rw");
        long i;
        for (i = deletionFile.length() - 1; i >= 0; i--) {
            deletionFile.seek(i);
            String line = deletionFile.readLine();
            logger.info(line);
            if (line != null && line.equals("<IDList/>")) {
                deletionFile.seek(i);
                deletionFile.writeBytes("<IDList>");
                break;
            }
            if (line != null && line.equals("</IDList>")) {
                deletionFile.seek(i);
                break;
            }
        }
        if (i == -1) {
            deletionFile.seek(0);
            deletionFile
                    .writeBytes("<?xml version=\"1.0\" encoding=\"iso-8859-1\"?><IDList>");
        }
        for (String id : deletionSet) {
            deletionFile.writeBytes("<id>" + id + "</id>");
        }

        deletionFile.writeBytes("</IDList>");
        deletionFile.close();
    }

    /**
     * {@inheritDoc}
     */
    public int insertRowSet(String rowsetXML) throws IndexException {
        try {
            XMLProfileParser XMLparser = new XMLProfileParser();
            XMLparser.readString(rowsetXML, null);
            XMLparser.setRootNode("ROWSET");
            String[][] fieldData;
            int docCount = 0;
            while (XMLparser.setNextField()) {
                docCount++;
                fieldData = XMLparser.getSubFields();
                if (fieldData != null) {
                    // logger.info("** adding row from rowset");
                    Document doc = new Document();
                    //docContents - the sum of all the contents (except for the id the document, the col ID,
                    // the lang, and the full payload) 
                    String docContents = "";
                    //sum up the contents (if any) of each field and its subfields
                    for (int i = 0; i < idxType.getNumberOfFields(); i++) {

                        Field.Store store;
                        Field.Index index;
                        IndexField idxTypeField = idxType.getFields()[i];
                        if (idxTypeField.store)
                            store = Field.Store.YES;
                        else
                            store = Field.Store.NO;
                        if (idxTypeField.index) {
                            if (idxTypeField.tokenize)
                                index = Field.Index.TOKENIZED;
                            else
                                index = Field.Index.UN_TOKENIZED;
                        } else
                            index = Field.Index.NO;

                        String fieldContentSum = "";
                        for (int ii = 0; ii < fieldData[0].length; ii++) {
                            String baseName = fieldData[0][ii];
                            String fieldContents = fieldData[1][ii];

                            if ((idxTypeField.name).equals(baseName)
                                    && fieldContents != null) {
                            	
                            	//for the payload field we won't add its contents in the docContents field
                            	if(!baseName.equals(IndexType.PAYLOAD_FIELD))
                            	{
                            		docContents += " " + fieldContents;
                            	}
                                
                                fieldContentSum += " " + fieldContents;
                                
                            } else if (isDescendant(baseName, idxTypeField)
                                    && fieldContents != null) {
                                
                                fieldContentSum += " " + fieldContents;
                                
                            }
                        }
                        if(!fieldContentSum.equals("")) {
                        	doc.add(new Field(idxTypeField.name,
                                        fieldContentSum, store, index));
                        	//TODO: comment out after testing
                            logger.debug("added field in document: " + idxTypeField.name + ", " + fieldContentSum);
                        }
                    }
                    Field contents = new Field("_contents", docContents,
                            Field.Store.YES, Field.Index.TOKENIZED);
                    doc.add(contents);
                    //TODO: comment out after testing
                    logger.debug("added field in document: _contents, " + docContents);
                    
                    TokenStream tokens = NoStopAnalyser.tokenStream("_contents", new StringReader(docContents));
                    int wordCount = 0;
                    while(tokens.next() != null){
                        wordCount++;
                    }
                    doc.add(new Field("_wordcount", ""+wordCount,
                            Field.Store.YES, Field.Index.NO));
                    //TODO: comment out after testing
                    logger.debug("added field in document: _wordcount, " + wordCount);
                    

                    // Find the id for this document
                    for (int ii = 0; ii < fieldData[0].length; ii++) {
                        if (fieldData[0][ii].equalsIgnoreCase(IndexType.DOCID_FIELD) 
                                && fieldData[1][ii] != null) {

                            Field.Store store;
                            Field.Index index;
                            store = Field.Store.YES;
                            index = Field.Index.TOKENIZED;
                            String data = fieldData[1][ii];

                            doc.add(new Field(fieldData[0][ii].toLowerCase(),
                                    data, store, index));
                            //TODO: comment out after testing
                            logger.debug("added field in document: " + fieldData[0][ii] + ", " + data);
                        }
                    }
                    if (ramWriter != null) {
                        ramWriter.addDocument(doc);
                        ramWriter.flush();
                    } else
                        throw new IndexException("Index not open for insertion");
                }
            }
            return docCount;
        } catch (Exception e) {
        	logger.error("Error while inserting rowset.", e);
            throw new IndexException(e);
        }
    }

    /**
     * A method used to check if one field is the descendant of another.
     * 
     * @param descendantFieldName -
     *            The name of the field to check whether is the descendent.
     * @param parentField -
     *            The field to check whether is the ascendent.
     * @return TRUE if parentField contains a descendent with a name which
     *         equals (case insensitive) descendantFieldName.
     */
    private boolean isDescendant(String descendantFieldName,
            IndexField parentField) {
        IndexField childField;
        for (Iterator children = parentField.childrenFields.iterator(); children
                .hasNext();) {
            childField = ((IndexField) children.next());
            if (childField.name.equalsIgnoreCase(descendantFieldName))
                return true;
            else if (isDescendant(descendantFieldName, childField))
                return true;
        }
        return false;
    }

}
