package org.gcube.elasticsearch.helpers;

import static org.elasticsearch.common.xcontent.XContentFactory.jsonBuilder;
import static org.elasticsearch.index.query.QueryBuilders.termQuery;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.concurrent.TimeUnit;

import org.elasticsearch.ElasticSearchException;
import org.elasticsearch.action.admin.indices.alias.get.IndicesGetAliasesResponse;
import org.elasticsearch.action.admin.indices.delete.DeleteIndexRequest;
import org.elasticsearch.action.admin.indices.delete.DeleteIndexResponse;
import org.elasticsearch.action.admin.indices.exists.indices.IndicesExistsRequest;
import org.elasticsearch.action.admin.indices.flush.FlushResponse;
import org.elasticsearch.action.admin.indices.refresh.RefreshResponse;
import org.elasticsearch.action.bulk.BulkRequestBuilder;
import org.elasticsearch.action.count.CountRequestBuilder;
import org.elasticsearch.action.count.CountResponse;
import org.elasticsearch.action.deletebyquery.DeleteByQueryRequestBuilder;
import org.elasticsearch.action.deletebyquery.DeleteByQueryResponse;
import org.elasticsearch.action.get.MultiGetItemResponse;
import org.elasticsearch.action.get.MultiGetRequestBuilder;
import org.elasticsearch.action.get.MultiGetResponse;
import org.elasticsearch.action.index.IndexRequestBuilder;
import org.elasticsearch.action.index.IndexResponse;
import org.elasticsearch.action.search.SearchRequestBuilder;
import org.elasticsearch.action.search.SearchResponse;
import org.elasticsearch.action.search.SearchType;
import org.elasticsearch.client.Client;
import org.elasticsearch.common.unit.TimeValue;
import org.elasticsearch.common.xcontent.XContentBuilder;
import org.elasticsearch.index.query.QueryBuilders;
import org.elasticsearch.search.SearchHit;
import org.gcube.indexmanagement.common.FullTextIndexType;
import org.gcube.indexmanagement.common.IndexField;
import org.gcube.indexmanagement.common.IndexType;
import org.gcube.indexmanagement.common.XMLProfileParser;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;


public class ElasticSearchHelper {
	private static final Logger logger = LoggerFactory.getLogger(ElasticSearchHelper.class);

	private static long SCROLL_TIMEOUT = 100000;
	private static int MAX_RESULTS = 2000;
	
	
	private static final String HIGHLIGHT_PRE_TAG = "&lt;b&gt;";
	private static final String HIGHLIGHT_POST_TAG = "&lt;/b&gt;";
	
	public static final String SECURITY_FIELD = "sid";
	
	
//	@SuppressWarnings("unused")
//	private static Analyzer NoStopAnalyser = new StandardAnalyzer(Version.LUCENE_CURRENT);

	public static void commit(Client client, String indexName) {
		try
		{
			logger.info("flush request: " + indexName);
			long before = System.currentTimeMillis();
			FlushResponse flushResponse = client.admin().indices().prepareFlush(indexName).execute().actionGet();
			long after = System.currentTimeMillis();
			logger.info("Time for the flush request : " + (after - before) / 1000.0 + " secs");
			logger.info("flush response  failed shards: " + flushResponse.getFailedShards());
			logger.info("refresh request : " + indexName);
			before = System.currentTimeMillis();
			RefreshResponse refreshResponse = client.admin().indices().prepareRefresh(indexName).execute().actionGet();
			after = System.currentTimeMillis();
			logger.info("Time for the flush request : " + (after - before) / 1000.0 + " secs");
			logger.info("refresh response failed shards : " + refreshResponse.getFailedShards());
		}
		catch(Exception e)
		{
			logger.error("Exception while commiting:",e);
		}
	}

	

	public static SearchHit[] queryElasticSearch(Client client, String indexName, String queryString, int maxHits, List<String> projections) {
		return queryElasticSearch(client, indexName, queryString, maxHits, null, projections, 0, 0, 0);
	}
	
	public static SearchHit[] queryElasticSearch(Client client, String indexName, String queryString, int maxHits, int from, List<String> projections) {
		return queryElasticSearch(client, indexName, queryString, maxHits, null, projections, 0, 0, from);
	}

	public static SearchHit[] queryElasticSearch(Client client, String indexName, String queryString, int maxHits,
			List<String> highlightedFields, List<String> projections, int maxFragmentSize, int maxFragmentCnt, int from) {

		queryString = "{\"query_string\" : { \"query\" : \"" + queryString + "\" }}";

		SearchRequestBuilder srb = client.prepareSearch(indexName)
				.setSearchType(SearchType.DFS_QUERY_THEN_FETCH)
				.setSize(MAX_RESULTS)
				.setFrom(from)
				.setQuery(queryString);
		
		if (projections != null && projections.size() > 0)
			srb.addFields(projections.toArray(new String[projections.size()]));
		
		if (projections == null || projections.contains(IndexType.DOCID_FIELD) == false)
			srb.addField(IndexType.DOCID_FIELD); // standard fields for results
		
		if (highlightedFields != null) {
			for (String hilightedField : highlightedFields)
				if (!SnippetsHelper.NOT_HIGHLIGHTED_FIELDS.contains(hilightedField))
					srb.addHighlightedField(hilightedField, maxFragmentSize, maxFragmentCnt);
			srb.setHighlighterOrder("score");
			srb.setHighlighterPreTags(HIGHLIGHT_PRE_TAG);
			srb.setHighlighterPostTags(HIGHLIGHT_POST_TAG);
		}

		if (maxHits > 0)
			srb.setSize(maxHits);

		logger.trace("query request : " + srb.toString());
		SearchResponse response = srb.execute().actionGet();
		logger.info("query time : " + response.getTookInMillis());
		logger.trace("query response : " + response);

		return response.getHits().getHits();
	}

	public static SearchResponse queryElasticSearchScroll(Client client, String indexName, String queryString,
			int maxHits, List<String> projections) {
		return queryElasticSearchScroll(client, indexName, queryString, maxHits, null, projections, 0, 0);
	}

	/* TODO: fix score problem */
	public static SearchResponse queryElasticSearchScroll(Client client, String indexName, String queryString,
			int maxHits, List<String> highlightedFields, List<String> projections, int maxFragmentSize, int maxFragmentCnt) {

		queryString = "{\"query_string\" : { \"query\" : \"" + queryString + "\" }}";

		SearchRequestBuilder srb = client.prepareSearch(indexName).setTrackScores(true)
				// .setExplain(true)
				.setSearchType(SearchType.DFS_QUERY_THEN_FETCH).setScroll(TimeValue.timeValueMillis(SCROLL_TIMEOUT))
				.setQuery(queryString);

		if (projections != null && projections.size() > 0)
			srb.addFields(projections.toArray(new String[projections.size()]));
		
		if (maxHits > 0)
			srb.setSize(maxHits);

		if (highlightedFields != null) {
			for (String hilightedField : highlightedFields)
				if (!SnippetsHelper.NOT_HIGHLIGHTED_FIELDS.contains(hilightedField))
					srb.addHighlightedField(hilightedField, maxFragmentSize, maxFragmentCnt);
			srb.setHighlighterOrder("score");
			
			srb.setHighlighterPreTags(HIGHLIGHT_PRE_TAG);
			srb.setHighlighterPostTags(HIGHLIGHT_POST_TAG);
		}

		logger.info("query request : " + srb.toString());
		SearchResponse scrollResponse = srb.execute().actionGet();
		logger.info("query response : " + scrollResponse);

		return scrollResponse;
	}
	
	public static Long collectionDocumentsCountElasticSearch(Client client, String indexName, String collectionID) {
		
//		CountRequestBuilder countRequest = client.prepareCount(indexName).setQuery(termQuery(IndexType.COLLECTION_FIELD, collectionID));
//
//		logger.info("query request : " + countRequest.request());
//		CountResponse countResponse = countRequest.execute().actionGet();
//		logger.info("query response : " + countResponse);
		
		String queryString = IndexType.COLLECTION_FIELD + " == " + collectionID;

		Long count = queryCountElasticSearch(client, indexName, queryString);
				
		return count;
		
	}

	public static Long queryCountElasticSearch(Client client, String indexName, String queryString) {

		queryString = "{\"query_string\" : { \"query\" : \"" + queryString + "\" }}";

		CountRequestBuilder countRequest = client.prepareCount(indexName).setQuery(queryString.getBytes());

		logger.info("query request : " + countRequest.request());
		CountResponse countResponse = countRequest.execute().actionGet();
		logger.info("query response : " + countResponse);

		return countResponse.getCount();
	}

	public static SearchResponse getNextSearchResponse(Client client, SearchResponse scrollResponse) {
		scrollResponse = client.prepareSearchScroll(scrollResponse.getScrollId())
				.setScroll(TimeValue.timeValueMillis(SCROLL_TIMEOUT)).execute().actionGet();

		logger.info("scroll response : " + scrollResponse);

		return scrollResponse;
	}
	
	public static boolean delete(Client client, String indexName) {
		logger.info("deleting index : " + indexName);
		
//		Map<String, IndexMetaData> map = client.admin().cluster().state(new ClusterStateRequest()).actionGet(30, TimeUnit.SECONDS).getState().getMetaData().getIndices();
//		boolean exists = map.containsKey(indexName);
		
		boolean exists = client.admin().indices().exists(new IndicesExistsRequest(indexName)).actionGet().isExists();
		
		
		if (!exists) {
			logger.info("index : " + indexName + " does not exist");
			return true;
		} else {
			logger.info("index : " + indexName + " exists");
			DeleteIndexResponse delete = client.admin().indices().delete(new DeleteIndexRequest(indexName)).actionGet(30, TimeUnit.SECONDS);
			if (!delete.isAcknowledged()) {
				logger.error("Index wasn't deleted");
			}
			
			logger.info("deleted index : " + indexName + " ? " + delete.isAcknowledged());
			
			return delete.isAcknowledged();
		}
	}

	public static void deleteDocuments(Client client, String indexName, List<String> docIDs) {
		logger.info("docIds to be deleted : " + docIDs);

		for (String docID : docIDs) {
			logger.info("deleting document with " + IndexType.COLLECTION_FIELD + " : " + docID);

			DeleteByQueryRequestBuilder requestBuilder = client.prepareDeleteByQuery(indexName).setQuery(
					termQuery(IndexType.DOCID_FIELD, docID));
			logger.info("delete request : " + requestBuilder.request());
			requestBuilder.execute().actionGet();
		}
		commit(client, indexName);
	}
	
	
	
	public static Boolean deleteCollections(Client client, String indexName, Collection<String> colls){
		Boolean ret = true;
		for (String coll : colls){
			Boolean collDelete = deleteCollection(client, indexName, coll);
			if (!collDelete) {
				logger.warn("couldn't delete collection : " + coll);
				ret = false;
			}
		}
		return ret;
	}
	
	public static Boolean deleteCollection(Client client, String indexName, String collID) {
		logger.info("collId to be deleted : " + collID);

		logger.info("deleting document with " + IndexType.COLLECTION_FIELD + " : " + collID);

		DeleteByQueryRequestBuilder requestBuilder = client.prepareDeleteByQuery(indexName).setQuery(
				termQuery(IndexType.COLLECTION_FIELD, collID));
		logger.info("delete request : " + requestBuilder.request());
		DeleteByQueryResponse response = requestBuilder.execute().actionGet();
		
		logger.info("delete response : " + response.toString());
		
		commit(client, indexName);
		
		return true;
	}
	
	public static Boolean collectionExists(Client client, String indexName, String collID) {
		logger.info("collId to be checked : " + collID);

		logger.info("checking collection with " + IndexType.COLLECTION_FIELD + " : " + collID);

		SearchRequestBuilder requestBuilder = client.prepareSearch(indexName).setQuery(
				termQuery(IndexType.COLLECTION_FIELD, collID));
		logger.info("search request : " + requestBuilder.request());
		SearchResponse response = requestBuilder.execute().actionGet();
		
		if (response.getHits().getTotalHits() > 0){
			return true;
		} else {
			return false;
		}
		
	}
	
	
	public static Set<String> indicesOfCollection(Client client, String indexName, String collID) {
		Set<String> indices = new HashSet<String>();
		
		logger.info("collId to be checked : " + collID);

		logger.info("checking collection with " + IndexType.COLLECTION_FIELD + " : " + collID);

		SearchRequestBuilder requestBuilder = client.prepareSearch(indexName).setQuery(
				termQuery(IndexType.COLLECTION_FIELD, collID));
		logger.info("search request : " + requestBuilder.request());
		SearchResponse response = requestBuilder.execute().actionGet();
		
		for (SearchHit hit : response.getHits().getHits()){
			indices.add(hit.getIndex());
		}
		
		return indices;
	}
	
	public static Set<String> getIndicesOfAlias(Client client, String alias) {
		IndicesGetAliasesResponse aliasResponse = client.admin().indices().prepareGetAliases(alias).get();
		
		if (aliasResponse.getAliases() != null)
			return aliasResponse.getAliases().keySet();
		return null;
		
	}
	
	public static List<MultiGetItemResponse> getMultipleDocumentsOfAlias(Client client, String alias, Collection<String> docIDs) {
		Set<String> indicesOfAlias = getIndicesOfAlias(client, alias);
		if (indicesOfAlias == null)
			return new ArrayList<MultiGetItemResponse>();
		return getMultipleDocuments(client, indicesOfAlias, docIDs);
		
	}
	
	public static List<MultiGetItemResponse> getMultipleDocuments(Client client, Set<String> indices, Collection<String> docIDs) {
		List<MultiGetItemResponse> documents = new ArrayList<MultiGetItemResponse>(docIDs.size());
		
		for (String index : indices) {
			MultiGetRequestBuilder grb = new MultiGetRequestBuilder(client).add(index, null, docIDs);
			MultiGetResponse mgr = grb.get();
			
			for (MultiGetItemResponse response : mgr.getResponses()){
				if (response != null && response.getResponse() != null && response.getResponse().isExists()){
					documents.add(response);
					//System.out.println(response.getResponse().getSourceAsString());
				}
			}
		}
		
		return documents;
	}
	
	public static Set<String> getAllCollectionsOfIndex(Client client, String indexName) {
		logger.info("getting all collections of index : " + indexName);

		Set<String> collections = new HashSet<String>();
		SearchRequestBuilder requestBuilder = client.prepareSearch(indexName).setQuery(QueryBuilders.matchAllQuery()).addField(IndexType.COLLECTION_FIELD);
		SearchResponse sr = requestBuilder.execute().actionGet();
		for (SearchHit hit : sr.getHits().getHits()){
			String collection = hit.getFields().get(IndexType.COLLECTION_FIELD).getValue().toString();
			collections.add(collection);
		}

		return collections;
	}

	public static void clearIndex(Client client, String indexName, String indexTypeName) {
		logger.info("index to be deleted : ");
		DeleteByQueryResponse dr = client.prepareDeleteByQuery(indexName).setQuery(QueryBuilders.matchAllQuery())
				.setTypes(indexTypeName).execute().actionGet();
		logger.info("Delete response : " + dr.toString());

		commit(client, indexName);
	}

	public static void insertSimple(String jsonDoc, Client indexClient, String indexName, String indexType,
			Set<String> allowedIndexTypes) throws ElasticSearchException, IOException {

		if (!allowedIndexTypes.contains(indexType))
			throw new ElasticSearchException("index type : " + indexType + " is not in registered index types : "
					+ allowedIndexTypes);

		IndexResponse response = indexClient.prepareIndex(indexName, indexType).setSource(jsonDoc).execute()
				.actionGet();
		logger.info("indexResponse : " + response);
	}

	public static void insertBulk(BulkRequestBuilder bulkRequest, String jsonDoc, Client indexClient, String indexName,
			String indexType, Set<String> allowedIndexTypes) throws ElasticSearchException, IOException {

		if (!allowedIndexTypes.contains(indexType + "-" + indexName))
			throw new ElasticSearchException("index type : " + indexType + " is not in registered index types : "
					+ allowedIndexTypes);

		IndexRequestBuilder indexRequest = indexClient.prepareIndex(indexName, indexType).setSource(jsonDoc);
		bulkRequest.add(indexRequest);
		logger.info("indexRequest : " + indexRequest);
	}

	public static int insertRowSet(BulkRequestBuilder bulkRequest, Client client, String indexName,
			FullTextIndexType idxType, Set<String> allowedIndexTypes, String rowsetXML, Set<String> securityIdentifiers) {
		logger.trace("indexName : " + indexName);
		logger.trace("idxType : " + idxType.getIndexTypeName());
		logger.trace("allowedIndexTypes : " + allowedIndexTypes);
		logger.trace("rowsetXML : " + rowsetXML);

		int docCount = 0;
		try {
			XMLProfileParser XMLparser = new XMLProfileParser();
			try {
			XMLparser.readString(rowsetXML, null);
			} catch (Exception e) {
				logger.warn("problem in processing : " + rowsetXML);
				throw e;
			}
			XMLparser.setRootNode("ROWSET");
			String[][] fieldData;
			while (XMLparser.setNextField()) {
				docCount++;
				fieldData = XMLparser.getSubFields();
				if (fieldData != null) {
					Map<String, Object> document = new HashMap<String, Object>();

					document.put(SECURITY_FIELD, securityIdentifiers);
					// String docContents = "";

					for (IndexField idxTypeField : idxType.getFields()) {
						StringBuilder fieldContentSum = new StringBuilder();

						for (int ii = 0; ii < fieldData[0].length; ii++) {
							String baseName = fieldData[0][ii];
							String fieldContents = fieldData[1][ii];

							if ((idxTypeField.name).equals(baseName) && fieldContents != null) {

								// for the payload field we won't add its
								// contents in the docContents field
								// if
								// (!baseName.equals(IndexType.PAYLOAD_FIELD))
								// docContents += " " + fieldContents;

								fieldContentSum.append(fieldContents);
								fieldContentSum.append(" ");

							} else if (QueryParser.isDescendant(baseName, idxTypeField) && fieldContents != null) {
								fieldContentSum.append(fieldContents);
								fieldContentSum.append(" ");
							}
						}

						if (fieldContentSum.length() > 0) {
							// TODO: store & index

							document.put(idxTypeField.name, fieldContentSum.toString().trim());
							logger.trace("added field in document: " + idxTypeField.name + ", " + fieldContentSum);
						}
					}

					// TODO: store & index & tokenize
					// document.put("_contents", docContents);
					//
					// logger.info("added field in document: _contents, "
					// + docContents);
					// TokenStream tokens = NoStopAnalyser.tokenStream(
					// "_contents", new StringReader(docContents));
					//
					// int wordCount = 0;
					// while (tokens.incrementToken())
					// wordCount++;
					//
					// document.put("_wordcount", wordCount);
					//
					// logger.info("added field in document: _wordcount, "
					// + wordCount);

					for (int ii = 0; ii < fieldData[0].length; ii++) {
						if (fieldData[0][ii].equalsIgnoreCase(IndexType.DOCID_FIELD) && fieldData[1][ii] != null) {
							String data = fieldData[1][ii];

							// TODO:store & index
							// Store is done in FullTextNode addIndexType
//							Map<String, String> field = new HashMap<String, String>();
//							field.put("value", data);
//							field.put("store", "yes");
//							field.put("index", "analyzed");

							//document.put(fieldData[0][ii].toLowerCase(), data.trim());
							document.put(IndexType.DOCID_FIELD, data.trim());

							logger.trace("added field in document: " + fieldData[0][ii] + ", " + data);
						}
					}
					logger.trace("************* INSERT");
					insertBulk(bulkRequest, createJSONObject(document).string(), client, indexName,
							idxType.getIndexTypeName(), allowedIndexTypes);
				}
			}
		} catch (Exception e) {
			logger.warn("Exception while inserting documents", e);
		}

		return docCount;
	}

	public static XContentBuilder createJSONObject(Map<String, ? extends Object> keyValues) throws IOException {
		XContentBuilder xcb = jsonBuilder();

		xcb = xcb.startObject();
		for (Entry<String, ? extends Object> keyvalue : keyValues.entrySet())
			xcb = xcb.field(keyvalue.getKey(), keyvalue.getValue());

		return xcb.endObject();
	}
	
	public static String createKnownHostsString(Map<String,Integer> hosts) {
		boolean first = true;

		StringBuffer strBuf = new StringBuffer();		
		for (Map.Entry<String, Integer> entry : hosts.entrySet()) {
			if (!first) {
				strBuf.append(", ");
				first = false;
			}
			strBuf.append(entry.getKey());
//			strBuf.append(":");
//			strBuf.append(entry.getValue());
		}
		
		return strBuf.toString();
	}
}
