package org.gcube.elasticsearch.helpers;

import static org.elasticsearch.index.query.QueryBuilders.termQuery;
import gr.uoa.di.madgik.grs.buffer.IBuffer.Status;
import gr.uoa.di.madgik.grs.proxy.tcp.TCPWriterProxy;
import gr.uoa.di.madgik.grs.record.GenericRecord;
import gr.uoa.di.madgik.grs.record.GenericRecordDefinition;
import gr.uoa.di.madgik.grs.record.RecordDefinition;
import gr.uoa.di.madgik.grs.record.field.FieldDefinition;
import gr.uoa.di.madgik.grs.record.field.StringField;
import gr.uoa.di.madgik.grs.record.field.StringFieldDefinition;
import gr.uoa.di.madgik.grs.writer.GRS2WriterException;
import gr.uoa.di.madgik.grs.writer.RecordWriter;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.elasticsearch.action.search.SearchRequestBuilder;
import org.elasticsearch.action.search.SearchResponse;
import org.elasticsearch.client.Client;
import org.elasticsearch.search.SearchHit;
import org.gcube.common.core.scope.GCUBEScope;
import org.gcube.elasticsearch.FTNodeCache;
import org.gcube.indexmanagement.common.FullTextIndexType;
import org.gcube.indexmanagement.common.IndexException;
import org.gcube.indexmanagement.common.IndexField;
import org.gcube.indexmanagement.common.IndexType;
import org.gcube.indexmanagement.common.XMLProfileParser;
import org.gcube.indexmanagement.lucenewrapper.LuceneGcqlProcessor;
import org.gcube.indexmanagement.lucenewrapper.LuceneGcqlQueryContainer;
import org.gcube.indexmanagement.resourceregistry.RRadaptor;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class QueryParser {

	private static final Logger logger = LoggerFactory.getLogger(QueryParser.class);

	private static Set<String> getIndexTypesByCollectionID(Map<String, Set<String>> indexTypesByCollIDs, String collID,
			Client client, String indexName) {
		Set<String> indexTypes = null;

		if (indexTypesByCollIDs.containsKey(collID)) {
			indexTypes = indexTypesByCollIDs.get(collID);
			logger.info("indexTypes for collectionID found in cache");
		} else {
			logger.info("indexTypes for collectionID not found in cache");
			indexTypes = new HashSet<String>();

			SearchRequestBuilder srb = client.prepareSearch(indexName)
					.setQuery(termQuery(IndexType.COLLECTION_FIELD, collID)).setNoFields();

			logger.info("query request : " + srb.toString());

			SearchResponse response = srb.execute().actionGet();

			logger.info("query response : " + response);

			for (SearchHit hit : response.getHits().getHits())
				indexTypes.add(hit.getType());
		}

		logger.info("for collectionID : " + collID + " indexTypes found : " + indexTypes);

		return indexTypes;
	}

	public static Set<String> getIndexTypesByCollectionIDs(Map<String, Set<String>> indexTypesByCollIDs,
			List<String> collIDs, Client client, String indexName) {
		Set<String> indexTypes = new HashSet<String>();

		for (String collID : collIDs)
			indexTypes.addAll(getIndexTypesByCollectionID(indexTypesByCollIDs, collID, client, indexName));

		return indexTypes;
	}

	public static List<String> createPresentableForIndexTypes(Map<String, List<String>> presentableFieldsPerIndexType,
			Set<String> indexTypes) {
		List<String> presentables = new ArrayList<String>();
		for (String idxType : indexTypes)
			presentables.addAll(presentableFieldsPerIndexType.get(idxType));

		logger.info("for indexTypes : " + indexTypes + " presentables found : " + presentables);
		return presentables;
	}
	
	public static List<String> createSearchablesForIndexTypes(Map<String, List<String>> searchablesFieldsPerIndexType,
			Set<String> indexTypes) {
		List<String> searchables = new ArrayList<String>();
		for (String idxType : indexTypes)
			searchables.addAll(searchablesFieldsPerIndexType.get(idxType));

		logger.info("for indexTypes : " + indexTypes + " searchables found : " + searchables);
		return searchables;
	}
	

	private static String regexCollID = IndexType.COLLECTION_FIELD + "\\s*==\\s*\"(\\S+)\"";
	private static Pattern patternCollID = Pattern.compile(regexCollID);

	public static List<String> getCollectionsIDFromQuery(String queryString) {
		List<String> matches = new ArrayList<String>();
		Matcher m = patternCollID.matcher(queryString);

		while (m.find())
			matches.add(m.group(1).trim());

		return matches;
	}
	
	public static FieldDefinition[] createFieldDefinition(List<String> presentable, List<String> projections, RRadaptor adaptor)
			throws Exception {
		ArrayList<FieldDefinition> fieldDef = new ArrayList<FieldDefinition>();
		// add three more fields for the score, the statistics and the docID
		fieldDef.add(new StringFieldDefinition(IndexType.SCORE_FIELD));
		// fieldDef.add(new StringFieldDefinition(IndexType.STATS_FIELD));
		fieldDef.add(new StringFieldDefinition(IndexType.DOCID_FIELD));

		// these cases correspond to the way the worker fills the RS
		// the plus 3 fields are for score, stats and docID

		FieldDefinition[] fd = null;

		if (projections == null || projections.size() == 0) {
			fd = fieldDef.toArray(new FieldDefinition[fieldDef.size()]);
			logger.info("No projections found");
		} else {
			//TODO: fix code. 2 branches in one
			if (projections.contains(IndexType.WILDCARD)) {
				// return all the presentable fields (we assume that its the
				// updater's responsibility
				// to check for the fields to be returnable, stored) except for
				// the
				// full payload
				for (String fieldName : presentable) {

					String fieldID = adaptor.getFieldIDFromName(fieldName);
					// if a field is not the ObjectID or full payload field
					
					if (!fieldName.equalsIgnoreCase(IndexType.DOCID_FIELD)
							&& !fieldName.equalsIgnoreCase(IndexType.PAYLOAD_FIELD)) {
						fieldDef.add(new StringFieldDefinition(fieldID));
					}
				}
			} else {
				for (String fieldName : projections) {
					String fieldID = adaptor.getFieldIDFromName(fieldName);
					// if a field is not the ObjectID or full payload field
					
					if (!fieldName.equalsIgnoreCase(IndexType.DOCID_FIELD)
							&& !fieldName.equalsIgnoreCase(IndexType.PAYLOAD_FIELD)) {
						fieldDef.add(new StringFieldDefinition(fieldID));
					}
					
//					String fieldID = adaptor.getFieldIDFromName(current);
//					fieldDef.add(new StringFieldDefinition(fieldID));
				}
			}
		}
		// fieldDef.add(new StringFieldDefinition(IndexType.PAYLOAD_FIELD));

		fd = fieldDef.toArray(new FieldDefinition[fieldDef.size()]);

		return fd;
	}

	/**
	 * A method used to check if one field is the descendant of another.
	 * 
	 * @param descendantFieldName
	 *            - The name of the field to check whether is the descendent.
	 * @param parentField
	 *            - The field to check whether is the ascendent.
	 * @return TRUE if parentField contains a descendent with a name which
	 *         equals (case insensitive) descendantFieldName.
	 */
	public static boolean isDescendant(String descendantFieldName, IndexField parentField) {
		IndexField childField;
		for (Iterator<?> children = parentField.childrenFields.iterator(); children.hasNext();) {
			childField = ((IndexField) children.next());
			if (childField.name.equalsIgnoreCase(descendantFieldName))
				return true;
			else if (isDescendant(descendantFieldName, childField))
				return true;
		}
		return false;
	}

	public static String createIndexTypekey(String indexType, String scope) {
		return indexType + "_" + scope;
	}

	public static FullTextIndexType retrieveIndexType(String indexTypeStr, String scope, FTNodeCache cache) {
		GCUBEScope gcubeScope = GCUBEScope.getScope(scope);
		logger.info("gcubeScope : " + gcubeScope);
		return retrieveIndexType(indexTypeStr, gcubeScope, cache);
	}

	public static FullTextIndexType retrieveIndexType(String indexTypeStr, GCUBEScope gcubeScope, FTNodeCache cache) {

		FullTextIndexType indexType = null;
		String scope = gcubeScope.toString();

		logger.info("Retrieving index type for : " + indexTypeStr);
		if (cache.cachedIndexTypes.containsKey(createIndexTypekey(indexTypeStr, scope))) {
			logger.info("Index type : " + indexTypeStr + " found in cache");
			indexType = cache.cachedIndexTypes.get(createIndexTypekey(indexTypeStr, scope));
		} else {
			logger.info("Index type : " + indexTypeStr + " NOT found in cache");

			indexType = new FullTextIndexType(indexTypeStr, gcubeScope);
			
			logger.info("Retrieved from IS indextype : " + indexType);
			
			addFullTextIndexTypeIntoCache(indexTypeStr, scope, indexType, cache);
		}
		// logger.info("Index type returned : " + indexType);

		return indexType;
	}

	public static FullTextIndexType addFullTextIndexTypeIntoCache(String indexTypeStr, String scope,
			FullTextIndexType indexType, FTNodeCache cache) {

		if(cache.cachedIndexTypes.containsKey(createIndexTypekey(indexTypeStr, scope)))
			return indexType;
		logger.info("Index type : " + indexTypeStr + "adding into cache");
		cache.cachedIndexTypes.put(createIndexTypekey(indexTypeStr, scope), indexType);

		logger.info("Cache : " + cache.cachedIndexTypes);
		return indexType;
	}

	public static boolean writeSearchHitInResultSet(SearchHit hit, RecordWriter<GenericRecord> rsWriter,
			List<String> projections, List<String> presentables, int maxFragmentCount, long rsTimeout)
			throws GRS2WriterException {
		if (rsWriter.getStatus() != Status.Open)
			return false;
		// the current RS record
		GenericRecord rec = new GenericRecord();
		// the fields for this record
		ArrayList<gr.uoa.di.madgik.grs.record.field.Field> fields = new ArrayList<gr.uoa.di.madgik.grs.record.field.Field>();

		// TODO: other statistics? wc? terms?

		Map<String, Object> docMap = hit.getSource();
		logger.info("Hit from index : ");
		logger.info("-------------------------------------");
		if (logger.isInfoEnabled())
			for (Entry<String, Object> f : docMap.entrySet())
				logger.info(f.getKey() + ":" + f.getValue());
		logger.info("-------------------------------------");
		// field 0 is the score

		logger.info("Adding score field with value : " + hit.getScore());
		
		fields.add(new StringField(String.valueOf(hit.getScore())));
		// TODO: can we remove this?
		//String docStatistics = "<docStatistics><rank>" + String.valueOf(hit.getScore()) + "</rank></docStatistics>";
		//fields.add(new StringField(docStatistics));

		String fieldContentDocID = docMap.containsKey(IndexType.DOCID_FIELD.toLowerCase()) ? docMap.get(
				IndexType.DOCID_FIELD.toLowerCase()).toString() : "NoMetaId";// TODO:
																				// this
																				// is

		logger.info("Adding " + IndexType.DOCID_FIELD.toLowerCase() + " field with value : " + fieldContentDocID);
		fields.add(new StringField(fieldContentDocID));

		// fields.add(new
		// StringField(XMLTokenReplacer.XMLResolve(doc.get(IndexType.PAYLOAD_FIELD))));
		// fields.add(new StringField("dummy payload"));

		if (projections != null && projections.size() > 0) {
			List<String> returnFields = null;
			// in case there is the wildcard in projections
			if (projections.contains(IndexType.WILDCARD))
				returnFields = presentables;
			else
				returnFields = projections;

			logger.info("returnFields : " + returnFields);

			for (String fieldName : returnFields) {
				if (fieldName.equalsIgnoreCase(IndexType.DOCID_FIELD)
						|| fieldName.equalsIgnoreCase(IndexType.PAYLOAD_FIELD))
					continue;

				String fieldContent = null;

				if (fieldName.equals(IndexType.SNIPPET))
					fieldContent = SnippetsHelper.createSnippetString(hit, maxFragmentCount);
				else
					fieldContent = docMap.containsKey(fieldName) ? docMap.get(fieldName).toString() : "";

				// TODO: do we need this???
				fieldContent = XMLProfileParser.escapeForXML(fieldContent);

				logger.info("adding field : " + fieldName + " with value : " + fieldContent);
				fields.add(new StringField(fieldContent));
			}
		}

		// while the reader hasn't stopped reading
		if (rsWriter.getStatus() != Status.Open)
			return false;

		// set the fields in the record
		rec.setFields(fields.toArray(new gr.uoa.di.madgik.grs.record.field.Field[fields.size()]));

		while (!rsWriter.put(rec, rsTimeout, TimeUnit.SECONDS)) {
			// while the reader hasn't stopped reading
			if (rsWriter.getStatus() != Status.Open)
				break;
		}

		return true;
	}

	public static RecordWriter<GenericRecord> initRSWriterForSearchHits(List<String> presentables,
			List<String> projections, RRadaptor adaptor) throws IndexException, GRS2WriterException {
		logger.info("Initializing gRS2 writer");
		logger.info("(1/3) getting field definitions");
		FieldDefinition[] fieldDef = null;
		try {
			fieldDef = createFieldDefinition(presentables, projections, adaptor);
		} catch (Exception e) {
			logger.error("Could not create field definition: ", e);
			throw new IndexException(e);
		}

		logger.info("(2/3) creating record definitions");
		RecordDefinition[] definition = new RecordDefinition[] { new GenericRecordDefinition(fieldDef) };

		logger.info("(3/3) creating rsWriter");
		return new RecordWriter<GenericRecord>(new TCPWriterProxy(), definition);

	}
	
	public static LuceneGcqlQueryContainer convertToLuceneQuery(String query, List<String> presentables, List<String> seachables, RRadaptor rradaptor) throws IndexException {
		LuceneGcqlQueryContainer queryContainer = null;
		
		try {
			queryContainer = (LuceneGcqlQueryContainer) new LuceneGcqlProcessor().processQuery((ArrayList<String>) presentables, (ArrayList<String>) seachables,
					query, rradaptor);
		} catch (Exception e) {
			e.printStackTrace();
			logger.warn("Error while converting the query to lucene query : ", e);
		}
		return queryContainer;
	}
	
	public static String getLuceneQueryFromQueryString(LuceneGcqlQueryContainer queryContainer) throws IndexException {
		if (queryContainer.getLuceneQuery() == null || queryContainer.getLuceneQuery().query == null)
			throw new IndexException("Error while getting the query part of the lucene query. queryContainer : " + queryContainer.getLuceneQuery());
		String query = queryContainer.getLuceneQuery().query;
		return query.replace("\"", "\\\"");
	}
	
	
	
	public static List<String> getProjectionsQueryFromQueryString(LuceneGcqlQueryContainer queryContainer) throws IndexException {
		if (queryContainer.getProjectedFields() == null || queryContainer.getProjectedFields().size() == 0)
			throw new IndexException("Error while getting the projections from the lucene query. Projected fields : " + queryContainer.getProjectedFields());
		List<String> projections = new ArrayList<String>(queryContainer.getProjectedFields().values());
		return projections;
	}
	
}
