package org.gcube.couchbase.entities;

import java.io.Serializable;
import java.io.StringReader;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;

import net.spy.memcached.util.StringUtils;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;
import org.xml.sax.InputSource;

import com.google.gson.Gson;

/**
 * Simple parsing of the Rowsets that are produced from the gDTS and are fed to
 * the index.
 * 
 * After the parsing collections of the keys and fields are kept as
 * ForwardIndexDocument that can be easily transformed to JSON so they can be
 * stored in Couchbase.
 * 
 * 
 * @author Alex Antoniadis
 * 
 */
public class ForwardIndexDocument implements Serializable {

	private static final long serialVersionUID = 1L;
	private static final Logger logger = LoggerFactory.getLogger(ForwardIndexDocument.class);
	private static Gson gson = new Gson();

	private static final String ELEMENT_KEY = "KEY";
	private static final String ELEMENT_KEYNAME = "KEYNAME";
	private static final String ELEMENT_KEYVALUE = "KEYVALUE";
	private static final String ELEMENT_VALUE = "VALUE";
	private static final String ELEMENT_FIELD = "FIELD";
	private static final String ATTRIBUTE_FIELDNAME = "name";

	private static final String ID_KEY_FIELD = "ObjectID";
	private static final String COLLECTION_FIELD = "gDocCollectionID";
	
	public static final String LANGUAGE_FIELD = "gDocCollectionLang";
	public static final String LANG_UNKNOWN = "unknown";
	public static final String DOCID_FIELD = "ObjectID";

	private Map<String, ArrayList<String>> keys = null;
	private Map<String, ArrayList<String>> fields = null;

	public ForwardIndexDocument() {
		this.keys = new HashMap<String, ArrayList<String>>();
		this.fields = new HashMap<String, ArrayList<String>>();
	}

	public ForwardIndexDocument(String rowset) throws Exception {
		this();
		this.parseXML(rowset);
	}

	public void parseXML(String rowset) throws Exception {
		parseXML(this.getKeys(), this.getFields(), rowset);
	}

	public static void parseXML(Map<String, ArrayList<String>> keys, Map<String, ArrayList<String>> fields,
			String rowset) throws Exception {
		Document doc = loadXMLFromString(rowset);
		doc.getDocumentElement().normalize();

		long before = System.currentTimeMillis();

		NodeList keyNodeList = doc.getElementsByTagName(ELEMENT_KEY);

		logger.info("xml key nodelist length : " + keyNodeList.getLength());
		
		for (int i = 0; i < keyNodeList.getLength(); i++) {
			Element keyElement = (Element) keyNodeList.item(i);

			String keyName = keyElement.getElementsByTagName(ELEMENT_KEYNAME).item(0).getTextContent().trim();
			String keyValue = keyElement.getElementsByTagName(ELEMENT_KEYVALUE).item(0).getTextContent().trim();

			//empty language case
			if (keyName.equalsIgnoreCase(LANGUAGE_FIELD) && (keyValue == null ||  keyValue.trim().length() == 0)){
				keyValue = LANG_UNKNOWN;
			}
				
			if (keyValue.length() == 0)
				continue;

			if (keys.containsKey(keyName)) {
				keys.get(keyName).add(keyValue);
			} else {
				ArrayList<String> l = new ArrayList<String>();
				l.add(keyValue);
				keys.put(keyName, l);
			}
		}
		keyNodeList = null;

		Element valueElement = (Element) doc.getElementsByTagName(ELEMENT_VALUE).item(0);
		NodeList fieldList = valueElement.getElementsByTagName(ELEMENT_FIELD);
		
		logger.info("xml field nodelist length : " + fieldList.getLength());
		
		for (int i = 0; i < fieldList.getLength(); i++) {
			Element field = (Element) fieldList.item(i);

			String fieldName = field.getAttribute(ATTRIBUTE_FIELDNAME).trim();
			String fieldValue = field.getTextContent().trim();

			//empty language case
			if (fieldName.equalsIgnoreCase(LANGUAGE_FIELD) && (fieldValue == null ||  fieldValue.trim().length() == 0)){
				fieldValue = LANG_UNKNOWN;
			}
			
			if (fieldValue.length() == 0)
				continue;

			if (fields.containsKey(fieldName)) {
				fields.get(fieldName).add(fieldValue);
			} else {
				ArrayList<String> l = new ArrayList<String>();
				l.add(fieldValue);
				fields.put(fieldName, l);
			}
		}
		
		long after = System.currentTimeMillis();
		logger.info("parse xml after : " + (after - before) / 1000.0 + " secs");
	}

	public String getID() {
		// return calculateID(this.keys);
		return getDocId();
	}

	public String getDocLang() {
		ArrayList<String> langs = this.getKeys().get(LANGUAGE_FIELD);
		if (langs == null) {
			logger.info("Lang is null for doc with id : " + this.getID());
			return LANG_UNKNOWN;
		} else if (langs.size() == 0) {
			logger.info("No languages found for doc with id : " + this.getID());
			return LANG_UNKNOWN;
		} else if (langs.size() > 1) {
			logger.info("Multiple languages found for doc with id : " + this.getID());
			logger.info("languages are : " + langs + " picking first...");
			return langs.get(0);
		} else {
			logger.info("language found for doc with id : " + this.getID());
			logger.info("language is : " + langs + " picking first...");
			return langs.get(0);
		}
	}

	public String getDocId() {
		ArrayList<String> docIDs = this.getKeys().get(DOCID_FIELD);
		if (docIDs == null) {
			logger.info("DocID is null for doc");
			return null;
		} else if (docIDs.size() != 1) {
			logger.info("0 or more than 1 docIDs found for doc");
			logger.info("docIDs are : " + docIDs);
			return null;
		} else {
			logger.info("DocID found for doc");
			logger.info("docID is : " + docIDs + " picking first...");
			return docIDs.get(0);
		}
	}

	public String getColId() {
		ArrayList<String> colIDs = this.getKeys().get(COLLECTION_FIELD);
		if (colIDs == null) {
			logger.info("ColId is null for doc with id : " + this.getID());
			return null;
		} else if (colIDs.size() != 1) {
			logger.info("0 or more than 1 colIDs found for doc with id : " + this.getID());
			logger.info("colIDs are : " + colIDs);
			return null;
		} else {
			logger.info("ColId found for doc with id : " + this.getID());
			logger.info("colIDs is : " + colIDs + " picking first...");
			return colIDs.get(0);
		}
	}

	@SuppressWarnings("unused")
	private static String calculateID(Map<String, ArrayList<String>> keys) {
		return StringUtils.join(keys.get(ID_KEY_FIELD), "_");
	}

	@Override
	public String toString() {
		return "ID : " + this.getID() + ", values : " + this.toJSON();
	};

	// value is the whole document keys + fields

	public String toJSON() {
		return gson.toJson(this);
	}

	// for each key in keys a view should be created with name
	// KEYNAME_DATATYPE_view
	// in the initialization the views can be determined

	public static Document loadXMLFromString(String xml) throws Exception {
		long before = System.currentTimeMillis();
		
		DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
		DocumentBuilder builder = factory.newDocumentBuilder();
		InputSource is = new InputSource(new StringReader(xml));
		Document doc = builder.parse(is);

		long after = System.currentTimeMillis();
		logger.info("load xml after : " + (after - before) / 1000.0 + " secs");
		
		return doc;
	}

	public Map<String, ArrayList<String>> getKeys() {
		return keys;
	}

	public Map<String, ArrayList<String>> getFields() {
		return fields;
	}
	
	
	/*public static void main(String[] args) throws GRS2ReaderException, URISyntaxException {
		String resultSetLocation = "grs2-proxy://jazzman.di.uoa.gr:52679?key=efbc794e-176f-43bf-b676-3b18c1536eb3#TCP";
		long RSTIMEOUT = 5;
		logger.info("Initializing reader at resultset : " + resultSetLocation);
		ForwardReader<Record> reader = new ForwardReader<Record>(new URI(resultSetLocation));
		reader.setIteratorTimeout(RSTIMEOUT);
		reader.setIteratorTimeUnit(TimeUnit.MINUTES);
		
		try {
			logger.info("Initializing resultset reader iterator");
			Iterator<Record> it = reader.iterator();
			long before, after;
			int rowSetCount = 1;
			while (it.hasNext()) {
				logger.info("Getting result : " + rowSetCount);
				
				before = System.currentTimeMillis();
				Record result = it.next();
				String rowset = RowsetParser.getRowsetFromResult(result);
				ForwardIndexDocument doc = new ForwardIndexDocument(rowset);
				System.out.println("doc : " + doc.getDocId());
				
				after = System.currentTimeMillis();
				logger.info("Time for parsing rowset from record : " + (after - before) / 1000.0 + " secs");
			}
		}catch (Exception e) {
			logger.info("Exception while feeding", e);
		}
	}*/
}
