package gr.uoa.di.madgik.grs.bridge.xml;

import java.io.DataInput;
import java.io.DataOutput;
import java.io.IOException;
import java.util.Hashtable;

import org.gcube.common.searchservice.searchlibrary.resultset.elements.ResultElementGeneric;
import org.gcube.common.searchservice.searchlibrary.resultset.helpers.RSConstants;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

import gr.uoa.di.madgik.grs.GRS2Exception;
import gr.uoa.di.madgik.grs.bridge.GCubeRecord;
import gr.uoa.di.madgik.grs.bridge.exceptions.GCubeBridgeException;
import gr.uoa.di.madgik.grs.buffer.GRS2BufferException;
import gr.uoa.di.madgik.grs.record.GRS2RecordDefinitionException;
import gr.uoa.di.madgik.grs.record.GRS2RecordSerializationException;
import gr.uoa.di.madgik.grs.record.field.Field;
import gr.uoa.di.madgik.grs.record.field.StringField;

/**
 * {@link GCubeRecord} implementation targeting a ResultSet Element that carries only an XML payload 
 * 
 * @author gpapanikos
 *
 */
public class GCubeXMLRecord extends GCubeRecord
{
	/**
	 * Default value for id, collection and rank
	 */
	public static final String DefaultValue="none";
	private String id=GCubeXMLRecord.DefaultValue;
	private String collection=GCubeXMLRecord.DefaultValue;
	private String rank=GCubeXMLRecord.DefaultValue;
	
	public static String RECORD_ID_NAME = ResultElementGeneric.RECORD_ID_NAME;
	public static String RECORD_COLLECTION_NAME = ResultElementGeneric.RECORD_COLLECTION_NAME;
	public static String RECORD_RANK_NAME = ResultElementGeneric.RECORD_RANK_NAME;
	
	/**
	 * Create a new instance
	 */
	public GCubeXMLRecord()
	{
		this.setFields(new Field[]{new StringField()});
	}
	
	/**
	 * Create a new instance
	 * 
	 * @param payload the payload
	 */
	public GCubeXMLRecord(String payload)
	{
		this.setFields(new Field[]{new StringField(payload)});
	}
	
	/**
	 * Create a new instance
	 * 
	 * @param id the object id
	 * @param collection the collection id
	 */
	public GCubeXMLRecord(String id,String collection)
	{
		this.setId(id);
		this.setCollection(collection);
		this.setFields(new Field[]{new StringField()});
	}
	
	/**
	 * Create a new instance
	 * 
	 * @param id the object id
	 * @param collection the collection id
	 * @param payload the payload
	 */
	public GCubeXMLRecord(String id,String collection,String payload)
	{
		this.setId(id);
		this.setCollection(collection);
		this.setFields(new Field[]{new StringField(payload)});
	}
	
	/**
	 * Create a new instance
	 * 
	 * @param id the object id
	 * @param collection the collection id
	 * @param rank the rank
	 * @param payload the payload
	 */
	public GCubeXMLRecord(String id,String collection,String rank,String payload)
	{
		this.setId(id);
		this.setCollection(collection);
		this.setRank(rank);
		this.setFields(new Field[]{new StringField(payload)});
	}

	/**
	 * Object ID value
	 * 
	 * @param id Object ID value
	 */
	public void setId(String id)
	{
		this.id = id;
	}

	/**
	 * Object ID value
	 * 
	 * @return Object ID value
	 */
	public String getId()
	{
		return id;
	}

	/**
	 * Collection ID value
	 * 
	 * @param collection Collection ID value
	 */
	public void setCollection(String collection)
	{
		this.collection = collection;
	}

	/**
	 * Collection ID value
	 * 
	 * @return Collection ID value
	 */
	public String getCollection()
	{
		return collection;
	}

	/**
	 * Rank value
	 * 
	 * @param rank rank value
	 */
	public void setRank(String rank)
	{
		this.rank = rank;
	}

	/**
	 * Rank value
	 * 
	 * @return rank value
	 */
	public String getRank()
	{
		return rank;
	}
	
	/**
	 * Utility function setting the {@link Field} payload without the client needing to locate it and set it there
	 * 
	 * @param payload the payload to set
	 * @throws GCubeBridgeException the payload field could not be located
	 */
	public void setPayload(String payload) throws GCubeBridgeException
	{
		try
		{
			this.getField(GCubeXMLRecordDefinition.PayloadFieldName).setPayload(payload);
		} catch (GRS2Exception e)
		{
			throw new GCubeBridgeException("could not set the record's payload",e);
		}
	}
	
	/**
	 * Utility function returning the {@link Field} payload without the client needing to locate it and get it from there
	 * 
	 * @return the string payload
	 * @throws GCubeBridgeException the payload field could not be located
	 */
	public String getPayload() throws GCubeBridgeException
	{
		try
		{
			return this.getField(GCubeXMLRecordDefinition.PayloadFieldName).getPayload();
		} catch (GRS2Exception e)
		{
			throw new GCubeBridgeException("could not retrieve the record's payload",e);
		}
	}
	
	/**
	 * {@inheritDoc}
	 * 
	 * @see gr.uoa.di.madgik.grs.bridge.GCubeRecord#downgrade()
	 */
	@Override
	public String downgrade() throws GCubeBridgeException
	{
		//ResultElementBase copied code
		String xml=this.getPayload();
		StringBuilder validXML=new StringBuilder("<"+RSConstants.RecordTag);
		validXML.append(" "+ResultElementGeneric.RECORD_ID_NAME+"=\""+this.id+"\"");
		validXML.append(" "+ResultElementGeneric.RECORD_COLLECTION_NAME+"=\""+this.collection+"\"");
		validXML.append(" "+ResultElementGeneric.RECORD_RANK_NAME+"=\""+this.rank+"\"");
		if(xml==null || xml.trim().length() == 0) validXML.append("/>");
		else
		{
			validXML.append(">");
			if(xml!=null) validXML.append(xml);
			validXML.append("</"+RSConstants.RecordTag+">");
		}
		return validXML.toString();
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @see gr.uoa.di.madgik.grs.bridge.GCubeRecord#upgrade(java.lang.String)
	 */
	@Override
	public void upgrade(String record) throws GCubeBridgeException
	{
		//ResultElementBase copied code
		if(!(record.trim().startsWith("<"+RSConstants.RecordTag) && 
				(record.trim().endsWith("</"+RSConstants.RecordTag+">") || 
				record.trim().endsWith("/>")))) throw new GCubeBridgeException("record is not valid");
		String payload=null;
		String attrsS=null;
		Hashtable<String, String> attrs=new Hashtable<String, String>();
		if(record.endsWith("/>")){
			payload=null;
			attrsS=record.substring(("<".length()+RSConstants.RecordTag.length()),(record.length()-"/>".length()));
		}
		else{
			payload=record.substring((record.indexOf(">")+">".length()),record.lastIndexOf("</"+RSConstants.RecordTag+">"));
			if ( ("<".length()+RSConstants.RecordTag.length()) <= (record.indexOf(">")-">".length())){
				attrsS=record.substring(("<".length()+RSConstants.RecordTag.length()),(record.indexOf(">")-">".length()));
			}else{
				attrsS="";
			}
		}
		String []attrsSA=attrsS.split("\"");
		try{
			for(int i=0;i<attrsSA.length-1;i+=2){
				try{
					attrs.put(attrsSA[i].trim().substring(0,attrsSA[i].trim().lastIndexOf("=")), attrsSA[i+1]);
				}catch(Exception e){}
			}
		}catch(Exception e){}
		this.setPayload(payload != null ? payload : "");
		if(attrs.containsKey(ResultElementGeneric.RECORD_ID_NAME)) this.id=attrs.get(ResultElementGeneric.RECORD_ID_NAME);
		if(attrs.containsKey(ResultElementGeneric.RECORD_COLLECTION_NAME)) this.collection=attrs.get(ResultElementGeneric.RECORD_COLLECTION_NAME);
		if(attrs.containsKey(ResultElementGeneric.RECORD_RANK_NAME)) this.rank=attrs.get(ResultElementGeneric.RECORD_RANK_NAME);
	}
	
	/**
	 * {@inheritDoc}
	 * 
	 * @see gr.uoa.di.madgik.grs.record.Record#getField(java.lang.String)
	 */
	@Override
	public StringField getField(String name) throws GRS2RecordDefinitionException, GRS2BufferException
	{
		return (StringField)super.getField(name);
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @see gr.uoa.di.madgik.grs.record.Record#extendSend(java.io.DataOutput)
	 */
	@Override
	public void extendSend(DataOutput out) throws GRS2RecordSerializationException
	{
		try
		{
			out.writeUTF(this.id);
			out.writeUTF(this.collection);
			out.writeUTF(this.rank);
		}catch(IOException ex)
		{
			throw new GRS2RecordSerializationException("Could not send record's contents", ex);
		}
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @see gr.uoa.di.madgik.grs.record.Record#extendReceive(java.io.DataInput)
	 */
	@Override
	public void extendReceive(DataInput in) throws GRS2RecordSerializationException
	{
		try
		{
			this.id=in.readUTF();
			this.collection=in.readUTF();
			this.rank=in.readUTF();
		}catch(IOException ex)
		{
			throw new GRS2RecordSerializationException("Could not receive record's contents", ex);
		}
	}

	public void extendSendToXML(Document doc, Element element) throws GRS2RecordSerializationException
	{
		Element elm = doc.createElement("id");
		elm.setTextContent(this.id);
		element.appendChild(elm);
		
		elm = doc.createElement("collection");
		elm.setTextContent(this.collection);
		element.appendChild(elm);
		
		elm = doc.createElement("rank");
		elm.setTextContent(this.rank);
		element.appendChild(elm);
	}
	
	public void extendReceiveFromXML(Element element) throws GRS2RecordSerializationException
	{
		this.id = element.getElementsByTagName("id").item(0).getTextContent();
		this.collection = element.getElementsByTagName("collection").item(0).getTextContent();
		this.rank = element.getElementsByTagName("rank").item(0).getTextContent();
	}
	
	/**
	 * {@inheritDoc}
	 * 
	 * @see gr.uoa.di.madgik.grs.record.Record#extendDeflate(java.io.DataOutput)
	 */
	@Override
	public void extendDeflate(DataOutput out) throws GRS2RecordSerializationException
	{
		try
		{
			out.writeUTF(this.id);
			out.writeUTF(this.collection);
			out.writeUTF(this.rank);
		}catch(IOException ex)
		{
			throw new GRS2RecordSerializationException("Could not deflate record", ex);
		}
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @see gr.uoa.di.madgik.grs.record.Record#extendInflate(java.io.DataInput, boolean)
	 */
	@Override
	public void extendInflate(DataInput in, boolean reset) throws GRS2RecordSerializationException
	{
		try
		{
			this.id=in.readUTF();
			this.collection=in.readUTF();
			this.rank=in.readUTF();
		}catch(IOException ex)
		{
			throw new GRS2RecordSerializationException("Could not inflate record", ex);
		}
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @see gr.uoa.di.madgik.grs.record.Record#extendMakeLocal()
	 */
	@Override
	protected void extendMakeLocal()
	{
		//nothing to reset
	}
	
	/**
	 * {@inheritDoc}
	 * 
	 * @see gr.uoa.di.madgik.grs.record.Record#extendDispose()
	 */
	@Override
	public void extendDispose()
	{
		this.id=null;
		this.collection=null;
		this.rank=null;
	}
}
