/*
 * Decompiled with CFR 0.152.
 */
package org.gcube.dataanalysis.oscar;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.PrintWriter;
import java.util.Calendar;
import java.util.List;
import java.util.Vector;
import org.apache.commons.io.FileSystemUtils;
import org.apache.commons.io.FileUtils;
import org.gcube.dataanalysis.oscar.OscarMerger;
import org.gcube.dataanalysis.oscar.util.FTPDownloader;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import ucar.nc2.Attribute;
import ucar.nc2.FileWriter2;
import ucar.nc2.NetcdfFile;
import ucar.nc2.NetcdfFileWriter;
import ucar.nc2.dataset.NetcdfDataset;
import ucar.nc2.util.CancelTaskImpl;

public class IncrementalOscarMerger {
    private static final Integer OSCAR_START_YEAR = 2017;
    private String yearlyOscarRemoteFilePattern = "ftp://podaac-ftp.jpl.nasa.gov/allData/oscar/preview/L4/resource/LAS/oscar_third_deg_180/oscar_vel${YYYY}_180.nc";
    private String yearlyOscarLocalFilePattern = "oscar_vel${YYYY}_180.nc";
    private static String workdir = "/tmp/oscar-merger";
    private static String mergedFilePattern = "oscar_vel_${FROMYEAR}-${TOYEAR}_180.nc";
    private static String mergeDescriptorFilePattern = "oscar_vel_${FROMYEAR}-${TOYEAR}_180.xml";
    private Integer startYear;
    private Integer endYear;
    private static final Integer INTERVAL_SIZE = 3;
    private final Boolean DEBUG_REMOVE_FILES = true;
    private static final Logger logger = LoggerFactory.getLogger(OscarMerger.class);

    public IncrementalOscarMerger() {
        System.out.println("Doing some preliminary cleanup...");
        this.cleanup();
    }

    private Integer getStartYear() {
        if (this.startYear == null) {
            for (int year = OSCAR_START_YEAR.intValue(); year < Calendar.getInstance().get(1); ++year) {
                if (!this.checkRemoteYearFileExist(year)) continue;
                System.out.println("Start year of the merge is " + year);
                this.startYear = year;
                break;
            }
        }
        return this.startYear;
    }

    private Integer getEndYear() {
        if (this.endYear == null) {
            for (int year = Calendar.getInstance().get(1); year > OSCAR_START_YEAR; --year) {
                if (!this.checkRemoteYearFileExist(year)) continue;
                System.out.println("End year of the merge is " + year);
                this.endYear = year;
                break;
            }
        }
        return this.endYear;
    }

    private File getLastUsefulMergedFile() {
        int endYear;
        int startYear = this.getStartYear();
        for (int i = endYear = this.getEndYear().intValue(); i >= startYear; --i) {
            File f = new File(this.getWorkDir(), this.formatMergedFileName(startYear, i));
            if (!f.exists() || this.canStillChange(i)) continue;
            return f;
        }
        return null;
    }

    private boolean canStillChange(int year) {
        int currentYear = Calendar.getInstance().get(1);
        if (year == currentYear) {
            return true;
        }
        return year == currentYear - 1 && !this.checkRemoteYearFileExist(currentYear);
    }

    private boolean checkRemoteYearFileExist(int year) {
        return FTPDownloader.checkFtpFileExists(this.getSourceURLForYear(year));
    }

    private String getSourceURLForYear(Integer year) {
        return this.yearlyOscarRemoteFilePattern.replaceAll("\\$\\{YYYY\\}", year + "");
    }

    private File getLocalFileForYear(Integer year) {
        return new File(this.getWorkDir(), this.yearlyOscarLocalFilePattern.replaceAll("\\$\\{YYYY\\}", year + ""));
    }

    private File getMergedFilesForYears(Integer from, Integer to) {
        return new File(this.getWorkDir(), this.formatMergedFileName(from, to));
    }

    private File generateDescriptor(List<String> paths, File descriptorFile) {
        String out = "";
        out = out + "<netcdf xmlns=\"http://www.unidata.ucar.edu/namespaces/netcdf/ncml-2.2\">\n";
        out = out + "  <attribute name=\"title\" value=\"OSCAR Velocity Dataset\"/>\n";
        out = out + "  <aggregation type=\"joinExisting\" dimName=\"time\">\n";
        for (String p : paths) {
            out = out + "    <netcdf location=\"" + p + "\"/>\n";
        }
        out = out + "  </aggregation>\n";
        out = out + "</netcdf>";
        try (PrintWriter pw = new PrintWriter(descriptorFile);){
            pw.println(out);
        }
        catch (FileNotFoundException e) {
            e.printStackTrace();
        }
        return descriptorFile;
    }

    private File getWorkDir() {
        File out = new File(workdir);
        out.mkdir();
        return out;
    }

    private String formatMergedFileName(Integer fromYear, Integer toYear) {
        return mergedFilePattern.replaceAll("\\$\\{FROMYEAR\\}", fromYear.toString()).replaceAll("\\$\\{TOYEAR\\}", toYear.toString());
    }

    private String formatDescriptorFileName(Integer fromYear, Integer toYear) {
        return mergeDescriptorFilePattern.replaceAll("\\$\\{FROMYEAR\\}", fromYear.toString()).replaceAll("\\$\\{TOYEAR\\}", toYear.toString());
    }

    private Integer extractEndYear(String fileName) {
        for (int year = this.getStartYear().intValue(); year <= this.getEndYear(); ++year) {
            String name = this.formatMergedFileName(this.getStartYear(), year);
            if (fileName.equals(name)) {
                return year;
            }
            name = this.formatDescriptorFileName(this.getStartYear(), year);
            if (!fileName.equals(name)) continue;
            return year;
        }
        return null;
    }

    private int getIntervalSize() {
        return INTERVAL_SIZE;
    }

    public String merge() throws Exception {
        this.checkReady();
        File lastMerged = this.getLastUsefulMergedFile();
        if (lastMerged != null) {
            System.out.println("Found a previously merged file: " + lastMerged.getAbsolutePath());
        } else {
            File currentYearOscarFile = this.getLocalFileForYear(this.getStartYear());
            System.out.println("Downloading file to " + currentYearOscarFile.getAbsolutePath());
            this.downloadFile(this.yearlyOscarRemoteFilePattern.replaceAll("\\$\\{YYYY\\}", this.getStartYear() + ""), currentYearOscarFile, false);
            lastMerged = this.getMergedFilesForYears(this.getStartYear(), this.getStartYear());
            System.out.println("Renaming it as " + lastMerged.toString());
            FileUtils.moveFile(FileUtils.getFile(currentYearOscarFile, new String[0]), FileUtils.getFile(lastMerged, new String[0]));
        }
        System.out.println("End year of the last merged file is " + this.extractEndYear(lastMerged.getName()));
        int year = this.extractEndYear(lastMerged.getName()) + 1;
        while (year <= this.getEndYear()) {
            Vector<String> paths = new Vector<String>();
            paths.add(lastMerged.getAbsolutePath());
            int intervalEnd = year;
            for (int i = 0; i < this.getIntervalSize() && year + i <= this.getEndYear(); ++i) {
                intervalEnd = year + i;
                File currentYearOscarFile = this.getLocalFileForYear(intervalEnd);
                System.out.println("Downloading file for year " + intervalEnd);
                this.downloadFile(this.getSourceURLForYear(intervalEnd), currentYearOscarFile, false);
                paths.add(currentYearOscarFile.getAbsolutePath());
            }
            System.out.println(intervalEnd);
            System.out.println("Generating descriptor...");
            File mergeDescriptor = new File(this.getWorkDir(), this.formatDescriptorFileName(this.getStartYear(), intervalEnd));
            this.generateDescriptor(paths, mergeDescriptor);
            System.out.println("Descriptor is " + mergeDescriptor.getAbsolutePath());
            File mergedFile = this.getMergedFilesForYears(this.getStartYear(), intervalEnd);
            System.out.println("Merging files at " + mergedFile.getAbsolutePath());
            this.mergeDescribedFilesTo(mergeDescriptor, mergedFile);
            System.out.println("Merged");
            if (!this.canStillChange(this.extractEndYear(mergedFile.getName()))) {
                System.out.println("Removing " + lastMerged.getName() + " since the merged is stable. We'll keep it instead.");
                this.removeFileOrDir(lastMerged);
            } else {
                System.out.println("Keeping " + lastMerged.getName() + " since the merged one is not yet stable.");
            }
            System.out.println("Removing descriptor: " + mergeDescriptor.getAbsolutePath());
            this.removeFileOrDir(mergeDescriptor);
            System.out.println("Remove yearly files... ");
            for (int i = year; i <= intervalEnd; ++i) {
                System.out.println("Removing file " + this.getLocalFileForYear(i).getName());
                this.removeFileOrDir(this.getLocalFileForYear(i));
            }
            lastMerged = mergedFile;
            year = intervalEnd + 1;
        }
        return lastMerged.getAbsolutePath();
    }

    private void downloadFile(String source, File destination, Boolean forceDownload) throws IOException, Exception {
        if (forceDownload == null) {
            forceDownload = false;
        }
        if (!forceDownload.booleanValue() && destination.exists()) {
            System.out.println("File already downloaded and completed, skipping.");
            return;
        }
        if (!source.startsWith("ftp:")) {
            throw new Exception("Only ftp downloads currently supported");
        }
        String tmpFileName = destination.getAbsolutePath() + ".tmp";
        FTPDownloader ftpDownloader = new FTPDownloader(source);
        ftpDownloader.downloadFile(source, tmpFileName);
        System.out.println("FTP File downloaded successfully");
        ftpDownloader.disconnect();
        FileUtils.moveFile(FileUtils.getFile(tmpFileName), FileUtils.getFile(destination.getAbsolutePath()));
    }

    private void mergeDescribedFilesTo(File descriptor, File merged) throws Exception {
        String datasetIn = descriptor.getAbsolutePath();
        String datasetOut = merged.getAbsolutePath() + ".tmp";
        CancelTaskImpl cancel = new CancelTaskImpl();
        NetcdfFile ncfileIn = NetcdfDataset.openFile(datasetIn, cancel);
        System.out.println(String.format("NetcdfDatataset read from %s write to %s ", datasetIn, datasetOut));
        NetcdfFileWriter.Version version = NetcdfFileWriter.Version.netcdf4;
        FileWriter2 writer = new FileWriter2(ncfileIn, datasetOut, version, null);
        writer.getNetcdfFileWriter().setLargeFile(true);
        NetcdfFile ncfileOut = writer.write(cancel);
        for (Attribute a : ncfileOut.getGlobalAttributes()) {
            System.out.println(a);
        }
        if (ncfileOut != null) {
            ncfileOut.close();
        }
        ncfileIn.close();
        cancel.setDone(true);
        FileUtils.moveFile(FileUtils.getFile(datasetOut), merged);
        System.out.println(String.format("%s%n", cancel));
    }

    public void cleanup() {
        this.cleanup(false);
    }

    public void cleanup(boolean complete) {
        if (complete) {
            this.removeFileOrDir(this.getWorkDir());
        } else {
            Vector<String> filesToKeep = new Vector<String>();
            File lastUseful = this.getLastUsefulMergedFile();
            if (lastUseful != null) {
                filesToKeep.add(lastUseful.getAbsolutePath());
            }
            for (File f : this.getWorkDir().listFiles()) {
                if (filesToKeep.contains(f.getAbsolutePath())) {
                    System.out.println("Keeping " + f.getAbsolutePath());
                    continue;
                }
                this.removeFileOrDir(f);
            }
        }
    }

    private void removeFileOrDir(File file) {
        System.out.println("Removing " + file.getAbsolutePath());
        if (!file.getAbsolutePath().startsWith("/tmp")) {
            System.out.println("NOT REMOVING ANYTHING OUTSIDE /tmp");
            return;
        }
        if (file.getAbsolutePath().endsWith(".keep")) {
            System.out.println("NOT REMOVING " + file.getAbsolutePath() + " since it has to be kept");
            return;
        }
        if (this.DEBUG_REMOVE_FILES.booleanValue()) {
            FileUtils.deleteQuietly(file);
            System.out.println("Removed");
        } else {
            System.out.println("Removing IS DISABLED");
        }
    }

    public static void main(String[] args) {
        new IncrementalOscarMerger();
    }

    public void checkReady() throws Exception {
        this.checkEnoughDiskSpace();
    }

    private void checkEnoughDiskSpace() throws Exception {
        System.out.println("Checking needed space...");
        Long backMergedFileSize = 0L;
        File f = this.getLastUsefulMergedFile();
        if (f == null) {
            backMergedFileSize = 417518585L * (long)((double)(this.getEndYear() - 1 - this.getStartYear()) - 1.2);
        }
        System.out.println("Last useful merge: " + backMergedFileSize);
        Long newMergedFileSize = 417518585L * (long)((double)(this.getEndYear() - this.getStartYear()) - 1.2);
        System.out.println("New merge: " + newMergedFileSize);
        Long yearlyFilesSize = 1198005516L * (long)INTERVAL_SIZE.intValue();
        System.out.println("Yearly files: " + yearlyFilesSize);
        Long neededSpace = backMergedFileSize + newMergedFileSize + yearlyFilesSize;
        System.out.println("Needed disk space: " + neededSpace + " bytes");
        Long availableSpace = FileSystemUtils.freeSpaceKb(this.getWorkDir().getAbsolutePath()) * 1024L;
        System.out.println("Available disk space: " + availableSpace + " bytes");
        if (neededSpace >= availableSpace) {
            Double gb = (double)neededSpace.longValue() / 1024.0 / 1024.0 / 1024.0;
            String message = String.format("Not enough disk space. At least %2.2fGB of available disk needed.", gb);
            throw new Exception(message);
        }
        System.out.println("There's enough disk space to proceed");
        Double percent = (double)neededSpace.longValue() * 1.0 / (double)availableSpace.longValue() * 1.0 * 100.0;
        String message = String.format("I'm about to use %.0f%% of the available disk space (but I'll release it afterwards, I promise)", percent);
        System.out.println(message);
    }
}

