/**
 * 
 */
package org.gcube.dataanalysis.copernicus.cmems.importer.seplugin.thredds;

import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.gcube.dataanalysis.copernicus.cmems.importer.api.ChunkTimespan;
import org.gcube.dataanalysis.datasetimporter.util.TimeUtil;

/**
 * This class models a chunk of a dataset on Thredds.
 * 
 * @author Paolo Fabriani
 *
 */
public class ThreddsDatasetChunk {

    /**
     * The name of the file on thredds
     */
    private String fileName;

    /**
     * A pattern for chunk file names
     */
    private final static Pattern p = Pattern.compile("([^-]+)-([0-9]+).*");

    /**
     * The name of the chunk on thredds.
     */
    private String name;

    /**
     * The start time of the chunk.
     */
    private Calendar start;

    /**
     * The when the chunk was last updated.
     */
    private Calendar updated;

    /**
     * 
     * @return
     */
    private ChunkTimespan timeSpan;

    public ThreddsDatasetChunk() {
    }

    public ThreddsDatasetChunk(String fileName) throws ParseException {
        this.fileName = fileName;
        Matcher m = p.matcher(fileName);
        if (m.matches()) {
            this.setChunkStart(TimeUtil.toCalendar(m.group(2)));
        } else {
            throw new ParseException(fileName, 0);
        }
        Matcher m2 = p.matcher(fileName);
        if (m2.matches()) {
            this.setName(m.group(1));
        } else {
            throw new ParseException(fileName, 0);
        }
    }

    public String getName() {
        return name;
    }

    public void setName(String name) {
        this.name = name;
    }

    public Calendar getChunkStart() {
        return start;
    }

    public void setChunkStart(Calendar start) {
        this.start = start;
    }

    public Calendar getChunkUpdate() {
        return updated;
    }

    public Calendar getChunkEnd() {
        switch(this.getTimeSpan()) {
        case MONTH:
            return TimeUtil.getEndOfMonth(this.getChunkStart());
        case YEAR:
            return TimeUtil.getEndOfYear(this.getChunkStart());
        default:
            return TimeUtil.getEndOfDay(this.getChunkStart());
        }
    }

    public void setChunkUpdate(Calendar updated) {
        this.updated = updated;
    }

    public boolean needsUpdate() {
        // TODO: consider back time here
        return this.getChunkEnd().after(this.getChunkUpdate());
    }

    public String toString() {
        return this.getName() + "\n  Period:  "
                + TimeUtil.toString(this.getChunkStart()) + "  \n  Updated: "
                + TimeUtil.toString(this.getChunkUpdate()) + "  \n  Period:  "
                + this.getTimeSpan() + "  \n  Update?  " + this.needsUpdate();
    }

    /**
     * Guess the timespan of the chunk (day, month, year) by looking at the
     * length of the timestamp (8, 6, 4 chars) in the chunk file name.
     * 
     * @param chunkName
     *            the file name of the chunk
     * @return
     * @throws ParseException
     */
    public ChunkTimespan getTimeSpan() {
        if (this.timeSpan != null) {
            return this.timeSpan;
        } else {
            Matcher m = p.matcher(this.getFileName());
            if (m.matches()) {
                int l = m.group(2).length();
                if (l == 4)
                    return ChunkTimespan.YEAR;
                if (l == 6)
                    return ChunkTimespan.MONTH;
                if (l == 8)
                    return ChunkTimespan.DAY;
            }
            return ChunkTimespan.DAY;
        }
        // throw new ParseException(this.getName(), 0);
    }

    public String getFileName() {
        if (this.fileName != null) {
            return fileName;
        } else {
            return this.getName() + "-" + this.formatDatePart() + ".nc";
            // int ts = this.getTimeSpan();
            // String timeStamp = ""+this.getChunkStart().get(Calendar.YEAR);
            // if(Calendar.MONTH==ts || Calendar.DAY_OF_YEAR==ts)
            // ts+=this.getChunkStart().get(Calendar.MONTH);
            // if(Calendar.DAY_OF_YEAR==ts)
            // ts+=this.getChunkStart().get(Calendar.DAY_OF_YEAR);
            // this.fileName = this.getName()+"-"+this.get
        }
    }

    public void setFileName(String fileName) {
        this.fileName = fileName;
    }

    private String formatDatePart() {
        Calendar c = this.getChunkStart();
        DateFormat df;
        String datePart;
//        System.out.println(this.getTimeSpan() + " " + Calendar.YEAR);
        switch (this.getTimeSpan()) {
        case YEAR:
            df = new SimpleDateFormat("yyyy");
            datePart = df.format(c.getTime());
            break;
        case MONTH:
            df = new SimpleDateFormat("yyyyMM");
            datePart = df.format(c.getTime());
            break;
        default:
            df = new SimpleDateFormat("yyyyMMdd");
            datePart = df.format(c.getTime());
        }
        return datePart;
    }

    public void setTimeSpan(ChunkTimespan timeSpan) {
        this.timeSpan = timeSpan;
    }
}
