/**
 * 
 */
package org.gcube.dataanalysis.copernicus.cmems.client;

import java.net.URL;
import java.text.ParseException;
import java.util.Calendar;
import java.util.Collection;
import java.util.TreeSet;
import java.util.Vector;
import java.util.stream.Collectors;

import org.gcube.dataanalysis.copernicus.cmems.model.CmemsKeyword;
import org.gcube.dataanalysis.copernicus.cmems.model.CmemsProduct;
import org.gcube.dataanalysis.copernicus.cmems.model.CmemsRegion;
import org.gcube.dataanalysis.copernicus.cmems.model.CmemsVariable;
import org.gcube.dataanalysis.copernicus.cmems.model.RawCmemsProduct;
import org.gcube.dataanalysis.copernicus.motu.model.MotuServer;
import org.gcube.dataanalysis.datasetimporter.util.TextUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;

/**
 * @author Paolo Fabriani
 *
 */
public class CmemsClient {

    /**
     * A Logger for this class
     */
    private static Logger LOGGER = LoggerFactory.getLogger(CmemsClient.class);

    /**
     * The default URL of the CMEMS catalogue
     */
    private static String CATALOG_URL = "http://cmems-resources.cls.fr/index.php?option=com_csw&task=advancedsearch";

    /**
     * The cached list of products
     */
    private static Collection<CmemsProduct> cachedProducts = new Vector<>();

    /**
     * When the cache was last updated
     */
    private static Long cacheTimestamp;

    /**
     * For how long we'll keep the cache (in minutes)
     */
    private static final Long CACHE_TTL = 8 * 60L;

    /**
     * The endpoint of the service
     */
    private String endpoint;

    /**
     * Constructor with custom URL
     * @param endpoint
     *            the custom URL of the CMEMS catalogue
     */
    public CmemsClient(String endpoint) {
        this.endpoint = endpoint;
    }

    /**
     * Constructor with default URL
     */
    public CmemsClient() {
        this(CATALOG_URL);
    }

    /**
     * Download the set of CMEMS products.
     * @return a collection of all CMEMS products.
     * @throws Exception
     */
    private Collection<CmemsProduct> downloadProducts() throws Exception {
        ObjectMapper objectMapper = new ObjectMapper();
        JsonNode rootNode = objectMapper.readTree(new URL(this.endpoint));
        JsonNode docsNode = rootNode.path("products").path("response")
                .path("docs");
        Collection<CmemsProduct> out = new Vector<CmemsProduct>();
        for (RawCmemsProduct p : objectMapper.readValue(docsNode.toString(),
                RawCmemsProduct[].class)) {
            p.setAbstract(TextUtil.stripHTML(p.getAbstract()));
            CmemsProduct newP = new CmemsProduct(p);
            out.add(newP);
        }
        return out;
    }

    /**
     * Retrieve the set of all CMEMS products, possibly taking them from the
     * cache.
     * @return
     * @throws Exception
     */
    public Collection<CmemsProduct> getAllProducts() throws Exception {
        synchronized (cachedProducts) {
            if (this.isCacheExpired()) {
                LOGGER.debug("getting a fresh copy of products...");
                Collection<CmemsProduct> prods = this.downloadProducts();
                LOGGER.debug("got " + prods.size() + " products");
                cachedProducts = prods;
                cacheTimestamp = Calendar.getInstance().getTimeInMillis();
            }
            return cachedProducts;
        }
    }

    /**
     * Check whether the cache has expired.
     * @return true if expired, false otherwise.
     */
    private Boolean isCacheExpired() {
        if (cacheTimestamp == null) {
            return true;
        } else if ((Calendar.getInstance().getTimeInMillis()
                - cacheTimestamp) > CACHE_TTL * 60 * 1000) {
            return true;
        } else {
            return false;
        }
    }

    /**
     * Return the list of Motu servers publishing CMEMS datasets.
     * @return
     * @throws Exception
     */
    public Collection<MotuServer> getMotuServers() throws Exception {
        Collection<String> urls = new TreeSet<>();
        for (CmemsProduct psi : this.getAllProducts()) {
            if (psi.getMotuServer() != null) {
                urls.add(psi.getMotuServer());
            }
            // String tds = psi.getTds();
            // if (tds != null && !tds.equalsIgnoreCase("undefined")) {
            // String url = tds.substring(0, tds.indexOf("?"));
            // urls.add(url);
            // }
        }
        Collection<MotuServer> out = new Vector<>();
        for (String url : urls) {
            out.add(new MotuServer(url));
        }
        return out;
    }

    /**
     * Return the set of regional domains to which datasets belong.
     * @return
     * @throws Exception
     */
    public Collection<CmemsRegion> getRegionalDomains() throws Exception {
        Collection<CmemsRegion> out = new TreeSet<>();
        for (CmemsProduct product : this.getAllProducts()) {
            out.addAll(product.getGeographicalArea());
            // String[] parameters = product.getGeographicalArea();
            // for (String par : parameters) {
            // if (par != null && !par.trim().isEmpty() &&
            // !par.trim().equals("undefined")) {
            // CmemsRegion r = new CmemsRegion();
            // r.setName(par.trim());
            // out.add(r);
            // }
            // }
        }
        return out;
    }

    /**
     * Return the set of regional domains to which datasets belong.
     * @return
     * @throws Exception
     */
    public Collection<CmemsKeyword> getAllKewords() throws Exception {
        Collection<CmemsKeyword> out = new TreeSet<>();
        for (CmemsProduct product : this.getAllProducts()) {
            out.addAll(product.getAllKeywords());
        }
        return out;
    }

    /**
     * Return the set of variables in datasets.
     */
    public Collection<CmemsVariable> getOceanVariables() throws Exception {
        Collection<CmemsVariable> out = new TreeSet<>();
        for (CmemsProduct product : this.getAllProducts()) {
            out.addAll(product.getOceanVariables());
        }
        return out;
    }

    /**
     * Return the set of variables in datasets.
     */
    public Collection<CmemsVariable> getOceanKeys() throws Exception {
        Collection<CmemsVariable> out = new TreeSet<>();
        for (CmemsProduct product : this.getAllProducts()) {
            out.addAll(product.getOceanVariables());
        }
        return out;
    }

    /**
     * Return the lowest timestamp in datasets.
     * @return
     * @throws Exception
     */
    public Calendar getMinTime() throws Exception {
        Calendar out = Calendar.getInstance();
        for (CmemsProduct product : this.getAllProducts()) {
            Calendar c = product.getTemporalBegin();
            if (c.before(out)) {
                out = c;
            }
        }
        return out;
    }

    /**
     * Return the highest timestamp in datasets.
     * @return
     * @throws Exception
     */
    public Calendar getMaxTime() throws Exception {
        Calendar out = Calendar.getInstance();
        out.setTimeInMillis(0);
        for (CmemsProduct product : this.getAllProducts()) {
            if (product.getTemporalEnd() != null) {
                Calendar c = product.getTemporalEnd();
                if (c.after(out)) {
                    out = c;
                }
            }
        }
        return out;
    }

    /**
     * Search for products matching the given criteria.
     * @param options
     * @return
     */
    public Collection<CmemsProduct> searchProducts(SearchOptions options)
            throws Exception {
        Collection<CmemsProduct> output = new Vector<>();
        for (CmemsProduct product : this.getAllProducts()) {
            if (this.productMatches(product, options)) {
                output.add(product);
            }
        }
        return output;
    }

    /**
     * Inner method to verify the match of a product against given search
     * options.
     * @param product
     * @param options
     * @return
     */
    private boolean productMatches(CmemsProduct product,
            SearchOptions options) {
        
//        LOGGER.debug("matching...");
        
        String text = product.getExternalShortname() + " ";
        text += product.getAbstract() + " ";
        text += product.getShortDescription() + " ";
        text += product.getGeographicalArea().stream().map(i -> i.toString())
                .collect(Collectors.joining(" "));
        text += product.getAllKeywords().stream().map(i -> i.toString())
                .collect(Collectors.joining(" "));
        text += product.getOceanVariables().stream().map(i -> i.toString())
                .collect(Collectors.joining(" "));
        text += product.getOceanKeys().stream().map(i -> i.toString())
                .collect(Collectors.joining(" "));
        text += product.getTdsDatasets().stream().map(i -> i.toString())
                .collect(Collectors.joining(" "));

        if (options.isIgnoreCase()) {
            text = text.toLowerCase();
        }

        product.getGeographicalArea().stream().map(i -> i.toString())
                .collect(Collectors.joining(","));

        // match regional domains
        for (String k : options.getRegionalDomains()) {
            // String t = String.join(" ", product.getGeographicalArea());
            String t = product.getGeographicalArea().stream()
                    .map(i -> i.toString()).collect(Collectors.joining(" "));
            if (options.isIgnoreCase()) {
                k = k.toLowerCase();
                t = t.toLowerCase();
            }
            if (!t.contains(k)) {
                return false;
            }
        }

        // match variables
        for (String k : options.getVariables()) {
            // String t = String.join(" ", product.getOceanKeys());
            String t = product.getOceanKeys().stream().map(i -> i.toString())
                    .collect(Collectors.joining(" "));
            if (options.isIgnoreCase()) {
                k = k.toLowerCase();
                t = t.toLowerCase();
            }
            if (!t.contains(k)) {
                return false;
            }
        }

        // match dates (from)
        if (options.getFrom() != null) {
            try {
                if (product.getTemporalEnd() != null
                        && product.getTemporalEnd().before(options.getFrom())) {
                    return false;
                }
                if (options.getWholeTimeRange()
                        && product.getTemporalBegin() != null) {
                    if (product.getTemporalBegin().after(options.getFrom())) {
                        return false;
                    }
                }
            } catch (ParseException e) {
                e.printStackTrace();
            }
        }

        // Match dates (to)
        if (options.getTo() != null) {
            try {
                if (product.getTemporalBegin() != null
                        && product.getTemporalBegin().after(options.getTo())) {
                    return false;
                }
                if (options.getWholeTimeRange()
                        && product.getTemporalEnd() != null) {
                    if (product.getTemporalEnd().before(options.getTo())) {
                        return false;
                    }
                }
            } catch (ParseException e) {
                e.printStackTrace();
            }
        }

        // match all required keys
        for (String k : options.getMatchKeys()) {
            if (options.isIgnoreCase()) {
                k = k.toLowerCase();
            }
            if (!text.contains(k)) {
                return false;
            }
        }

        // ensure unmatch keys are not present
        for (String k : options.getUnmatchKeys()) {
            if (options.isIgnoreCase()) {
                k = k.toLowerCase();
            }
            if (text.contains(k)) {
                return false;
            }
        }
        return true;
    }

}
