package org.gcube.dataanalysis.executor.generators.v1;

import java.io.File;
import java.util.ArrayList;
import java.util.List;
import java.util.Properties;

import org.apache.axis.message.addressing.Address;
import org.apache.axis.message.addressing.EndpointReferenceType;
import org.gcube.contentmanagement.graphtools.utils.HttpRequest;
import org.gcube.contentmanagement.lexicalmatcher.utils.AnalysisLogger;
import org.gcube.contentmanagement.lexicalmatcher.utils.FileTools;
import org.gcube.dataanalysis.ecoengine.configuration.ALG_PROPS;
import org.gcube.dataanalysis.ecoengine.configuration.AlgorithmConfiguration;
import org.gcube.dataanalysis.ecoengine.configuration.INFRASTRUCTURE;
import org.gcube.dataanalysis.ecoengine.connectors.livemonitor.ResourceLoad;
import org.gcube.dataanalysis.ecoengine.connectors.livemonitor.Resources;
import org.gcube.dataanalysis.ecoengine.datatypes.ServiceType;
import org.gcube.dataanalysis.ecoengine.datatypes.StatisticalType;
import org.gcube.dataanalysis.ecoengine.datatypes.enumtypes.ServiceParameters;
import org.gcube.dataanalysis.ecoengine.interfaces.Generator;
import org.gcube.dataanalysis.ecoengine.interfaces.GenericAlgorithm;
import org.gcube.dataanalysis.ecoengine.interfaces.SpatialProbabilityDistributionNode;
import org.gcube.dataanalysis.ecoengine.utils.Operations;
import org.gcube.dataanalysis.executor.job.management.RemoteJobManager;

public class D4ScienceGenerator implements Generator {
	protected AlgorithmConfiguration config;
	protected SpatialProbabilityDistributionNode distributionModel;
	protected RemoteJobManager jobManager;
	protected static String defaultJobOutput = "execution.output";
	protected static String defaultScriptFile = "script.sh";
	protected String mainclass;
	protected boolean stop;
	protected static int maxSpeciesAllowedPerJob = 10;
	protected int speciesBlocksExecuted;
	protected int speciesBlocks;
	
	public D4ScienceGenerator(){
		this.stop = false;
	}			
	
	public D4ScienceGenerator(AlgorithmConfiguration config) {
		this.config = config;
		this.stop = false;
		AnalysisLogger.setLogger(config.getConfigPath() + AlgorithmConfiguration.defaultLoggerFile);
	}

	public void compute() throws Exception {
		try {
			String scope = config.getGcubeScope();
			if (scope == null)
				throw new Exception("Null Scope");
			int speciesNum = distributionModel.getNumberOfSpecies();
			int geoNum = distributionModel.getNumberOfGeoInfo();
			
			List<String> endpoints = config.getEndpoints();
			if (endpoints != null) {
				List<EndpointReferenceType> eprtList = new ArrayList<EndpointReferenceType>();
				for (String ep : endpoints) {
					eprtList.add(new EndpointReferenceType(new Address(ep)));
				}
				jobManager = new RemoteJobManager(scope, config.getNumberOfResources(), eprtList);
			} else
				jobManager = new RemoteJobManager(scope, config.getNumberOfResources());
			
			int numberOfResources = jobManager.getNumberOfNodes();
			//we split along species so if species are less than nodes, we should reduce the number of nodes

			if (numberOfResources > 0) {
				//chunkize the number of species in order to lower the computational effort of the workers
				int div = speciesNum/(numberOfResources*maxSpeciesAllowedPerJob);
				int rest = speciesNum%(numberOfResources*maxSpeciesAllowedPerJob);
				if (rest>0)
					div++;
				if (div ==0){
					div = 1;
				}
				int [] speciesChunks = Operations.takeChunks(speciesNum, div);
				speciesBlocks = speciesChunks.length;
				int offset = 0;
				boolean deletefiles = true;
				for (int i=0;i<speciesBlocks;i++){
					AnalysisLogger.getLogger().debug("D4ScienceGenerator-> EXECUTING THE BUNCH OF "+ speciesChunks[i]+" SPECIES - NUMBER "+(i+1)+" OF "+speciesBlocks+" WITH OFFSET "+offset);
					if (i==speciesBlocks-1)
						deletefiles = true;
					executeWork(geoNum, speciesChunks[i],offset,numberOfResources,deletefiles);
					offset += speciesChunks[i];
					speciesBlocksExecuted++;
				if (jobManager.wasAborted()) {
					AnalysisLogger.getLogger().debug("Warning: Job was aborted");
					break;
				}
				}
			}
			
		} catch (Exception e) {
			e.printStackTrace();
			throw e;
		} finally {
			speciesBlocksExecuted = speciesBlocks;
			shutdown();
		}
	}

	private void executeWork(int geoNum, int speciesNum,int speciesOffset,int numberOfResources,boolean deletefiles) throws Exception{
		int[] chunkSizes = Operations.takeChunks(speciesNum, numberOfResources);
		jobManager.setNumberOfNodes(chunkSizes.length);
		List<String> arguments = new ArrayList<String>();
		// chunkize respect to the cells: take a chunk of cells vs all species at each node!
		int order = speciesOffset;
		for (int i = 0; i < chunkSizes.length; i++) {
			String argumentString = "0_" + geoNum + "_"+order+"_" + chunkSizes[i] + "_./";
			arguments.add(argumentString);
			order += chunkSizes[i];
			AnalysisLogger.getLogger().debug("D4ScienceGenerator-> Argument "+i+": "+argumentString);
			}
		String owner = config.getParam("ServiceUserName");
		if (owner == null)
			throw new Exception("Null Owner");
		String pathToDir = config.getPersistencePath() + config.getModel();
		if (!(new File(pathToDir).exists()))
			throw new Exception("No Implementation of node-model found for algorithm " + pathToDir);

		if (mainclass == null)
			throw new Exception("No mainClass found for algorithm " + pathToDir);

		buildScriptFile(config.getModel(), defaultJobOutput, pathToDir, mainclass);

		// job.uploadAndExecute(serviceClass, serviceName, owner, directory, remotedirectory, tempDir, scriptName, argums);
		jobManager.uploadAndExecute(AlgorithmConfiguration.StatisticalManagerClass, AlgorithmConfiguration.StatisticalManagerService, owner, pathToDir, "/" + config.getModel() + "/", "./", defaultScriptFile, arguments, deletefiles);

	}
	
	// builds a job.sh
	public void buildScriptFile(String jobName, String jobOutput, String jarsPath, String fullMainClass) throws Exception {
		StringBuffer sb = new StringBuffer();
		sb.append("#!/bin/sh\n");
		sb.append("# " + jobName + "\n");
		sb.append("cd $1\n");
		sb.append("\n");
		sb.append("java -Xmx1024M -classpath ./:");
		File jarsPathF = new File(jarsPath);
		File[] files = jarsPathF.listFiles();

		for (File jar : files) {

			if (jar.getName().endsWith(".jar")) {
				sb.append("./" + jar.getName());
				sb.append(":");
			}
		}

		sb.deleteCharAt(sb.length() - 1);
		sb.append(" " + fullMainClass + " $2 " + jobOutput);
		sb.append("\n");

		AnalysisLogger.getLogger().trace("D4ScienceGenerator->Generating script in " + new File(jarsPath, defaultScriptFile).getAbsolutePath());
		FileTools.saveString(new File(jarsPath, defaultScriptFile).getAbsolutePath(), sb.toString(), true, "UTF-8");

	}

	@Override
	public List<StatisticalType> getInputParameters() {
		
//		List<StatisticalType> distributionModelParams = this.distributionModel.getInputParameters();
		List<StatisticalType> distributionModelParams = new ArrayList<StatisticalType>();
		distributionModelParams.add(new ServiceType(ServiceParameters.EPR_LIST, "EPRLIST","List of workers ep","",true));
		
		return distributionModelParams;
	}

	@Override
	public String getResources() {
		Resources res = new Resources();
		try {
			int activeNodes = jobManager.getActiveNodes();
			for (int i = 0; i < jobManager.getNumberOfNodes(); i++) {
				try {
					double value = (i < activeNodes) ? 100.00 : 0.00;
					res.addResource("Worker_" + (i + 1), value);
				} catch (Exception e1) {
				}
			}
		} catch (Exception e) {
			AnalysisLogger.getLogger().debug("D4ScienceGenerator->active nodes not ready");
		}
		if ((res != null) && (res.list != null))
			return HttpRequest.toJSon(res.list).replace("resId", "resID");
		else
			return "";
	}

	@Override
	public float getStatus() {
		try {
				float estimated = 0f; 
				if (speciesBlocksExecuted==0)
				{
					int nactiveNodes = jobManager.getActiveNodes();
					if (nactiveNodes>0)
					{
						estimated = 100f*(float)(0.5)/(float)(speciesBlocks);
					}
				}
				else{
				/*
				int estimatedproduction = distributionModel.getOverallProcessedInfo();
				int asintoticValue = (int)(0.025*(distributionModel.getNumberOfSpecies()*distributionModel.getNumberOfGeoInfo()));
				float estimated = (float) estimatedproduction*100f/(float)asintoticValue;
				*/
				
				estimated =  100f*(float)(speciesBlocksExecuted)/(float)speciesBlocks;
				/*
				if (estimated<minestimate)
					estimated = minestimate;
				
				float reductionFactor = (1000f/(1.5f*(float)distributionModel.getNumberOfSpecies()));
				float estimated = (float)estimatedproduction*100f/(float)(distributionModel.getNumberOfSpecies()*distributionModel.getNumberOfGeoInfo())*reductionFactor;
				*/
				}
				System.out.println("ESTIMATED : "+estimated);
				if (!stop)
					return Math.min(estimated, 95f);
				else
					return 100f;
							
		} catch (Exception e) {
			return 0f;
		}
	}


	@Override
	public StatisticalType getOutput() {
		return distributionModel.getOutput();
	}
	
	@Override
	public ALG_PROPS[] getSupportedAlgorithms() {
		ALG_PROPS[] p = { ALG_PROPS.PHENOMENON_VS_GEOINFO, ALG_PROPS.SPECIES_VS_CSQUARE_FROM_DATABASE };
		return p;
	}

	@Override
	public INFRASTRUCTURE getInfrastructure() {
		return INFRASTRUCTURE.D4SCIENCE;
	}

	@Override
	public void init() throws Exception {

		Properties p = AlgorithmConfiguration.getProperties(config.getConfigPath() + AlgorithmConfiguration.nodeAlgorithmsFile);
		mainclass = p.getProperty(config.getModel());
		distributionModel = (SpatialProbabilityDistributionNode) Class.forName(mainclass).newInstance();
		distributionModel.setup(config);

	}

	@Override
	public void setConfiguration(AlgorithmConfiguration config) {
		this.config = config;
	}

	@Override
	public void shutdown() {
		stop = true;
		try {
			jobManager.stop();
		} catch (Exception e) {
		}
		try {
			distributionModel.stop();
		} catch (Exception e) {
		}
	}

	@Override
	public String getLoad() {
		long tk = System.currentTimeMillis();
		int processedSpecies = distributionModel.getNumberOfProcessedSpecies();
		ResourceLoad rs = new ResourceLoad(tk, processedSpecies);
		return rs.toString();
	}

	private long lastTime;
	private int lastProcessed;

	@Override
	public String getResourceLoad() {
		long thisTime = System.currentTimeMillis();
		int processedRecords = distributionModel.getNumberOfProcessedSpecies();
		int estimatedProcessedRecords = 0;
		if (processedRecords == lastProcessed) {
			estimatedProcessedRecords = Math.round(((float) thisTime * (float) lastProcessed) / (float) lastTime);
		} else {
			lastProcessed = processedRecords;
			estimatedProcessedRecords = lastProcessed;
		}
		lastTime = thisTime;
		ResourceLoad rs = new ResourceLoad(thisTime, estimatedProcessedRecords);
		return rs.toString();
	}

	@Override
	public GenericAlgorithm getAlgorithm() {
		return distributionModel;
	}

	@Override
	public String getDescription() {
		return "A Generator on D4Science relying only on Executors";
	}

}
