package org.gcube.dataanalysis.executor.generators.v1;

import java.io.File;
import java.util.ArrayList;
import java.util.List;
import java.util.Properties;

import org.apache.axis.message.addressing.Address;
import org.apache.axis.message.addressing.EndpointReferenceType;
import org.gcube.contentmanagement.graphtools.utils.HttpRequest;
import org.gcube.contentmanagement.lexicalmatcher.utils.AnalysisLogger;
import org.gcube.contentmanagement.lexicalmatcher.utils.FileTools;
import org.gcube.dataanalysis.ecoengine.configuration.ALG_PROPS;
import org.gcube.dataanalysis.ecoengine.configuration.AlgorithmConfiguration;
import org.gcube.dataanalysis.ecoengine.configuration.INFRASTRUCTURE;
import org.gcube.dataanalysis.ecoengine.connectors.livemonitor.ResourceLoad;
import org.gcube.dataanalysis.ecoengine.connectors.livemonitor.Resources;
import org.gcube.dataanalysis.ecoengine.datatypes.ServiceType;
import org.gcube.dataanalysis.ecoengine.datatypes.StatisticalType;
import org.gcube.dataanalysis.ecoengine.datatypes.enumtypes.ServiceParameters;
import org.gcube.dataanalysis.ecoengine.interfaces.Generator;
import org.gcube.dataanalysis.ecoengine.interfaces.GenericAlgorithm;
import org.gcube.dataanalysis.ecoengine.interfaces.SpatialProbabilityDistributionNode;
import org.gcube.dataanalysis.ecoengine.utils.Operations;
import org.gcube.dataanalysis.executor.job.management.QueueJobManager;

import com.thoughtworks.xstream.XStream;

public class D4ScienceQueueGenerator implements Generator {
	protected AlgorithmConfiguration config;
	protected SpatialProbabilityDistributionNode distributionModel;
	protected QueueJobManager jobManager;
	protected static String defaultJobOutput = "execution.output";
	protected static String defaultScriptFile = "script.sh";
	public static boolean deletefiles = true;
	protected String mainclass;
	protected boolean stop;
	protected static int maxSpeciesAllowedPerJob = 20;
	public static boolean forceUpload = true;
	protected int speciesBlocksExecuted;
	protected int speciesBlocks;

	public D4ScienceQueueGenerator(){
		this.stop = false;
	}
	
	public D4ScienceQueueGenerator(AlgorithmConfiguration config) {
		this.config = config;
		this.stop = false;
		AnalysisLogger.setLogger(config.getConfigPath() + AlgorithmConfiguration.defaultLoggerFile);
	}

	public void compute() throws Exception {
		try {
			String scope = config.getGcubeScope();
			if (scope == null)
				throw new Exception("Null Scope");
			int speciesNum = distributionModel.getNumberOfSpecies();
			int geoNum = distributionModel.getNumberOfGeoInfo();

			List<String> endpoints = config.getEndpoints();
			if (endpoints != null) {
				List<EndpointReferenceType> eprtList = new ArrayList<EndpointReferenceType>();
				for (String ep : endpoints) {
					eprtList.add(new EndpointReferenceType(new Address(ep)));
				}
				jobManager = new QueueJobManager(scope, config.getNumberOfResources(), eprtList);
			} else
				jobManager = new QueueJobManager(scope, config.getNumberOfResources());

			int numberOfResources = jobManager.getNumberOfNodes();
			// we split along species so if species are less than nodes, we should reduce the number of nodes

			if (numberOfResources > 0) {
				// chunkize the number of species in order to lower the computational effort of the workers
				int div = speciesNum / (numberOfResources * maxSpeciesAllowedPerJob);
				int rest = speciesNum % (numberOfResources * maxSpeciesAllowedPerJob);
				if (rest > 0)
					div++;
				if (div == 0) {
					div = 1;
				}

				
				executeWork(geoNum, speciesNum, 0, div, deletefiles, forceUpload);

				if (jobManager.wasAborted()) {
					AnalysisLogger.getLogger().debug("Warning: Job was aborted");
					distributionModel.postProcess(false,true);
					throw new Exception("Job System Error");
				}
				else{
					//postprocess
					distributionModel.postProcess(jobManager.hasResentMessages(),false);
				}
				
				
			} else {
				AnalysisLogger.getLogger().debug("Warning: No Workers available");
				throw new Exception("No Workers available");
			}
			
		} catch (Exception e) {
			AnalysisLogger.getLogger().error("ERROR: An Error occurred ", e);
			e.printStackTrace();
			throw e;
		} finally {
			speciesBlocksExecuted = speciesBlocks;
			shutdown();
		}
	}

	private void executeWork(int geoNum, int speciesNum, int speciesOffset, int numberOfResources, boolean deletefiles, boolean forceUpload) throws Exception {
		int[] chunkSizes = Operations.takeChunks(speciesNum, numberOfResources);
		List<String> arguments = new ArrayList<String>();
		// chunkize respect to the cells: take a chunk of cells vs all species at each node!
		int order = speciesOffset;
		for (int i = 0; i < chunkSizes.length; i++) {
			String argumentString = "0 " + geoNum + " " + order + " " + chunkSizes[i] + " ./ "+mainclass;
			arguments.add(argumentString);
			order += chunkSizes[i];
			AnalysisLogger.getLogger().debug("D4ScienceGenerator-> Argument " + i + ": " + argumentString);
		}
		String owner = config.getParam("ServiceUserName");
		if (owner == null)
			throw new Exception("Null Owner");
		String pathToDir = config.getPersistencePath() + config.getModel();
		if (!(new File(pathToDir).exists()))
			throw new Exception("No Implementation of node-model found for algorithm " + pathToDir);

		if (mainclass == null)
			throw new Exception("No mainClass found for algorithm " + pathToDir);

		buildScriptFile(config.getModel(), defaultJobOutput, pathToDir, mainclass);

		// job.uploadAndExecute(serviceClass, serviceName, owner, directory, remotedirectory, tempDir, scriptName, argums);
		jobManager.uploadAndExecuteChunkized(AlgorithmConfiguration.StatisticalManagerClass, AlgorithmConfiguration.StatisticalManagerService, owner, pathToDir, "/" + config.getModel() + "/", "./", defaultScriptFile, arguments, new XStream().toXML(config), deletefiles, forceUpload);

	}

	// builds a job.sh
	public void buildScriptFile(String jobName, String jobOutput, String jarsPath, String fullMainClass) throws Exception {
		File expectedscript = new File(jarsPath, defaultScriptFile);
		if (!expectedscript.exists()) {
			StringBuffer sb = new StringBuffer();
			sb.append("#!/bin/sh\n");
			sb.append("# " + jobName + "\n");
			sb.append("cd $1\n");
			sb.append("\n");
			sb.append("java -Xmx1024M -classpath ./:");
			File jarsPathF = new File(jarsPath);
			File[] files = jarsPathF.listFiles();

			for (File jar : files) {

				if (jar.getName().endsWith(".jar")) {
					sb.append("./" + jar.getName());
					sb.append(":");
				}
			}

			sb.deleteCharAt(sb.length() - 1);
			sb.append(" " + fullMainClass + " $2 " + jobOutput);
			sb.append("\n");

			AnalysisLogger.getLogger().trace("D4ScienceGenerator->Generating script in " + expectedscript.getAbsolutePath());
			FileTools.saveString(expectedscript.getAbsolutePath(), sb.toString(), true, "UTF-8");
		}
		AnalysisLogger.getLogger().trace("D4ScienceGenerator->Script " + expectedscript.getAbsolutePath()+" yet exists!");
	}

	@Override
	public List<StatisticalType> getInputParameters() {
		
		List<StatisticalType> distributionModelParams = new ArrayList<StatisticalType>();
		distributionModelParams.add(new ServiceType(ServiceParameters.USERNAME,"ServiceUserName","The final user Name"));
		
		return distributionModelParams;
	}


	@Override
	public String getResources() {
		Resources res = new Resources();
		try {
			int activeNodes = jobManager.getActiveNodes();
			for (int i = 0; i < activeNodes; i++) {
				try {
					res.addResource("Worker_" + (i + 1), 100);
				} catch (Exception e1) {
				}
			}
		} catch (Exception e) {
			AnalysisLogger.getLogger().debug("D4ScienceGenerator->active nodes not ready");
		}
		if ((res != null) && (res.list != null))
			return HttpRequest.toJSon(res.list).replace("resId", "resID");
		else
			return "";
	}

	@Override
	public float getStatus() {
		try {
			if (stop)
				return 100f;
			else
				return Math.max(0.5f, jobManager.getStatus() * 100f);
		} catch (Exception e) {
			return 0f;
		}
	}

	@Override
	public StatisticalType getOutput() {
		return distributionModel.getOutput();
	}

	@Override
	public ALG_PROPS[] getSupportedAlgorithms() {
		ALG_PROPS[] p = { ALG_PROPS.PARALLEL_SPECIES_VS_CSQUARE_FROM_DATABASE};
		return p;
	}

	@Override
	public INFRASTRUCTURE getInfrastructure() {
		return INFRASTRUCTURE.D4SCIENCE;
	}

	@Override
	public void init() throws Exception {

		Properties p = AlgorithmConfiguration.getProperties(config.getConfigPath() + AlgorithmConfiguration.nodeAlgorithmsFile);
		mainclass = p.getProperty(config.getModel());
		distributionModel = (SpatialProbabilityDistributionNode) Class.forName(mainclass).newInstance();
		distributionModel.setup(config);

	}

	@Override
	public void setConfiguration(AlgorithmConfiguration config) {
		this.config = config;
		AnalysisLogger.setLogger(config.getConfigPath() + AlgorithmConfiguration.defaultLoggerFile);
	}

	@Override
	public void shutdown() {
		
		try {
			jobManager.stop();
		} catch (Exception e) {
		}
		try {
			distributionModel.stop();
		} catch (Exception e) {
		}
		stop = true;
	}

	@Override
	public String getLoad() {
		long tk = System.currentTimeMillis();
		int processedSpecies = distributionModel.getNumberOfProcessedSpecies();
		ResourceLoad rs = new ResourceLoad(tk, processedSpecies);
		return rs.toString();
	}

	private long lastTime;
	private int lastProcessed;

	@Override
	public String getResourceLoad() {
		long thisTime = System.currentTimeMillis();
		int processedRecords = distributionModel.getNumberOfProcessedSpecies();
		int estimatedProcessedRecords = 0;
		if (processedRecords == lastProcessed) {
			estimatedProcessedRecords = Math.round(((float) thisTime * (float) lastProcessed) / (float) lastTime);
		} else {
			lastProcessed = processedRecords;
			estimatedProcessedRecords = lastProcessed;
		}
		lastTime = thisTime;
		ResourceLoad rs = new ResourceLoad(thisTime, estimatedProcessedRecords);
		return rs.toString();
	}
	
	
	@Override
	public GenericAlgorithm getAlgorithm() {
		return distributionModel;
	}

	@Override
	public String getDescription() {
		return "A D4Science Cloud Processor for Species Distributions";
	}

}
