package org.gcube.datatransfer.agent.impl.porttype;

import java.io.File;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.FutureTask;

import org.apache.axis.components.uuid.UUIDGen;
import org.apache.axis.components.uuid.UUIDGenFactory;
import org.gcube.common.core.contexts.GCUBEServiceContext;
import org.gcube.common.core.faults.GCUBEFault;
import org.gcube.common.core.porttypes.GCUBEPortType;

import org.gcube.common.core.utils.logging.GCUBELog;
import org.gcube.datatransfer.agent.impl.context.AgentContext;
import org.gcube.datatransfer.agent.impl.context.ServiceContext;

import org.gcube.datatransfer.agent.impl.state.AgentResource;
import org.gcube.datatransfer.agent.impl.state.AgentResource.FutureWorker;
import org.gcube.datatransfer.agent.impl.utils.TransferUtils;
import org.gcube.datatransfer.common.objs.LocalSource;
import org.gcube.datatransfer.common.objs.LocalSources;
import org.gcube.datatransfer.common.outcome.TransferStatus;
import org.gcube.datatransfer.agent.impl.worker.SyncWorker;
import org.gcube.datatransfer.agent.impl.worker.Worker;
import org.gcube.datatransfer.agent.impl.worker.async.DataStorageASyncWorker;
import org.gcube.datatransfer.agent.impl.worker.async.LocalFileTransferASyncWorker;
import org.gcube.datatransfer.agent.impl.worker.async.StorageManagerASyncWorker;
import org.gcube.datatransfer.agent.impl.worker.async.TreeManagerAsyncWorker;
import org.gcube.datatransfer.agent.impl.worker.sync.LocalFileTransferSyncWorker;
import org.gcube.datatransfer.agent.impl.worker.sync.LocalTransferSyncWorker;
import org.gcube.datatransfer.agent.impl.worker.sync.StorageManagerSyncWorker;
import org.gcube.datatransfer.agent.impl.worker.sync.TreeManagerSyncWorker;
import org.gcube.datatransfer.common.utils.Utils;
import org.gcube.datatransfer.agent.stubs.datatransferagent.CancelTransferFault;
import org.gcube.datatransfer.agent.stubs.datatransferagent.CancelTransferMessage;
import org.gcube.datatransfer.agent.stubs.datatransferagent.DestData;
import org.gcube.datatransfer.agent.stubs.datatransferagent.GetTransferOutcomesFault;
import org.gcube.datatransfer.agent.stubs.datatransferagent.MonitorTransferFault;
import org.gcube.datatransfer.agent.stubs.datatransferagent.MonitorTransferReportMessage;
import org.gcube.datatransfer.agent.stubs.datatransferagent.SourceData;
import org.gcube.datatransfer.agent.stubs.datatransferagent.StartTransferMessage;
import org.gcube.datatransfer.agent.stubs.datatransferagent.StorageType;
import org.gcube.datatransfer.agent.stubs.datatransferagent.TransferFault;
import org.gcube.datatransfer.agent.stubs.datatransferagent.TransferType;

/**
 * 
 * @author Andrea Manzi(CERN)
 *
 */
public class DataTransferAgent extends GCUBEPortType {

	protected final GCUBELog logger = new GCUBELog(DataTransferAgent.class);

	/** The UUIDGen */
	private static final UUIDGen uuidgen = UUIDGenFactory.getUUIDGen();
	public DataTransferAgent(){
		
	}

	public AgentResource getResource() throws Exception {
		return (AgentResource) AgentContext.getContext().getAgent();
	}

	/**
	 * Starts a new transfer
	 * @param message
	 * @return
	 * @throws TransferFault
	 */
	public String startTransfer(StartTransferMessage message) throws TransferFault {

		logger.info("Start Transfer invoked in scope " + message.getSource().getScope());

		String id =  uuidgen.nextUUID();
		try {

			//woker which consumes an RS of files
			if (message.getSource().getType().getValue().compareTo(TransferType.LocalFileBasedTransfer.getValue())==0)
			{
				logger.debug("Local transfer from  URI :"+ message.getSource().getInputURIs()[0]);
				LocalTransferSyncWorker worker = new LocalTransferSyncWorker(id,message.getSource(), message.getDest());
				return (String)worker.call();	
			}
			//check if the operation is sync or async
			else if(message.isSyncOp()) {
				logger.debug("Sync operation");
				logger.debug("TransferType: "+ message.getSource().getType().getValue());
				return startSyncTask(id, message.getSource(), message.getDest());
			}
			else {
				logger.debug("ASync operation");
				logger.debug("TransferType: "+ message.getSource().getType().getValue());
				FutureTask<Worker> task = startAsyncTask(id, message.getSource(), message.getDest());
				//CHANGED - Now we set the workerMap inside the startAsyncTask operation..
				//getResource().getWorkerMap().put(id, task);		
			}
		} catch (Exception e) {
			logger.error("Unable to perform the transfer", e);
			throw Utils.newFault(new TransferFault(), e);
		}
		
		logger.debug("Returning id : "+ id);
		return id;	

	}

	/**
	 * Cancel a  transfer
	 * @param message
	 * @return
	 * @throws TransferFault
	 * @throws GCUBEFault 
	 */
	public String cancelTransfer(CancelTransferMessage message) throws CancelTransferFault {
		String handlerID = message.getTransferID();
		
		
		if (handlerID == null)
			throw Utils.newFault(new CancelTransferFault(), new Exception("The Transfer ID is null"));
		
		FutureTask futureTask = null;
		Worker worker=null;
		FutureWorker futureWorker = null;
		try {
			futureWorker = getResource().getWorkerMap().get(handlerID);
			if (futureWorker != null) {
				futureTask = futureWorker.getFutureTask();
				worker  = futureWorker.getWorker();
				if(futureTask==null)logger.debug("null futureTask");
				else if (worker==null)logger.debug("null worker");
				else{
					if(worker.getThreadList()==null)logger.debug("null threadList");
					else worker.getThreadList().stop();
					
					futureTask.cancel(message.isForceStop());
				}
			}
			else logger.debug("null futureWorker");
		} catch (Exception e) {
			logger.error("Unable to cancel the transfer", e);
			throw Utils.newFault(new CancelTransferFault(), e);
		}	
		finally{
			try {
				getResource().getWorkerMap().remove(handlerID);
				//Set status CANCEL on the DB
				ServiceContext.getContext().getDbManager().updateTransferObjectStatus(handlerID, TransferStatus.CANCEL.name());
			} catch (Exception e) {
				throw Utils.newFault(new CancelTransferFault(), e);
			}
		}
		return handlerID;
	}
	
	
	public  FutureTask<Worker>  startAsyncTask(String id,SourceData source ,DestData dest) throws Exception {
		logger.debug("startAsyncTask has been reached ... ");
		Worker worker = null;
		FutureTask<Worker> task = null;
		if (source.getType().getValue().compareTo(TransferType.TreeBasedTransfer.getValue()) ==0)
			 worker= new  TreeManagerAsyncWorker(id, source, dest);
		else if (source.getType().getValue().compareTo(TransferType.FileBasedTransfer.getValue()) ==0) {
			if (dest.getOutUri().getOptions().getStorageType().getValue().compareTo(StorageType.StorageManager.getValue()) == 0)
				 worker= new  StorageManagerASyncWorker(id,source, dest);
			else if (dest.getOutUri().getOptions().getStorageType().getValue().compareTo(StorageType.DataStorage.getValue()) == 0)
			 	 worker= new  DataStorageASyncWorker(id,source, dest);
			else worker= new  LocalFileTransferASyncWorker(id,source, dest);
		}
		task = new FutureTask<Worker> (worker);
		worker.setTask(task);
		Thread t = new Thread(task);
		t.start();
		
		//store status Queued;
		ServiceContext.getContext().getDbManager().storeTransfer(TransferUtils.createTransferJDO(id));
		
		FutureWorker futureWorker = new AgentResource.FutureWorker();
		futureWorker.setFutureTask(task);
		futureWorker.setWorker(worker);
		
		getResource().getWorkerMap().put(id, futureWorker);	
		return task;
		 
	}
	
	private  String  startSyncTask(String id,SourceData source ,DestData dest) throws Exception {

		SyncWorker worker =  null;
		FutureTask<Worker> task = null;
		
		if (source.getType().getValue().compareTo(TransferType.TreeBasedTransfer.getValue()) ==0){
			 worker= new  TreeManagerSyncWorker(id,source, dest);
			 worker.call();
			 return worker.getOutcomeLocator();
		}
		else if (source.getType().getValue().compareTo(TransferType.FileBasedTransfer.getValue()) ==0) {
			if (dest.getOutUri().getOptions().getStorageType().getValue().compareTo(StorageType.StorageManager.getValue()) == 0)
				 worker= new  StorageManagerSyncWorker(id,source, dest);
			else 
				worker= new  LocalFileTransferSyncWorker(id,source, dest);
		}
		task = new FutureTask<Worker> (worker);
		Thread t = new Thread(task);
		t.start();
		
		return worker.getOutcomeLocator(); 
	}


	public String monitorTransfer(String transferId) throws MonitorTransferFault {
		
		if (transferId == null)
			throw Utils.newFault(new MonitorTransferFault(), new Exception("The Transfer ID is null"));
		String status ="";
		try {
			status = ServiceContext.getContext().getDbManager().getTransferStatus(transferId);
		}catch (Exception e){
			throw Utils.newFault(new MonitorTransferFault(), e);
		}
		return status;
	}
	
	public String getTransferOutcomes(String transferId) throws GetTransferOutcomesFault {
		
		if (transferId == null)
			throw Utils.newFault(new GetTransferOutcomesFault(), new Exception("The Transfer ID is null"));
		
		String rs = "";
		try {
			rs = ServiceContext.getContext().getDbManager().getTransferObjectOutComeAsRS(transferId);
		}catch (Exception e){
			e.printStackTrace();
			throw  Utils.newFault(new GetTransferOutcomesFault(),e);
		}
		return rs;
	}
	
	public String getLocalSources(String filePath) {
		String vfsRoot=((String) ServiceContext.getContext().getProperty("vfsRoot", true));
		
		if(!filePath.endsWith("/"))filePath=filePath+"/";
		
		String path=null;
		if((!vfsRoot.endsWith("/"))&&(!filePath.startsWith("/"))){
			path = vfsRoot+"/"+filePath;
		}
		else path = vfsRoot+filePath;
		
		LocalSources sources = new LocalSources();
		List<LocalSource> list = new ArrayList<LocalSource>();

		File main = new File(path);
		if(!main.isDirectory())return null;
		String[] children = main.list();
		if(children!=null){
			if(children.length>0){
				for(String tmp:children){
					File child = new File(path+tmp);
					if(child.isDirectory()){
						LocalSource dir = new LocalSource();
						dir.setDirectory(true);
						dir.setPath(child.getAbsolutePath());
						dir.setVfsRoot(vfsRoot);
						
						list.add(dir);
					}
					else{
						LocalSource file = new LocalSource();
						file.setDirectory(false);
						file.setPath(child.getAbsolutePath());
						
						file.setVfsRoot(vfsRoot);
						
						file.setSize(child.length());
						list.add(file);
					}
				}
			}
		}
		sources.setList(list);
		return sources.toXML();
	}

	public MonitorTransferReportMessage monitorTransferWithProgress(String transferId) throws MonitorTransferFault {
		
		if (transferId == null)
			throw Utils.newFault(new MonitorTransferFault(), new Exception("The Transfer ID is null"));
		
		MonitorTransferReportMessage message = null;
		try {
			message = ServiceContext.getContext().getDbManager().getTrasferProgress(transferId);
		}catch (Exception e){
			e.printStackTrace();
			throw  Utils.newFault(new MonitorTransferFault(),e);
		}
		return message;
	}

	@Override
	protected GCUBEServiceContext getServiceContext() {
		return ServiceContext.getContext();
	}

}