package org.gcube.datatransfer.agent.library;

import gr.uoa.di.madgik.commons.server.PortRange;
import gr.uoa.di.madgik.commons.server.TCPConnectionManager;
import gr.uoa.di.madgik.commons.server.TCPConnectionManagerConfig;
import gr.uoa.di.madgik.grs.proxy.tcp.TCPConnectionHandler;
import gr.uoa.di.madgik.grs.proxy.tcp.TCPStoreConnectionHandler;
import gr.uoa.di.madgik.grs.proxy.tcp.TCPWriterProxy;

import java.io.File;
import java.net.InetAddress;
import java.net.URL;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.List;

import org.gcube.common.core.scope.GCUBEScope;
import org.gcube.common.core.security.GCUBESecurityManager;
import org.gcube.common.core.security.GCUBESecurityManagerImpl;
import org.gcube.common.core.utils.logging.GCUBEClientLog;
import org.gcube.data.trees.patterns.TreePattern;
import org.gcube.datatransfer.agent.library.calls.AgentServiceCall;
import org.gcube.datatransfer.agent.library.grs.GRSFileWriter;
import org.gcube.datatransfer.agent.library.outcome.FileTransferOutcome;
import org.gcube.datatransfer.agent.library.utils.Utils;
import org.gcube.datatransfer.agent.stubs.datatransferagent.CancelTransferMessage;
import org.gcube.datatransfer.agent.stubs.datatransferagent.DestData;
import org.gcube.datatransfer.agent.stubs.datatransferagent.InputPattern;
import org.gcube.datatransfer.agent.stubs.datatransferagent.OutUriData;
import org.gcube.datatransfer.agent.stubs.datatransferagent.SourceData;
import org.gcube.datatransfer.agent.stubs.datatransferagent.StartTransferMessage;
import org.gcube.datatransfer.agent.stubs.datatransferagent.TransferFault;
import org.gcube.datatransfer.agent.stubs.datatransferagent.TransferType;


/**
 * 
 * 
 * @author Andrea Manzi(CERN)
 *
 */
public class AgentLibrary {
	
	static {
		 List<PortRange> ports=new ArrayList<PortRange>(); 
		    ports.add(new PortRange(4000, 4050));           
		    try {
				TCPConnectionManager.Init(
				  new TCPConnectionManagerConfig(InetAddress.getLocalHost().getHostName(),
				    ports,                               
				    true                                   
				));
			} catch (UnknownHostException e) {
				e.printStackTrace();
			}
		    TCPConnectionManager.RegisterEntry(new TCPConnectionHandler());      //Register the handler for the gRS2 incoming request
		    TCPConnectionManager.RegisterEntry(new TCPStoreConnectionHandler()); //Register the handler for the gRS2 store incoming requests
	}
	
	private GCUBESecurityManagerImpl secMan= new GCUBESecurityManagerImpl(){	public boolean isSecurityEnabled() {return false;}};	

	private AgentServiceCall call= null;
	
	public GCUBEClientLog logger= new GCUBEClientLog(this.getClass());

	/**
	 * Creates an Agent Library to access an Agent Service on the given scope
	 * @param scope the scope 
	 * @throws Exception exception
	 */
	public AgentLibrary (GCUBEScope scope, GCUBESecurityManagerImpl ... secManager) throws Exception {
		if (secManager.length !=0)
			secMan = secManager[0];
		this.call = new AgentServiceCall(scope, new GCUBESecurityManager[]{secMan});
	}

	/**
	 * Creates an Agent Library to access the given instance of Agent Service
	 * 
	 * @param scope Agent service scope
	 * @param endpoint endpoint
	 * @param port port
	 * @throws Exception  Exception
	 */
	public AgentLibrary (GCUBEScope scope,String endpoint, String port,GCUBESecurityManagerImpl ... secManager) throws Exception {
		if (secManager.length !=0)
			secMan = secManager[0];
		this.call = new AgentServiceCall(scope, new GCUBESecurityManager[]{secMan});
		this.call.setEndpoint(endpoint, port);
	}

	/**
	 * Start a transfer of Data represented by the given pattern and SourceIDs to the data Storage represented by the given outputStorageId
	 * 
	 * @param patternInput
	 * @param inputSourceID
	 * @param outputSourceId
	 * @param scope
	 * @return the ID of the transfer;
	 * @throws Exception
	 */
	public String startTransfer(TreePattern patternInput, String inputSourceID, String outputStorageId,GCUBEScope scope) throws Exception{
		
		StartTransferMessage message = new StartTransferMessage();
		DestData dest = new DestData();
		dest.setOutSourceId(outputStorageId);
		dest.setScope(scope.toString());
		message.setDest(dest);
		SourceData source = new SourceData();
		InputPattern input =  new InputPattern();
		input.setPattern(Utils.toHolder(patternInput));
		input.setSourceId(inputSourceID);
		source.setInputSource(input);
		source.setScope(scope.toString());
		source.setType(TransferType.TreeBasedTransfer);
		message.setSource(source);
		
		String transferID = null;
		
		try {
			transferID = call.startTransfer(message);
		
		} catch (TransferFault e) {
			e.printStackTrace();
			throw e;
		}
		return transferID;
		
	}
	
	/**
	 * 
	 * @param inputURL
	 * @param outputPath
	 * @param scope
	 * @param overwrite
	 * @return
	 * @throws Exception
	 */
	public String startTransfer(URL inputURL, String outputPath,GCUBEScope scope,boolean overwrite) throws Exception{
		
		StartTransferMessage message = new StartTransferMessage();
		DestData dest = new DestData();
		OutUriData outURI = new OutUriData();
		outURI.setOutUris(new String[] {outputPath});
		outURI.setOverwrite(overwrite);
		dest.setOutUri(outURI);
		dest.setScope(scope.toString());
		message.setDest(dest);
		SourceData source = new SourceData();
		source.setInputURIs(new String[]{inputURL.toString()});

		source.setScope(scope.toString());
		source.setType(TransferType.FileBasedTransfer);
		message.setSource(source);
		
		String transferID = null;
		
		try {
			transferID = call.startTransfer(message);
		
		} catch (TransferFault e) {
			e.printStackTrace();
			throw e;
		}
		return transferID;
		
	}
	
	/**
	 * Cancel a scheduled/running transfer 
	 * 
	 * @param transferId
	 * @param forceStop cancel the transfer even if it's running 
	 * @throws Exception 
	 */
	public void cancelTransfer(String transferId, boolean forceCancel) throws Exception{
		
		CancelTransferMessage cancelMessage = new CancelTransferMessage();
		cancelMessage.setForceStop(forceCancel);
		cancelMessage.setTransferID(transferId);
		call.cancelTransfer(cancelMessage);
	}
	
	/**
	 * 
	 * @param inputFiles
	 * @param destinationFolder
	 * @param overwrite
	 * @return an Array of <code>FileTransferOutcome</code> 
	 * @throws Exception
	 */
	public  ArrayList<FileTransferOutcome>  copyLocalFiles(File []  inputFiles,File destinationFolder,boolean overwrite) throws Exception
	{
		GRSFileWriter writer=new GRSFileWriter(new TCPWriterProxy(),inputFiles);	
		
		StartTransferMessage message = new StartTransferMessage();
		SourceData sourceData = new SourceData();	
		sourceData.setScope(this.call.getScopeManager().getScope().toString());
		sourceData.setInputURIs(new String[]{writer.getLocator().toString()});
		sourceData.setType(TransferType.LocalFileBasedTransfer);
		
		DestData destData = new DestData();
		OutUriData uri = new OutUriData();
		uri.setOutUris(new String[] {destinationFolder.getAbsolutePath()});
		uri.setOverwrite(overwrite);
		destData.setOutUri(uri);
		message.setSource(sourceData);
		message.setDest(destData);
		writer.start();	
		writer.join();
		String outcomeRSURI = this.call.startLocalFileTransfer(message);	
		return Utils.getOutcomes(outcomeRSURI);
	}
}
